//-------------------------------------------------------------------------------
// Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
// Copyright (c) 2011- kotemaru@kotemaru.org
//-------------------------------------------------------------------------------
/**
 * DataBase クラス。
 * JavaScript から JDBC および SQL を容易に使う為のクラス。
 */

/**
 * コンストラクタ。
 * <li>指定されたデータソースに接続する。
 * <li>データソースを省略した場合にはDerbyに接続する。
 * <li>初期状態でトランザクションモード。
 * @param dataSourceName データソース名。
 */
function DataBase(dataSourceName) {
	this.IS_DEBUG = __ENV__.LOG.isDebugEnabled();
	if (dataSourceName == null) {
		this.connect = __ENV__.getDBConnection();
	} else {
		this.connect = __ENV__.getDBConnection(dataSourceName);
	}
	this.connect.setAutoCommit(false);
}

/**
 * トランザクション処理の補助関数。
 * <li>処理関数をトランザクションの開始と終了の内側で実行する。
 * <li>処理関数が例外を発生させた場合は rollback する。
 * @param dataSourceName データソース名。
 * @param func 処理関数
 * @return func の戻り値
 */
DataBase.transaction = function(dataSourceName, func){
	var db = new DataBase(dataSourceName);
	try {
		db.connect.setAutoCommit(false);
		return func(db);
	} catch (e) {
		db.rollback();
		if (e.rhinoException) {
			throw e.rhinoException;
		} else if (e.fileName && e.lineNumber){
			throw e.message+"("+e.fileName+"#"+e.lineNumber+")";
		} else {
			throw e;
		}
	} finally {
		try {
			db.commit();
		} finally {
			db.close();
		}
	}
}


//--------------------------------------------------------------------
/**
 * トランザクションのコミット。
 */
DataBase.prototype.commit = function(){
	this.connect.commit();
}
/**
 * トランザクションのROLLBACK。
 */
DataBase.prototype.rollback = function(){
	this.connect.rollback();
}
/**
 * データベース接続を閉じる。
 */
DataBase.prototype.close = function(){
	this.connect.close();
}

/**
 * テーブルを作成する。
 * <li>カラム情報の要素名がカラム名、値がカラムの型情報になる。
 * <li>カラムの順序は指定できない。
 * @param name テーブル名
 * @param cols カラム情報
 */
DataBase.prototype.createTable = function(name, cols){
	var sql = "CREATE TABLE "+name+"(";
	for (var key in cols) sql += key +" "+cols[key]+",";
	sql = sql.replace(/,$/,")");
	this._execute(sql);
}

/**
 * テーブルを破棄する。
 * @param name テーブル名
 * @param isIgnoreError trueなら例外を無視する。
 */
DataBase.prototype.dropTable = function(name, isIgnoreError){
	try {
		this._execute("DROP TABLE "+name);
	} catch(e) {
		if (!isIgnoreError) throw e;
		__ENV__.LOG.warn(e);
	}
}

/**
 * SQL文の実行。
 * @param sql SQL文
 * @param args 引数配列
 */
DataBase.prototype.execute = function(sql, args){
	return this._execute(sql, args);
}

/**
 * クエリSQL文の実行。
 * <li>戻り値の ResultSet は編集、スクロール可能。
 * @param sql SQL文
 * @param args 引数配列
 * @return DataBase.Select のオブジェクト
 */
DataBase.prototype.query = function(sql, args){
	var st = this._getStatement(sql, args, null, "rw");
	var rset = st.executeQuery();
	return new DataBase.Select(st, rset, "rw");
}

/**
 * クエリSQL文の実行。
 * <li>戻り値の ResultSet は編集、スクロール不可能。
 * @param sql SQL文
 * @param args 引数配列
 * @return DataBase.Select のオブジェクト
 */
DataBase.prototype.queryFirst = function(sql, args){
	var st = this._getStatement(sql, args, null, "ro");
	var rset = st.executeQuery();
	return new DataBase.Select(st, rset, "ro");
}

/**
 * SELECT-SQL文の実行。
 * <li>戻り値の ResultSet は編集、スクロール不可能。
 * <li>where で指定した 要素名=要素値 をWHERE句の条件とする。大小比較はできない。
 * <li>order で指定した要素値を ORDER BY句に設定する。要素値は"ASC"|"DESC"。
 * <li>表現しきれない場合は query() を使う。
 * @param name テーブル名
 * @param where WHERE句の指定
 * @param order ORDER BY句の指定
 * @return DataBase.Select のオブジェクト
 */
DataBase.prototype.select = function(name, where, order){
	var sql = "SELECT * FROM "+name 
		+ this._sqlWhere(where)
		+ this._sqlOrder(order);
	var st = this._getStatement(sql, where, null, "ro");
	var rset = st.executeQuery();
	return new DataBase.Select(st, rset, "ro");
}

/**
 * INSERT-SQL文の実行。
 * <li>args で指定した 要素名=要素値 の行をテーブルに追加する。
 * <li>未定義のカラムはデフォルトとなる。
 * <li>存在しないカラム名を指定した場合は例外となる。
 * <li>表現しきれない場合は execute() を使う。
 * @param name テーブル名
 * @param args 引数配列
 */
DataBase.prototype.insert = function(name, args){
	var names = this._sqlNames(args, " (")+")";
	var values = this._sqlValues(args, " VALUES(")+")";
	var sql = "INSERT INTO "+ name + names + values;
	return this._execute(sql, args);
}

/**
 * UPDATE-SQL文の実行。
 * <li>args で指定した 要素名=要素値 の行を更新する。
 * <li>where で指定した 要素名=要素値 をWHERE句の条件とする。大小比較はできない。
 * <li>表現しきれない場合は execute() を使う。
 * @param name テーブル名
 * @param args 引数配列
 * @param where WHERE句の指定
 */
DataBase.prototype.update = function(name, args, where){
	var values = this._sqlSets(args, " SET ");
	var sql = "UPDATE "+ name + values + this._sqlWhere(where);
	return this._execute(sql, args, where);
}

/**
 * DELETE-SQL文の実行。(deleteがJavaScriptの予約語なので)
 * <li>where で指定した 要素名=要素値 をWHERE句の条件とする。大小比較はできない。
 * <li>表現しきれない場合は execute() を使う。
 * @param name テーブル名
 * @param where WHERE句の指定
 */
DataBase.prototype.remove = function(name, where){
	var sql = "DELETE FROM "+ name + this._sqlWhere(where);
	return this._execute(sql, where);
}

//-----------------------------------------------------------
/**
 * DataBase.Select 内部クラス。
 * <li>クエリの戻り値なる ResultSet のラッパー。
 * <li>イテレータを持つ。
 */

/**
 * コンストラクタ。
 * <li>内部利用のみ。
 */
DataBase.Select = function(st, rset, mode) {
	this.statement = st;
	this.rset = rset;
	this.mode = mode;
	this.rowNumber = 0;
	this.hasNext = false;
}
DataBase.Select.prototype = {
	/**
	 * クエリを閉じる。
	 */
	close : function() {
		this.statement.close();
	},

	/**
	 * カラム名のリストを得る。
	 * <li>カラム名は全て英小文字に変換する。
	 */
	getColumnNames: function() {
		if (this.columnNames) return this.columnNames;
		var meta = this.rset.getMetaData();
		this.columnNames = [];
		for (var i=0; i<meta.getColumnCount(); i++) {
			this.columnNames.push(meta.getColumnName(i+1).toLowerCase());
		}
		return this.columnNames;
	},

	/**
	 * カーソルを次の行に移動する。
	 */
	next: function() {
		var res = this.rset.next();
		if (res)	this.rowNumber++;
		this.hasNext = res;
		return res;
	},
	absolute: function(row) {
		var res = this.rset.absolute(row);
		if (res)	this.rowNumber = row;
		return res;
	},

	/**
	 * 次の行を得る。
	 * @return 行のオブジェクト
	 */
	getNextRow: function() {
		if (this.next()) return this.getCurrentRow();
		return null;
	},

	/**
	 * 複数の行を配列で得る。
	 * @param start 開始行(省略値=1)
	 * @param end   終了行(省略値=最終行)
	 * @return 行の配列
	 */
	getRows: function(start, end) {
		if (start == undefined) start = 1;
		if (end == undefined) end = 2147483648;

		var rows = [];
		if (this.mode == "rw") {
			this.absolute(start);
		} else {
			if (this.rowNumber > start) {
				throw new Error("Current over start row."
						+" cur="+this.rowNumber+", start="+start);
			}
			while (this.rowNumber < start && this.next());
		}
		while (this.hasNext && this.rowNumber < end) {
			rows.push(this.getCurrentRow());
			this.next();
		}
		return rows;
	},


	/**
	 * 現在行をオブジェクトで得る。
	 * <li>要素名がカラム名となる。
	 * @return 行のオブジェクト
	 */
	getCurrentRow: function() {
		var names = this.getColumnNames();
		var data = {};
		for (var i=0; i<names.length; i++) {
			data[names[i]] = this._toJson(this.rset.getObject(i+1));
		}
		return data;
	},


	/**
	 * 現在行に値を設定する。
	 * <li>要素名がカラム名となる。
	 * <li>編集可能な ResultSet が必要。
	 * @param data 行のオブジェクト
	 */
	setCurrentRow: function(data) {
		for (var name in data) {
			this.rset.updateObject(name, data[name]);
		}
	},


	/**
	 * 現在行に行を挿入する。
	 * <li>要素名がカラム名となる。
	 * <li>編集可能な ResultSet が必要。
	 * @param data 行のオブジェクト
	 */
	insCurrentRow: function(data) {
		this.rset.moveToInsertRow();
		this.setCurrentRow(data)
		this.rset.insertRow();
		this.rset.moveToCurrentRow();
	},
	/**
	 * 現在行に削除する。
	 * <li>編集可能な ResultSet が必要。
	 */
	delCurrentRow: function() {
		this.rset.deleteRow();
	},

	//-------------------------------------------------------------------
	// Need Scrollable.

	/**
	 * 行数を得る。
	 * <li>スクロール可能な ResultSet が必要。
	 * @return 行数
	 */
	length: function() {
		return this.rset.getFetchSize();
	},

	/**
	 * 指定行をオブジェクトで得る。
	 * <li>要素名がカラム名となる。
	 * <li>スクロール可能な ResultSet が必要。
	 * @param row 行番号(1始まり)
	 * @return 行のオブジェクト
	 */
	getRow: function(row) {
		this.absolute(row);
		return this.getCurrentRow();
	},

	/**
	 * 指定行に値を設定する。
	 * <li>要素名がカラム名となる。
	 * <li>スクロール可能な ResultSet が必要。
	 * @param row 行番号(1始まり)
	 * @param data 行のオブジェクト
	 */
	setRow: function(row, data) {
		this.absolute(row);
		return this.setCurrentRow(data);
	},

	/**
	 * 指定行に行を挿入する。
	 * <li>要素名がカラム名となる。
	 * <li>編集可能な ResultSet が必要。
	 * @param row 行番号(1始まり)
	 * @param data 行のオブジェクト
	 */
	insRow: function(row, data) {
		this.absolute(row);
		this.rset.insCurrentRow(data);
	},
	/**
	 * 指定行を削除する。
	 * <li>編集可能な ResultSet が必要。
	 * @param row 行番号(1始まり)
	 */
	delRow: function(row) {
		this.absolute(row);
		this.rset.deleteRow();
	},

	get: function(row, column) {
		this.absolute(row);
		return this.rset.getObject(column);
	},

	set: function(row, column, data) {
		this.absolute(row);
		this.rset.updateObject(column, data);
	},

	_toJson: function(val) {
		if (val == null) return null;
		if (val instanceof java.lang.Number) return val.intValue();
		return ""+val.toString();
	},

	__iterator__: function() {
		return new DataBase.SelectIterator(this);
	}
}
DataBase.SelectIterator = function(select) {
	this.select = select;
}
DataBase.SelectIterator.prototype = {
	next: function() {
		with (this.select) {
			if (rset.next()) {
				return getCurrentRow();
			} else {
				throw StopIteration;
			}
		}
	}
}


//-----------------------------------------------------------
// Private methods.
DataBase.prototype._execute = function(sql, args1, args2){
	var st = this._getStatement(sql, args1, args2, "ro");
	try {
		return st.executeUpdate();
	} finally {
		st.close();
	}
}
DataBase.prototype._getStatement = function(sql, args1, args2, mode){
	this.log(sql, args1, args2);
	var st;
	if (mode == "rw") {
		st = this.connect.prepareStatement(sql,
			java.sql.ResultSet.TYPE_SCROLL_INSENSITIVE,
			java.sql.ResultSet.CONCUR_UPDATABLE);
	} else { // readonly
		st = this.connect.prepareStatement(sql,
			java.sql.ResultSet.TYPE_FORWARD_ONLY,
			java.sql.ResultSet.CONCUR_READ_ONLY);
	}
	try {
		var n = 1;
		n = this._setStatementParam(st, n, args1);
		n = this._setStatementParam(st, n, args2);
		return st;
	} catch (e) {
		st.close();
		throw e;
	}
}
DataBase.prototype._setStatementParam = function(st, n, args){
	if (args == null) return n;

	if (args instanceof Array) {
		for (var i=0; i<args.length; i++) st.setObject(n++, args[i]);
	} else {
		for (var k in args) {
			if (k != "") {
				//__ENV__.LOG.debug("-->"+n+":"+k+":"+args[k]);
				st.setObject(n++, args[k]);
			}
		}
	}
	return n;
}
//-----------------------------------------------------------
DataBase.prototype._sqlWhere = function(props){
	if (props == null) return "";
	var where = "";
	for (var k in props) where += " AND "+ k +"=?";
	where = where.replace(/^[ ]AND/," WHERE");
	return where;
}
DataBase.prototype._sqlOrder = function(props){
	if (props == null) return "";
	var order = "";
	for (var k in props) order += ", "+k + " "+props[k];
	order = order.replace(/^,/," ORDER BY");
	return order;
}
DataBase.prototype._sqlSets = function(props, token){
	if (props == null) return "";
	var values = "";
	for (var k in props) {
		if (k != "") values += ","+ k +"=?";
	}
	values = values.replace(/^,/, token);
	return values;
}
DataBase.prototype._sqlNames = function(props, token){
	if (props == null) return "";
	var values = "";
	for (var k in props) {
		if (k != "") values += ","+ k ;
	}
	values = values.replace(/^,/, token);
	return values;
}
DataBase.prototype._sqlValues = function(props, token){
	if (props == null) return "";
	var values = "";
	for (var k in props) {
		if (k != "") values += ",?";
	}
	values = values.replace(/^,/, token);
	return values;
}

DataBase.prototype.log = function(sql, args1, args2 ){
	if (this.IS_DEBUG) {
		try {
			__ENV__.LOG.debug("DataBase: "+sql+" : "+uneval(args1)+","+uneval(args2));
		} catch (e) {
			__ENV__.LOG.debug("DataBase: "+sql+" : "+e);
		}
	}
}

DataBase.prototype.logStatement = function(st) {
	if (this.IS_DEBUG) {
		__ENV__.LOG.debug("DataBase: "+st);
	}
}


// EOF
