/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs.ssjs;
import java.io.*;
import java.net.*;
import java.text.*;
import java.util.*;
import org.mozilla.javascript.*;

/**
Object ::= "{" [Members] "}";
Members ::= Pair ["," Pair]...;
Pair ::= String ":" Value;
Array ::= "[" [Elements] "]";
Elements :: = Value ["," Value]...;
Value ::= 
	  String
	| Number
	| Object
	| Array
	| "true"
	| "false"
	| "null"
;

String ::= /^"([^"]|\\[\\"bfnrt]|\\u[0-9a-fA-F]{4})*"$/;
Number ::= /^-?[0-9]+([.][0-9]+)?([eE][-+]?[0-9]+)?$/;
*/

public class JSONParser {
	private static final int TT_NUMBER = StreamTokenizer.TT_NUMBER;
	private static final int TT_STRING = (int) '"';
	private static final int TT_WORD   = StreamTokenizer.TT_WORD;
	private static final int TT_TRUE   = 1000003;
	private static final int TT_FALSE  = 1000004;
	private static final int TT_NULL   = 1000005;

	private Context cx;
	private Scriptable topScope;
	private String stringValue;
	private double numberValue;
	private Reader reader;

	public JSONParser(Context cx, Scriptable scope) {
		this.cx = cx;
		this.topScope = scope;
	}

	public Scriptable parse(String data) {
		return parse(new StringReader(data));
	}

	public Scriptable parse(Reader reader) {
		this.reader = reader;
		return pObject(nextToken(),topScope);
	}

	private Scriptable pObject(int tkn, Scriptable scope) {
		boolean hasParentheses = ('(' == tkn);
		if (hasParentheses) tkn = nextToken();
		if ('{' != tkn) error("{", tkn);
		Scriptable obj = cx.newObject(scope);
		tkn = nextToken();
		if ('}' != tkn) {
			pPair(tkn, obj);
			tkn = nextToken();
		}
		while (',' == tkn) {
			pPair(nextToken(), obj);
			tkn = nextToken();
		}
		if ('}' != tkn) error("}", tkn);
		if (hasParentheses && nextToken() != ')') error("Not found ')'");
		return obj;
	}

	private void pPair(int tkn, Scriptable scope) {
		if (TT_STRING != tkn && TT_WORD != tkn) error("Not property name", tkn);
		String name = stringValue;
		if (':' != nextToken()) error(":", tkn);
		Object val = pValue(scope);
		scope.put(name, scope, val);
	}

	private Object pValue(Scriptable scope)  {
		return pValue(nextToken(), scope);
	}
	private Object pValue(int tkn, Scriptable scope)  {
		if (tkn == TT_STRING) return cx.javaToJS(stringValue, scope);
		if (tkn == TT_NUMBER) return cx.javaToJS(new Double(numberValue), scope);
		if (tkn == TT_NULL)   return cx.javaToJS(null, scope);
		if (tkn == TT_TRUE)   return cx.javaToJS(Boolean.TRUE, scope);
		if (tkn == TT_FALSE)  return cx.javaToJS(Boolean.FALSE, scope);
		if (tkn == '{') return pObject(tkn, scope);
		if (tkn == '[') return pArray(tkn, scope);
		error("Value", tkn);
		return null;
	}


	private Scriptable pArray(int tkn, Scriptable scope)  {
		if ('[' != tkn) error("[", tkn);
		Scriptable array = cx.newArray(scope, 0);
		tkn = nextToken();
		if (']' == tkn) return array;
		int idx = 0;
		array.put(idx++, array, pValue(tkn, scope));
		tkn = nextToken();
		while (',' == tkn) {
			array.put(idx++, array, pValue(scope));
			tkn = nextToken();
		}
		if (']' != tkn) error("]", tkn);
		return array;
	}

	private StringBuffer tokenBuff = new StringBuffer();
	private StringBuffer hexBuff = new StringBuffer(4);
	private final int nextToken() {
		try {
			stringValue = null;

			int ch = read();
			while (isSpace(ch)) ch = read();

			if (ch == '{' || ch == '}' 
				|| ch == '[' || ch == ']'
				|| ch == ':' || ch == ','
			) {
				return ch;
			}

			tokenBuff.setLength(0);
			if (isLatin(ch)) {
				while (isLatin(ch) || isDigit(ch)) {
					tokenBuff.append((char)ch);
					ch = read();
				}
				unread(ch);
				stringValue = tokenBuff.toString();

				if ("null".equals(stringValue)) return TT_NULL;
				if ("true".equals(stringValue)) return TT_TRUE;
				if ("false".equals(stringValue)) return TT_FALSE;
				//error("Bad keyword "+stringValue);
				return TT_WORD;
			}

			if (isDigit(ch)) {
				while (isDigit(ch) || ch == '.' || ch == '+' 
					|| ch == '-' || ch == 'e' || ch == 'E') {
					tokenBuff.append((char)ch);
					ch = read();
				}
				stringValue = tokenBuff.toString();
				numberValue = Double.parseDouble(stringValue);
				unread(ch);
				return TT_NUMBER;
			}

			if (ch == '"') {
				ch = read();
				while (ch != '"') {
					if (ch == '\\') {
						ch = read();
						if (ch == '\\') ch = '\\';
						else if (ch == 'n') ch = '\n';
						else if (ch == 'r') ch = '\r';
						else if (ch == 't') ch = '\t';
						else if (ch == 'f') ch = '\f';
						else if (ch == 'b') ch = '\b';
						else if (ch == 'u') {
							hexBuff.setLength(0);
							hexBuff.append((char)read());
							hexBuff.append((char)read());
							hexBuff.append((char)read());
							hexBuff.append((char)read());
							ch = (char)Integer.parseInt(hexBuff.toString(),16);
						}
					}
					tokenBuff.append((char)ch);
					if (tokenBuff.length() > 1024000) {
						error("String too long. "+tokenBuff);
					}
					ch = read();
				}
				stringValue = tokenBuff.toString();
				return TT_STRING;
			}

			error("Unknown token", ch);
			return -1;
		} catch (IOException e) {
			error(e.toString());
			return -1;
		}
	}


	private int unreadChar = -1;
	private int read() throws IOException {
		if (unreadChar >= 0) {
			int ch = unreadChar;
			unreadChar = -1;
			return ch;
		}
		return reader.read();
	}
	private void unread(int ch) {
		if (unreadChar >= 0) {
			error("Duplicate unread");
		}
		unreadChar = (int)ch;
	}

	private boolean isSpace(int ch) {
		return ch == ' ' || ch == '\r' || ch == '\n' || ch == '\t';
	}
	private boolean isLatin(int ch) {
		return ('a' <= ch && ch <= 'z')
			|| ('A' <= ch && ch <= 'Z')
			|| ch == '_' || ch == '$';
	}
	private boolean isDigit(int ch) {
		return ('0' <= ch && ch <= '9');
	}

	private void error() {
		error("");
	}
	private void error(String msg) {
		throw new RuntimeException(msg);
	}
	private void error(String key, int val) {
		throw new RuntimeException("Not "+key+" "+val);
	}
}






