/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs.proc;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import java.text.* ;
import javax.servlet.*;
import javax.servlet.http.*;
import javax.xml.transform.*;
import javax.xml.transform.stream.*;

import org.kotemaru.auth.* ;
import org.kotemaru.util.* ;
import org.kotemaru.wsjs.* ;
import org.kotemaru.wsjs.exjs.* ;
import org.kotemaru.util.jsg.*;
import org.kotemaru.util.jsg.Source;

public class ExjsProc extends ProcessorBase {
	private static final String EXT_EXJS = ".exjs";
	private static final String CT_JAVASCRIPT = "application/javascript; charset=utf-8";
	private static final String EXJS_BNF = "/lib/exjs/exjs.bnf";
	private static final String EXJS_JSG = "/lib/exjs/exjs.jsg";

	private long lastUpdateTime = -1;
	private byte[] buffer = null;
	private final String cacheCtrl;

	public ExjsProc() {
		super();
		cacheCtrl = null;
	}

	private ExjsProc(WsjsContext ctx, Page page) throws IOException {
		super(page);
		String pageName = page.getPageName();
		cacheCtrl = Config.hasPermitRead(pageName, PAMFactory.getVisitUser())
				? "public" : "private";
		update(ctx);
	}
	public Processor getInstance(WsjsContext ctx, Page page) throws IOException  {
		if (!page.hasExt(EXT_EXJS)) return null;
		if (!page.exists()) return null;
		return new ExjsProc(ctx, page);
	}
	public void dispose() {
		lastUpdateTime = -1;
		buffer = null;
		super.dispose();
	}

	public void processing(WsjsContext ctx) throws IOException {
		super.access();
		HttpServletResponse res = ctx.getResponse();
		byte[] buff = getCacheBuffer(ctx);
		if (ProcUtil.check304(ctx, lastUpdateTime, cacheCtrl)) return;
		res.setContentType(CT_JAVASCRIPT);
		res.setContentLength(buff.length);
		res.getOutputStream().write(buff);
	}


	public synchronized byte[] getCacheBuffer(WsjsContext ctx) throws IOException {
		update(ctx);
		return buffer;
	}

	public double getCacheScore() {
		return super.getCacheScore() * 10.0;
	}

	public boolean update(WsjsContext ctx)  throws IOException {
		if (page.lastModified() > lastUpdateTime) {
			lastUpdateTime = page.lastModified();
			try {
				buffer = compile(ctx, page);
				super.accessReset(buffer.length);
			} catch (ErrorPageException e) {
				throw e;
			} catch (Exception e) {
				e.printStackTrace();
				throw new ErrorPageException(500, 
					"exjsCompileError", page.getPageName(), e.toString(), e);
			}
			return true;
		}
		return false;		
	}



	// TODO: 最適化
	private byte[] compile(WsjsContext ctx, Page page) throws Exception {
		Processor bnfProc = ctx.getProcessor(EXJS_BNF);
		Processor jsgProc = ctx.getProcessor(EXJS_JSG);
		String bnfSrc = bnfProc.getPage(ctx).getBodyString();
		String jsgSrc = jsgProc.getPage(ctx).getBodyString();

		// init BNF. TODO: cache
		BnfParser bnfParser = new BnfParser(new Source(bnfSrc));
		if (bnfParser.parse() == null) {
			throw new ErrorPageException(500,
					"exjsCompileError", page.getPageName(), bnfParser.getDebugString());
		}

		// init JSG. .jsg -> XSLT.  TODO: cache
		JsgParser jsgParser = new JsgParser(new Source(jsgSrc));
		if (jsgParser.parse() == null) {
			throw new ErrorPageException(500,
					"exjsCompileError", page.getPageName(), jsgParser.getDebugString());
		}
		StreamSource src =
			new StreamSource(new StringReader(jsgParser.getString()));
		Transformer trans =
			TransformerFactory.newInstance().newTransformer(src) ;
		ErrHandler errHandler = new ErrHandler(ctx);
		trans.setErrorListener(errHandler);
		trans.setOutputProperty(OutputKeys.METHOD, "text");

		// .exjs -> XML
		String exjsSrc = page.getBodyString();
		Tokenizer tokenizer = new org.kotemaru.wsjs.exjs.JsTokenizer();
		BnfDriver driver = new BnfDriver(bnfParser, new Source(exjsSrc), tokenizer);
		if (driver.parse() == null) {
			throw new ErrorPageException(500,
					"exjsCompileError", page.getPageName(), driver.getDebugString());
		}
		String xml = driver.getString();
		//IOUtil.putPage("/tmp/1.xml", xml);

		// XML ->(XSLT)-> .js
		StreamSource inSource = new StreamSource(new StringReader(xml));
		//ByteArrayOutputStream bout = new ByteArrayOutputStream();
		Writer writer = new StringWriter();
		StreamResult outResult = new StreamResult(writer);
		try {
			trans.transform(inSource, outResult);
		} catch (Exception e) {
			if (errHandler.message != null) {
				e.printStackTrace();
				throw new ErrorPageException(500,
					"exjsCompileError", page.getPageName(), errHandler.message);
			} else {
				throw e;
			}
		}

		String res = trim(writer.toString());
		return res.getBytes("UTF-8");
	}

	private static class	ErrHandler implements ErrorListener {
		public String message = null;
		private final WsjsContext wsjsContext;
		public ErrHandler(WsjsContext ctx) {
			wsjsContext = ctx;
		}

		public void warning(TransformerException ex) {
			String msg = ex.getMessage();
			if (msg.startsWith("!!!")) {
				message = msg.substring(3);
			}
			wsjsContext.getLog().warn(ex.toString());
		}
		public void error(TransformerException ex) {
			wsjsContext.getLog().error(ex.toString());
		}
		public void fatalError(TransformerException ex) {
			wsjsContext.getLog().error(ex.toString());
		}
	};

//----------------------------------------------------------------
	private String SPACE = "                                                                ";
	private String trim(String str) {
		StringBuffer sbuf = new StringBuffer();
		int indent = 0;
		String[] lines = str.split("\n");
		for (int i=0; i<lines.length; i++) {
			String line = lines[i].trim();
			if (line.length() == 0) continue;
			int cnt0 = count(line, '{');
			int cnt1 = count(line, '}');
			if (line.charAt(0) == '}') {
				indent += (cnt0-cnt1);
				sbuf.append(indentSpace(indent));
			} else {
				sbuf.append(indentSpace(indent));
				indent += (cnt0-cnt1);
			}
			sbuf.append(line);
			sbuf.append('\r');
			sbuf.append('\n');
		}
		return sbuf.toString();
	}
	private int count(String line, char ch) {
		int cnt = 0;
		int pos = 0;
		while (pos < line.length() && (pos=line.indexOf(ch, pos)+1)>0) cnt++;
		return cnt;
	}
	private String indentSpace(int len) {
		return SPACE.substring(0, len*2);
	}


/*
	static class IndentWriter extends Writer {
		private StringBuffer sbuf = new StringBuffer();
		private boolean isNL = true;

		public void write(char[] buf, int off, int len) {
			sbuf.append(buf, off, len);
			for (int i=0; i<len; i++) {
				if (buf[off+i] == '\n') {
				}
			}

			String[] lines = s.split("\n");
			for (int i=0; i<lines.length; i++) {
				String line = lines[i].trim();
				if (line.length() == 0) continue;
				sbuf.append(line);
				sbuf.append('\n');
			}
		}
		public String toString() {
			return sbuf.toString();
		}

		public void close() {
			// nop.
		}
		public void flush() {
			// nop.
		}
	}
*/

}
