/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs.gae.lowapi;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import javax.servlet.*;
import javax.servlet.http.*;
import org.kotemaru.util.* ;
import org.kotemaru.auth.* ;
import org.kotemaru.wsjs.* ;
import java.util.jar.* ;


import com.google.appengine.api.datastore.Blob;
import com.google.appengine.api.datastore.Key;
import com.google.appengine.api.datastore.KeyFactory;
import com.google.appengine.api.datastore.*;
import com.google.appengine.api.memcache.*;

import java.util.logging.*;

/**
 */
public class RepositoryImpl extends RepositoryBase implements Const {
	public static final DatastoreServiceFactory DSF = new DatastoreServiceFactory();
	private static final String NS_MEMCACHE = RepositoryImpl.class.getName();
	private MemcacheService memcache = MemcacheServiceFactory.getMemcacheService(NS_MEMCACHE);

	/**
	 * コンストラクタ。
	 */
	public RepositoryImpl(String masterFileDir) {
		LOG.info("Repository(GAE low API): "+masterFileDir);
		if (masterFileDir != null) {
			//initRepository(new File(masterFileDir));
			initRepository2(new File(masterFileDir));
		}
	}


	protected void putCache(String pageName, Processor proc) {
		try {
			super.putCache(pageName, proc);
			memcache.put(pageName, IOUtil.serialize(proc));
			//memcache.put(pageName, proc);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
//System.out.println("putCache:"+pageName);
	}
	protected Processor getCache(String pageName) {
		try {
			Processor proc = super.getCache(pageName);
			if (proc != null) return proc;
//System.out.println("getCache:"+pageName);
			byte[] data = (byte[])memcache.get(pageName);
			if (data == null) return null;
			proc = (Processor) IOUtil.deserialize(data);
			//proc = (Processor) memcache.get(pageName);
			return proc;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	protected void removeCache(String pageName, Processor proc) {
//System.out.println("removeCache:"+pageName);
		super.removeCache(pageName, proc);
		memcache.delete(pageName);
	}

	public void removeCache(String pageName) {
		super.removeCache(pageName);
		memcache.delete(pageName);
	}

	public Page getPage(WsjsContext ctx, String pageName) throws IOException {
		Entity entity = getEntity(pageName, false);
		if (entity == null) return new PageImpl(this, pageName);
		return new PageImpl(this, entity);
	}

	public static Entity getEntity(String pageName, boolean withBody) {
		DatastoreService ds = DSF.getDatastoreService();
		return getEntity(ds, pageName, withBody);
	}
	public static Entity getEntity(DatastoreService ds, String pageName, boolean withBody) {
		Query query = new Query(KIND);
		Key key = KeyFactory.createKey(KIND, pageName);
		query.addFilter(Entity.KEY_RESERVED_PROPERTY, 
								Query.FilterOperator.EQUAL , key);
		PreparedQuery pq = ds.prepare(query);
		Entity entity = pq.asSingleEntity();
		if (entity == null) return null;
		if (withBody) ((Blob)entity.getProperty(BODY)).getBytes();
		return entity;
	}

	public static List<Entity> listEntity(String pageName, boolean withBody) {
		DatastoreService ds = DSF.getDatastoreService();
		Query query = new Query(KIND);
		query.addFilter(PARENT_PAGE_NAME,
								Query.FilterOperator.EQUAL, pageName);
		PreparedQuery pq = ds.prepare(query);
		Iterator<Entity> sel = pq.asIterator();
		List<Entity> list = new ArrayList<Entity>();
		while(sel.hasNext()){
			Entity entity = sel.next();
			if (withBody) ((Blob)(entity.getProperty(BODY))).getBytes();
			list.add(entity);
		}
		return list;
	}
	public static Entity newEntity(String pageName) {
		Entity entity = new Entity(KIND, pageName);
	  	entity.setProperty(DIRECTORY, false);
	  	entity.setProperty(PARENT_PAGE_NAME, getParentPageName(pageName));
		return entity;
	}

	public static void putEntity(Entity entity) {
		DatastoreService ds = DSF.getDatastoreService();
		ds.put(entity);
	}

	public static boolean delEntity(Entity entity) {
		DatastoreService ds = DSF.getDatastoreService();
		try {
			ds.delete(entity.getKey());
			return true;
		} catch (IllegalArgumentException e) {
			LOG.warn(e.toString());
			return false;
		}
	}
	public Page getParentPage(PageImpl page) {
		String pageName = getParentPageName(page.getPageName());
		Entity entity = getEntity(pageName, false);
		if (entity == null) return new PageImpl(this, pageName);
		return new PageImpl(this, entity);
	}

	private static void initRepository(File rootDir) {
		try {
			if (getEntity("/", false) != null) return;

			Entity entity =	new Entity(KIND, "/");
			//entity.setProperty(PAGE_NAME, "/");
			entity.setProperty(LENGTH, 0);
	 		entity.setProperty(LAST_MODIFIED, rootDir.lastModified());
	  		entity.setProperty(BODY, null);
	  		entity.setProperty(DIRECTORY, true);
	  		entity.setProperty(PARENT_PAGE_NAME, null);

			putEntity(entity);

			int prefixLen = rootDir.getCanonicalPath().length();
			putDir(prefixLen, rootDir);
		} catch (Exception e) {
			LOG.error(e);
			throw new RuntimeException(e);
		}
	}
	private static void putDir(int prefixLen, File dir) throws IOException {
		DatastoreService ds = DSF.getDatastoreService();
		File[] files = dir.listFiles();
		if (files == null) return;

		for (int i=0; i<files.length; i++) {
			File file = files[i];
			String pageName = file.getCanonicalPath().substring(prefixLen);
			Entity entity =	new Entity(KIND, pageName);
			//entity.setProperty(PAGE_NAME, pageName);
			entity.setProperty(LENGTH, file.length());
	 		entity.setProperty(LAST_MODIFIED, file.lastModified());
	  		entity.setProperty(DIRECTORY, file.isDirectory());
	  		entity.setProperty(PARENT_PAGE_NAME, getParentPageName(pageName));

			LOG.debug("init repository: "+pageName);
			LOG.info("init repository: "+pageName);
			if (!file.isDirectory()) {
				byte[] data = IOUtil.getFileBytes(file);
  				entity.setProperty(BODY, new Blob(data));
			}
			ds.put(entity);

			if (file.isDirectory()) {
				putDir(prefixLen, file);
			}
		}
	}
    public static String getParentPageName(String pageName) {
        int pos = pageName.lastIndexOf('/');
        if (pos > 0) {
            return pageName.substring(0,pos);
        } else {
            return "/";
        }
    }

	private static void initRepository2(File jarFile) {
		if (getEntity("/", false) != null) return;
		try {
			initRepositoryJar(jarFile);

			Entity entity =	new Entity(KIND, "/");
			entity.setProperty(LENGTH, 0);
	 		entity.setProperty(LAST_MODIFIED, jarFile.lastModified());
	  		entity.setProperty(BODY, null);
	  		entity.setProperty(DIRECTORY, true);
	  		entity.setProperty(PARENT_PAGE_NAME, null);
			putEntity(entity);
		} catch (Exception e) {
			LOG.error(e);
			throw new RuntimeException(e);
		}
	}
	private static void initRepositoryJar(File jarFile) throws IOException {
		DatastoreService ds = DSF.getDatastoreService();
		InputStream fileIn = new FileInputStream(jarFile);
		try {
			JarInputStream jarIn = new JarInputStream(fileIn);
			JarEntry file = jarIn.getNextJarEntry();
			while (file != null) {
				String pageName =	"/"+file.getName();
				pageName = pageName.replaceFirst("/$","");

				if (getEntity(ds, pageName, false) != null) return;

				Entity entity =	new Entity(KIND, pageName);
				entity.setProperty(LENGTH, 0);
		 		entity.setProperty(LAST_MODIFIED, file.getTime());
	  			entity.setProperty(PARENT_PAGE_NAME, getParentPageName(pageName));
	 	 		entity.setProperty(DIRECTORY, file.isDirectory());
				if (!file.isDirectory()) {
					byte[] data = IOUtil.streamToBytes(jarIn);
  					entity.setProperty(BODY, new Blob(data));
					entity.setProperty(LENGTH, data.length);
				}
				ds.put(entity);
				LOG.info("init repository: "+pageName);
				file = jarIn.getNextJarEntry();
			}
		} finally {
			fileIn.close();
		}
	}

}
