/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs.gae.lowapi;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import javax.servlet.*;
import javax.servlet.http.*;
import org.kotemaru.util.* ;
import org.kotemaru.auth.* ;
import org.kotemaru.auth.* ;
import org.kotemaru.wsjs.* ;

import com.google.appengine.api.datastore.Blob;
import com.google.appengine.api.datastore.*;

/**
 * Entityを実体して持つページ。
 */
public class PageImpl implements Page, Const {

	transient private RepositoryImpl repository;
	private String pageName = null;
	private Entity entity = null;
	private Set permitWriteRoles = null;

	public PageImpl(RepositoryImpl repo, Entity entity) {
		this.repository = repo;
		this.entity = entity;
	}
	public PageImpl(RepositoryImpl repo, String name) {
		this.repository = repo;
		this.pageName = name;
	}

	public String getPageName() {
		if (entity == null) return pageName;
		//return entity.getProperty(PAGE_NAME).toString();
		return entity.getKey().getName();
	}
	public boolean isDirectory() {
		if (entity == null) return false;
		return (Boolean) entity.getProperty(DIRECTORY);
	}
	public long lastModified() {
		return (Long) entity.getProperty(LAST_MODIFIED);
	}
	public long length() {
		return (Long) entity.getProperty(LENGTH);
	}
	public boolean exists() {
		return (entity != null);
	}

	public String getLastName() {
		String pname = getPageName();
		int pos = pname.lastIndexOf('/');
		return pname.substring(pos+1);
	}
	public Page[] listPages() {
		List<Entity> list =
			RepositoryImpl.listEntity(getPageName(), false);
		Page[] result = new Page[list.size()];
		for (int i=0; i<result.length; i++) {
			result[i] = new PageImpl(repository, list.get(i));
		}
		return result;
	}

	public boolean hasExt(String ext) {
		return getPageName().endsWith(ext);
	}


	public String getBodyString() throws IOException {
		return getBodyString(Config.getEncoding());
	}
	public String getBodyString(String charset) throws IOException {
		byte[] body = getBodyBytes();
		if (body == null) return null;
		return new String(body, charset);
	}
	public byte[] getBodyBytes() throws IOException {
		if (entity == null) return null;
		Entity e = repository.getEntity(getPageName(), true);
		if (e == null) return null;
		return ((Blob)entity.getProperty(BODY)).getBytes();
	}
	public void writeBody(OutputStream out) throws IOException {
		writeBody(out, true);
	}
	public void writeBody(OutputStream out, boolean isClose) throws IOException {
		byte[] body = getBodyBytes();
		try {
			if (body != null) out.write(body);
		} finally {
			if (isClose) out.close();
		}
	}

	public void putBodyString(String data) throws IOException {
		putBodyString(data, Config.getEncoding());
	}
	public void putBodyString(String data, String charset) throws IOException {
		putBodyBytes(data.getBytes(charset));
	}
	public void putBodyBytes(byte[] data) throws IOException {
		if (entity == null) {
			entity = RepositoryImpl.newEntity(pageName);
		}
		entity.setProperty(LENGTH, data.length);
 		entity.setProperty(LAST_MODIFIED, System.currentTimeMillis());
  		entity.setProperty(BODY, new Blob(data));
		repository.removeCache(pageName);
		RepositoryImpl.putEntity(entity);
		if (repository != null) repository.removeCache(getPageName());
	}

	public void putBodyStream(InputStream in) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		IOUtil.transrate(in, bout, false, false);
		byte[] body = bout.toByteArray();
		putBodyBytes(body);
	}
	public OutputStream getBodyOutputStream(boolean isAppend) throws IOException {
		if (entity == null) {
			entity = RepositoryImpl.newEntity(pageName);
		}
		if (repository != null) repository.removeCache(getPageName());
		return new BodyOutputStream(entity, isAppend);
	}
	public boolean mkdir() throws IOException {
		if (this.entity != null) return false;
		if (!hasParentDir(true)) return false;

		if (entity == null) {
			entity = RepositoryImpl.newEntity(pageName);
		}
  		entity.setProperty(DIRECTORY, true);
		entity.setProperty(LENGTH, 0);
 		entity.setProperty(LAST_MODIFIED, System.currentTimeMillis());
		repository.removeCache(pageName);
		RepositoryImpl.putEntity(entity);
		this.entity = entity;
		return true;
	}

	// 親Dirのチェック又は生成。
	private boolean hasParentDir(boolean isCreate) throws IOException  {
		Page parentPage = repository.getParentPage(this);
		if (parentPage != null) {
			if (parentPage.exists()) {
				return parentPage.isDirectory();
			} else if (isCreate) {
				return parentPage.mkdir();
			}
		}
		return false;
	}


	public boolean remove() throws IOException {
		if (repository != null) repository.removeCache(getPageName());
		return RepositoryImpl.delEntity(entity);
	}

	public Set<String> getPermitWriteRoles() {
		if (permitWriteRoles == null) {
			permitWriteRoles =Config.getPermitWriteRoles(this);
		}
		return permitWriteRoles;
	}

	public int compareTo(Object obj) {
		return getPageName().compareTo(((Page)obj).getPageName());
	}

	public String toString() {
		return "Page["+getPageName()+","+entity+"]";
	}

}
