/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import org.kotemaru.util.* ;
import javax.servlet.*;
import javax.servlet.http.*;
import org.kotemaru.auth.*;
import org.kotemaru.wsjs.proc.*;

/**
 * WSJSに対するリクエスト毎のコンテキスト。
 * <ul>
 * <li>サーブレット、リクエスト、レスポンス、ユーザを持つ。
 * </ul>
 */
public class WsjsContext implements WsjsConst {
	public static final Log DEFAULT_LOG = new Log();
	private WsjsServletIF wsjsServlet;
	private final User user;
	private Log log = DEFAULT_LOG;
	private AppConfig appConfig;

	private	final HttpServletRequest request;
	private final HttpServletResponse response;
	public HttpServletRequest getRequest(){
		return request;
	}
	public HttpServletResponse getResponse(){
		return response;
	}
	public Servlet getServlet(){
		return null;
	}

	/**
	 * コンストラクタ。<ul>
	 * <li>ユーザはリクエストから自動的に取得する。
	 * </ul>
	 * @param servlet     サーブレット・インスタンス
	 * @param request     HttpServletRequest インスタンス
	 * @param response    HttpServletResponse インスタンス
	 */
	public WsjsContext(WsjsServletIF servlet, 
			HttpServletRequest request, HttpServletResponse response) 
				throws Exception
	{
		//super(servlet, request, response);
		wsjsServlet = servlet;
		this.request = request;
		this.response = response;
		user = PAMFactory.getUser(request);
		if (!user.isRegistered()) {
			//HttpSession session = requset.getSession();
			//session.getAttribute("registerMode","yes");
			//String pathInfo = request.getPathInfo();
			String pathInfo = request.getRequestURI();
			if (!pathInfo.startsWith(LOGIN_DIR)) {
				throw new UnregisterUserException(this, user, "/");
			}
		}

		init();
	}


	/**
	 * ユーザを返す。
	 * @return ユーザ
	 */
	public User getUser() {
		return user;
	}

	public AppConfig getAppConfig() {
		return appConfig;
	}
	public Log getLog() {
		return log;
	}
	public Page getPageAccessor(String pageName) throws IOException {
		return new PageAccessor(this, getPage(pageName));
	}
	public Page getPage(String pageName) throws IOException {
		Repository repo = getWsjsServlet().getRepository();
		return repo.getPage(this, pageName);
	}

	/**
	 * サーブレットを返す。
	 * @return サーブレット
	 */
	public WsjsServletIF getWsjsServlet() throws IOException {
		return wsjsServlet;
	}

	/**
	 * ページ・プロセッサを取得する。
	 * <li>ページ名はリクエストのpathInfoから得る。
	 * <li>{@link #getProcessor(String)}に依存。
	 * @return ページ・プロセッサ
	 * @throws AccessDeniedException ユーザが権限を持たない場合。
	 */
	public Processor getProcessor() throws IOException {
		//String pageName = getRequest().getPathInfo();
		String pageName = getRequest().getRequestURI();
		return getProcessor(pageName);
	}

	/**
	 * ページ・プロセッサを取得する。
	 * <li><S>"ページ名.link" のページが存在すればそのプロセッサを返す。</S>
	 * <li>{@link #getProcessorNoLink(String,boolean)}に依存。
	 * @param pageName ページ名
	 * @return ページ・プロセッサ
	 * @throws AccessDeniedException ユーザが権限を持たない場合。
	 */
	public Processor getProcessor(String pageName) throws IOException {
/*
		Processor proc = getProcessorNoLink(pageName+LinkProc.EXT_LINK, false);
		if (proc == null) {
			proc = getProcessorNoLink(pageName, true);
		}
*/
		Processor proc = getProcessorNoLink(pageName, true);
		return proc;
	}


	/**
	 * ページ・プロセッサを取得する。
	 * <li>"ページ名.link" のページは検索しない。
	 * <li>アプリケーション用のリポジトリを検索後、
	 *     無ければシステムのリポジトリを検索する。
	 * @param pageName ページ名
	 * @param isCheck  権限のチェックを行うか否か
	 * @return ページ・プロセッサ。当該プロセッサが無ければnullを返す。
	 * @throws AccessDeniedException ユーザが権限を持たない場合。
	 */
	public Processor getProcessorNoLink(String pageName, boolean isCheck) throws IOException {
		Repository repository = wsjsServlet.getRepository();
		Processor proc = repository.getProcessor(this, pageName, isCheck);
		if (proc == null) {
			Repository systemRepository = wsjsServlet.getSystemRepository();
			proc = systemRepository.getProcessor(this, pageName, isCheck);
		}
		return proc;
	}
	/**
	 * ページ名の補正。
	 * <li>/_wsjs_/からのアクセスは/_wsjs_ を外す。
	 * @param pageName ページ名
	 * @return 補正後のページ名
	 */
	private String correctPageName(String pageName) throws MalformedURLException {
		String refer = getRequest().getHeader("Referer");
		if (refer == null) return pageName;
		URL url = new URL(refer);
		if (!url.getFile().startsWith("/_wsjs_/")) return pageName;
		return "/_wsjs_"+pageName;
	}


	private void init() throws IOException {
		//String pathInfo = getRequest().getPathInfo();
		String pathInfo = getRequest().getRequestURI();
		if (!pathInfo.startsWith(HOME_DIR)) return;
		int pos = pathInfo.indexOf('/',6);
		if (pos == -1) return;
		String userName = pathInfo.substring(6, pos);
		Repository repository = wsjsServlet.getRepository();

		User user = PAMFactory.getUser(userName);
		appConfig = getAppConfig(user);

		if (appConfig.isServerLog()) {
			String logFile = HOME_DIR+userName+PRIVATE_DIR+"server-log.txt";
			Page logPage = repository.getPage(this, logFile);
			this.log = new AppLog(logPage);
		}
	}


	public AppConfig getAppConfig(User user) {
		Map attrs = user.getAttributes();
		AppConfig conf;
		if (attrs instanceof AppConfig) {
			conf = (AppConfig) attrs;
		} else {
			conf = new AppConfig(user, attrs);
			user.setAttributes(conf);
		}
		return conf;
	}
	public void setAppConfig(User user, AppConfig conf) {
		user.setAttributes(conf);
	}

}
