/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import javax.servlet.*;
import javax.servlet.http.*;
import org.kotemaru.util.* ;
import org.kotemaru.auth.* ;

/**
 * ページのリポジトリ管理クラス。
 * <ul>
 * <li>ページをOSのファイルシステムにマップする。
 * <li>ルートとなるディレクトリを持つ。
 * <li>ページのエントリをキャッシュとして持つ。
 * <li>キャッシュの全体サイズを管理する。
 * <li>
 * </ul>
 */
public abstract class RepositoryBase implements Repository {
	protected HashMap<String,Processor> cache = new HashMap();
	protected int totalCacheSize = 0;


	/**
	 * ページ・プロセッサの取得。<ul>
	 * <li>isCheckがtrueでユーザに読み込み権が無ければ例外を上げる。
	 * <li>キャッシュにプロセッサが存在すればそれを返す。
	 * <li>キャッシュに無い場合は設定に従ってプロセッサを生成する。
	 * <li>新規プロセッサはキャッシュに登録して所属リポジトリに自身を設定する。
	 * <li>新規プロセッサが生成できない場合はnullを返す。
	 * <li>プロセッサがキャッシュ不要になっていた場合はキャッシュしない。
	 * <li>プロセッサが無効になっていた場合はキャッシュから削除する。
	 * </ul>
	 * @param ctx      WSJSコンテキスト
	 * @param pageName ページ名
	 * @param isCheck  権限のチェックを行うか否か
	 * @return プロセッサのインスタンス。新規プロセッサが生成できない場合はnullを返す。
	 * @throws AccessDeniedException ユーザが権限を持たない場合。
	 */
	public Processor getProcessor(WsjsContext ctx, String pageName, boolean isCheck) throws IOException {
		pageName = IOUtil.formalPageName(pageName);
		if (isCheck && !Config.hasPermitRead(pageName, ctx.getUser())) {
			throw new AccessDeniedException(ctx, ctx.getUser(), pageName);
		}

		synchronized (cache) {
			Processor proc = getCache(pageName);
			//if (!ProxyProc.isLocalhost(req)) proc = null;
			if (proc != null && !proc.isAvailable(ctx)) {
				removeCache(pageName, proc);
				proc = null;
			}

			if (proc == null) {
				Page page = getPage(ctx, pageName);
				proc = Config.getProcessor(ctx, page);
				if (proc == null) return null;
			}
			if (proc.isCachable()) {
				proc.setRepository(this);
				putCache(pageName, proc);
			}
			return proc;
		}
	}
	protected void putCache(String pageName, Processor proc) {
		cache.put(pageName, proc);
	}
	protected Processor getCache(String pageName) {
		return cache.get(pageName);
	}

	protected void removeCache(String pageName, Processor proc) {
		synchronized (cache) {
			totalCacheSize = totalCacheSize - proc.getCacheSize();
			proc.dispose();
			cache.remove(pageName);
		}
	}
	public void removeCache(String pageName) {
		synchronized (cache) {
			Processor proc = cache.get(pageName);
			if (proc == null) return;
			removeCache(pageName, proc);
		}
	}


	/**
	 * キャッシュサイズの通知。<ul>
	 * <li>各プロセッサはキャッシュバッファを確保した場合に呼び出す。
	 * <li>キャッシュバッファの合計を保持し上限を越えた場合に優先度の低いキャッシュを削除する。
	 * <li>優先度は {@link Processor#getCacheScore()} による。
	 * </ul>
	 * @param oldSize 旧キャッシュサイズ
	 * @param newSize 新キャッシュサイズ
	 */
	public void addTotalCacheSize(int oldSize, int newSize) {
		totalCacheSize = totalCacheSize - oldSize + newSize;
		if (totalCacheSize > Config.getLimitTotalCacheSize()) gc();
	}

	private void gc() {
		synchronized (cache) {
			Iterator<Map.Entry<String,Processor>> ite;

			int fittingSize = Config.getLimitTotalCacheSize() * 70 / 100;
			double min = 0.1;
			while (totalCacheSize > fittingSize) {
				int count = 0;
				double totalScore = 0.0;
				int    totalSize = 0;
				ite = cache.entrySet().iterator();
				while(ite.hasNext()){
					Map.Entry<String,Processor> ent = ite.next();
					Processor proc = ent.getValue();
					double score = proc.getCacheScore();
					if (score < min) {
						proc.dispose();
					} else {
						totalScore = totalScore + score;
						totalSize  += proc.getCacheSize();
						count++;
					}
				}
				totalCacheSize = totalSize;
				LOG.info("Dispose cache. score="+min+", free="
					+(100-totalCacheSize*100/Config.getLimitTotalCacheSize())+"%");
				min = (totalScore/count) / 2;
			}
		}
	}

}
