/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import java.text.* ;
import javax.servlet.*;
import javax.servlet.http.*;
import org.kotemaru.util.* ;
import org.mozilla.javascript.*;

/**
 * ページ・プロセッサのベース。
 * <ul>
 * <li>ページ名、ファイル、キャッシュスコア、所属リポジトリ、有効無効まで実装。
 * <li>このクラスを継承する場合のルール。<ul>
 *  <li>processing()の先頭で {@link #access()} を呼ぶ。
 *  <li>キャッシュバッファか確保したときに {@link #accessReset(int)} を呼ぶ。
 *  <li>dispose()の最後に super.dispose() を呼ぶ。
 *  <li>
 * </ul>
 * </ul>
 */
public abstract class ProcessorBase implements Processor {
	private boolean enable = true;
	private long createTime = -1;
	private int accessCount = 0;
	private int cacheSize = 0;
	transient private Repository repository = null;

	protected final Page page;

	/**
	 * ファクトリを生成するコンストラクタ。
	 * <li>ファクトリとしてのみ使用可能。
	 */
	protected ProcessorBase() {
		this(null);
	}

	/**
	 * コンストラクタ。
	 * @param pageName ページ名
	 * @param file     ページのファイル
	 */
	protected ProcessorBase(Page page) {
		this.page = page;
	}

	public void setRepository(Repository repo) {
		repository = repo;
		repository.addTotalCacheSize(0, cacheSize);
	}

	/**
	 * {@inheritDoc}
	 * <li>キャッシュサイズを０にリセット。
	 * <li>継承クラスのdispose()は呼び出し必須。
	 */
	public void dispose() {
		cacheSize = 0;
	}

	public String getPageName() {
		return page.getPageName();
	}
	public Page getPage(WsjsContext ctx) throws IOException {
		return page;
	}

	/**
	 * {@inheritDoc}
	 * <li>ファイルが存在すればtrueを返す。
	 */
	public boolean isAvailable(WsjsContext ctx) throws IOException  {
		Page page = getPage(ctx);
		if (page == null) return false;
		return page.exists();
	}

	public boolean isEnable() {
		return enable;
	}
	public void setEnable(boolean b) {
		enable = b;
	}

	/**
	 * {@inheritDoc}
	 * <li>ダミー。常にnullを返す。
	 */
	public byte[] getCacheBuffer(WsjsContext ctx) throws IOException {
		return null;
	}
	/**
	 * {@inheritDoc}
	 * <li>ダミー。常にnullを返す。
	 */
	public Script getScript(WsjsContext ctx) throws IOException {
		return null;
	}
	/**
	 * {@inheritDoc}
	 * <li>常に有効。
	 */
	public boolean isCachable() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 * <li>基準スコアを返す。
	 */
	public double getCacheScore() {
		// 基準値 1.0 = access/sec=1 size=10k 
		double time = (double)(System.currentTimeMillis() - createTime)/1000;
		double score = (double)accessCount * 10240 / time / (cacheSize+1);
		return score;
	}

	/**
	 * {@inheritDoc}
	 * <li>accessReset(int)で設定されたキャッシュサイズを返す。
	 */
	public int getCacheSize() {
		return cacheSize;
	}

	/**
	 * アクセス数をカウントする。
	 */
	protected void access() {
		accessCount++;
	}

	/**
	 * キャッシュ情報を初期化する。<ul>
	 * <li>キャッシュサイズを変更してリポジトリに通知する。
	 * <li>キャッシュの生成時間を現在時に更新する。
	 * <li>アクセス数を０にリセットする。
	 * </ul>
	 * @param size  キャッシュサイズ
	 */
	protected void accessReset(int size) {
		if (repository != null) {
			repository.addTotalCacheSize(cacheSize, size);
		}
		createTime = System.currentTimeMillis();
		cacheSize = size;
		accessCount = 0;
	}

}
