/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import javax.servlet.*;
import javax.servlet.http.*;
import org.mozilla.javascript.*;

/**
 * ページ・プロセッサのインターフェース。
 * <ul>
 * <li>ファクトリとしても機能するので引数無しコンストラクタが必須。
 * <li>基本的な要素はページ名、ファイル、キャッシュバッファ。
 * <li>このクラス自体がキャッシュのエントリとなる。
 * <li>キャッシュの全体管理は {@link Repository} クラスが行うので通知の必要がある。
 * <li>
 * </ul>
 */
public interface Processor extends Serializable {
	/**
	 * インスタンス生成。<ul>
	 * <li>指定された条件のページ・プロセッサのインスタンスを返す。
	 * <li>条件が一致しない場合は null を返す。<ul>
	 *  <li>ファイルが存在しない。
	 *  <li>拡張子が異なる。
	 *  <li>他、実装による。
	 * </ul>
	 * </ul>
	 * @param ctx      WSJSコンテキスト
	 * @param page     ページ
	 * @return プロセッサのインスタンス。条件が一致しない場合はnullを返す。
	 */
	public Processor getInstance(WsjsContext ctx, Page page) throws IOException;

	/**
	 * ページ処理の実行。
	 * <li>ページを処理しサーブレットの応答を返す。
	 * @param ctx      WSJSコンテキスト
	 */
	public void processing(WsjsContext ctx) throws ServletException, IOException;

	/**
	 * ページ名を返す。
	 * @return ページ名。
	 */
	public String getPageName();

	/**
	 * ページのファイルを返す。
	 * @param ctx      WSJSコンテキスト
	 * @return ファイル。
	 */
	public Page   getPage(WsjsContext ctx) throws IOException;

	/**
	 * キャッシュバッファの内容を返す。<ul>
	 * <li>キャッシュできなければnullを返す。
	 * <li>キャッシュが空でキャッシュ可能ならば内容を読み込んで返す。
	 * </ul>
	 * @param ctx      WSJSコンテキスト
	 * @return バイト列のデータ。持っていなければnullを返す。
	 */
	public byte[] getCacheBuffer(WsjsContext ctx) throws IOException;

	/**
	 * ページをコンパイルしたスクリプトを返す。<ul>
	 * <li>JavaScriptページでなければ null を返す。
	 * <li>通常はキャッシュとして保存する。
	 * </ul>
	 * @param ctx      WSJSコンテキスト
	 * @return コンパイルしたスクリプト。持っていなければnullを返す。
	 */
	public Script getScript(WsjsContext ctx) throws IOException;

	/**
	 * このページが有効か否かを返す。<ul>
	 * <li>ページのファイルが削除された場合などキャッシュが
	 *     外的要因で無効になっている場合がある。
	 * </ul>
	 * @param ctx      WSJSコンテキスト
	 * @return ページの有効／無効
	 */
	public boolean isAvailable(WsjsContext ctx) throws IOException;

	/**
	 * 所属リポジトリの設定。<ul>
	 * <li>リポジトリにキャッシュとしてエントリされた場合に設定される。
	 * <li>リポジトリの設定されたインスタンスはキャッシュバッファを確保した場合に
	 *     {@link Repository#addTotalCacheSize(int,int)}
	 *     を呼び出して通知しなければならない。
	 * </ul>
	 * @param repo  所属リポジトリ
	 */
	public void setRepository(Repository repo);

	/**
	 * キャッシュの破棄。<ul>
	 * <li>保持しているキャッシュバッファを破棄する。
	 * <li>但し、再読み込み可能な状態でなければならない。
	 * </ul>
	 */
	public void dispose();

	/**
	 * キャッシュ可能か否かを返す。<ul>
	 * <li>キャッシュ不可の場合はエントリもされずインスタンスは使い捨てとなる。
	 * </ul>
	 * @return キャッシュの有効／無効
	 */
	public boolean isCachable();

	/**
	 * GDSFに基づくキャッシュの優先度を返す。<ul>
	 * <li>値が大きい程、優先度が高い。
	 * <li>基本スコア = (アクセス数 * 10240 / キャッシュの生存時間[秒] / (キャッシュサイズ+1))
	 * <li>＃10kのデータが１秒に１回のアクセスで スコア=1.0 となる。 
	 * <li>基本スコアに実装毎の再読み込みのコストを補正して返す。
	 * </ul>
	 * @return キャッシュの優先度
	 */
	public double getCacheScore();

	/**
	 * キャッシュバッファのサイズを返す。
	 * @return キャッシュのサイズ
	 */
	public int getCacheSize();


	/**
	 * このクラスが有効か否かを返す。
	 * @return クラスが有効/無効
	 */
	public boolean isEnable();

	/**
	 * このクラスの有効／無効を設定する。
	 * <li>無効に設定するとリポジトリから無視される。
	 * @param b クラスが有効/無効
	 */
	public void setEnable(boolean b);
}
