/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.wsjs;

import java.io.* ;
import java.net.* ;
import java.util.* ;
import java.lang.reflect.* ;
import javax.servlet.*;
import javax.servlet.http.*;

import org.mozilla.javascript.*;
import org.kotemaru.util.* ;
import org.kotemaru.auth.* ;

public class PermitUtil {
	private static final String NONAME = "";
	private static final String MACRO_USER = "${user}";
	private static final String DENY = "${deny}";
	private static final String ASTER = "*";

	public static Matcher[] toDirMatchers(String[][] permissions) {
		Matcher[] matchers = new Matcher[permissions.length];
		for (int i=0; i<permissions.length; i++) {
			String patt = permissions[i][0];
			if (!patt.endsWith("/")) patt += "/";
			if (patt.indexOf(MACRO_USER) >= 0) {
				matchers[i] = new UserDirMatcher(patt);
			} else {
				matchers[i] = new PrefixMatcher(patt);
			}
		}
		return matchers;
	}

	public static boolean hasPermit(Matcher[] matchers, 
				String[][] permits, String pageName, User user) {
		for (int i=0; i<matchers.length; i++) {
			String dirPartsUserName = matchers[i].isMatch(pageName);
			if (dirPartsUserName == NONAME) {
				for (int j=1; j<permits[i].length	; j++) {
					if (user.hasRole(permits[i][j])) return true;
				}
				return false;
			} else if (dirPartsUserName != null) {
				for (int j=1; j<permits[i].length	; j++) {
					String role = permits[i][j];
					if (MACRO_USER.equals(role)) {
						if (user.getName().equals(dirPartsUserName)) return true;
					} else {
						if (user.hasRole(role)) return true;
					}
				}
				return false;
			}
		}
		return false;
	}


	public static Matcher[] toClassMatchers(String[][] permissions) {
		Matcher[] matchers = new Matcher[permissions.length];
		for (int i=0; i<permissions.length; i++) {
			String patt = permissions[i][0];
			if (patt.endsWith(".**")) {
				matchers[i] = new PrefixMatcher(patt.replaceAll("[*][*]$",""));
			} else if (patt.endsWith(".*")) {
				matchers[i] = new PackageMatcher(patt.replaceAll("[*]$",""));
			} else {
				matchers[i] = new FullMatcher(patt);
			}
		}
		return matchers;
	}
	public static boolean hasPermitClass(
				Matcher[] matchers, String[][] permits, 
				Matcher[] writeMatchers, String[][] writePermits, 
				String className, Page page, User owner) {

		Set<String> roles = page.getPermitWriteRoles();
		if (roles.size() == 0) return true; // 書込権無し
		boolean hasMacroUser = roles.contains(MACRO_USER);

		for (int i=0; i<matchers.length; i++) {
			if (matchers[i].isMatch(className) != null) {
				if (ASTER.equals(permits[i][1])) return true;
				if (DENY.equals(permits[i][1])) return false;
				Set roles0 = new HashSet(roles);

				for (int j=1; j<permits[i].length; j++) {
					String role = permits[i][j];
					if (hasMacroUser && owner != null && owner.hasRole(role)) {
						roles0.remove(MACRO_USER);
					}
					roles0.remove(role);
				}
				if (roles0.size() == 0) return true;
			}
		}
		return false;
	}

	public static Set<String> getPermitRoles(Matcher[] matchers, 
				String[][] permits, String pageName) {
		Set<String> result = new HashSet<String>();
		for (int i=0; i<matchers.length; i++) {
			String dirPartsUserName = matchers[i].isMatch(pageName);
			if (dirPartsUserName != null) {
				for (int j=1; j<permits[i].length	; j++) {
					result.add(permits[i][j]);
				}
			}
		}
		return result;
	}


	public static Matcher[] toJNDIMatchers(String[][] permissions) {
		Matcher[] matchers = new Matcher[permissions.length];
		for (int i=0; i<permissions.length; i++) {
			String patt = permissions[i][0];
			if (patt.indexOf(MACRO_USER) >= 0) {
				matchers[i] = new UserMatcher(patt);
			} else {
				matchers[i] = new FullMatcher(patt);
			}
		}
		return matchers;
	}
	public static boolean hasPermitJNDI(
				Matcher[] matchers, String[][] permits, 
				Matcher[] writeMatchers, String[][] writePermits, 
				String jndiName, Page page, User user) {

		Set<String> roles = page.getPermitWriteRoles();
		if (roles.size() == 0) return true; // 書込権無し
		boolean hasMacroUser = roles.contains(MACRO_USER);

		for (int i=0; i<matchers.length; i++) {
			if (matchers[i].isMatch(jndiName) != null) {
				if (ASTER.equals(permits[i][1])) return true;
				if (DENY.equals(permits[i][1])) return false;
				Set roles0 = new HashSet(roles);

				for (int j=1; j<permits[i].length; j++) {
					String role = permits[i][j];
					if (hasMacroUser && user.hasRole(role)) roles0.remove(MACRO_USER);
					roles0.remove(role);
				}
				if (roles0.size() == 0) return true;
			}
		}
		return false;
	}



//----------------------------------------------------------------------
// Matchers
	public static interface Matcher extends Serializable {
		public String isMatch(String source);
	}
	private static class FullMatcher implements Matcher {
		final String patt;
		public FullMatcher(String patt) {
			this.patt = patt;
		}
		public String isMatch(String source) {
			return source.equals(patt) ? NONAME : null;
		}
		public String toString(){return patt;}
	}
	private static class PrefixMatcher implements Matcher {
		final String patt;
		public PrefixMatcher(String patt) {
			this.patt = patt;
		}
		public String isMatch(String source) {
			return source.startsWith(patt) ? NONAME : null;
		}
		public String toString(){return patt+"**";}
	}
	private static class PackageMatcher implements Matcher {
		final String patt;
		public PackageMatcher(String patt) {
			this.patt = patt;
		}
		public String isMatch(String source) {
			if (!source.startsWith(patt)) return null;
			return (source.indexOf('.', patt.length())==-1) ? NONAME : null;
		}
		public String toString(){return patt+"*";}
	}
	private static class UserMatcher implements Matcher{
		final int idx;
		final String patt0;
		final String patt1;

		public UserMatcher(String patt) {
			idx = patt.indexOf(MACRO_USER);
			patt0 = patt.substring(0, idx);
			patt1 = patt.substring(idx+MACRO_USER.length());
		}

		public String isMatch(String source) {
			int srcLen = source.length();
			int len0 = patt0.length();
			int len1 = patt1.length();

			if (srcLen > len0+len1
				&& source.startsWith(patt0) && source.endsWith(patt1)) {
				String uname = source.substring(len0, srcLen-len1);
				return uname;
			}
			return null;
		}
		public String toString(){return patt0+MACRO_USER+patt1;}
	}

	private static class UserDirMatcher implements Matcher {
		final int idx;
		final String patt0;
		final String patt1;

		public UserDirMatcher(String patt) {
			idx = patt.indexOf(MACRO_USER);
			patt0 = patt.substring(0, idx);
			patt1 = patt.substring(idx+MACRO_USER.length());
		}

		public String isMatch(String source) {
			if (source.startsWith(patt0)) {
				int idx2 = source.indexOf("/",idx);
				if (idx2 == -1) return null;
				String uname = source.substring(idx, idx2);
				String after = source.substring(idx2);
				if (after.startsWith(patt1)) {
					return uname;
				}
			}
			return null;
		}
		public String toString(){return patt0+MACRO_USER+patt1;}
	}

}
