/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.auth.openid;
import java.util.*;
import javax.servlet.http.*;
import org.kotemaru.auth.*;
import javax.naming.*;
import javax.naming.directory.*;
import javax.servlet.*;
import java.io.*;
import java.net.*;
import org.apache.commons.httpclient.methods.*;
import org.apache.commons.httpclient.*;
import org.openid4java.discovery.Identifier;
import org.openid4java.message.AuthRequest;
import org.openid4java.consumer.ConsumerException;

/**
OpenID用PAM。

Ex:
class=auth.openid.PAMImpl
loginUrl=http://localhost/login/login-OpenID.ssjs
verifyUrl=http://localhost/login/login-OpenID-verify.ssjs
*/

public class PAMImpl extends PAMBase {
	private static final String OPENID_CONSUMER = "openid.consumer";
	private final String verifyUrl;

	public PAMImpl(Map props) {
		super(props);
		verifyUrl = (String)props.get("verifyUrl");
	}

	public String getName() {
		return "openid";
	}

	public String getDescription() {
		return "This PAM depends on OpenID server.\n";
	}
	public IdProvider getIdProvider() {
		return null;
	}

	public AuthResult authForRequest(HttpServletRequest request) throws Exception {
		return new AuthResultImpl(this, AuthResult.MOOT, PAMFactory.getVisitUser());
	}


	public String prepare(HttpServletRequest request, Map props) 
			throws Exception
	{
		String openidUrl = (String) props.get("openid_url");
		OpenIDConsumer consumer = new OpenIDConsumer(verifyUrl);
		AuthRequest authReq = consumer.authRequest(openidUrl);
		if (authReq == null) return null;
		request.getSession().setAttribute(OPENID_CONSUMER, consumer);
		return authReq.getDestinationUrl(true);
	}

	public AuthResult authenticate(HttpServletRequest request, Map props) throws Exception {
		OpenIDConsumer consumer =
			(OpenIDConsumer)request.getSession().getAttribute(OPENID_CONSUMER);
		String[] ids = consumer.verifyResponse(request);

		if (ids != null) {
			RoleManager roleMan = PAMFactory.getRoleManager();
			UserManager userMan = PAMFactory.getUserManager();
			String idp = consumer.getAuthRequest().getDestinationUrl(false);
			IdProvider idProvider = new IdProviderImpl(idp);
			String uid = encodeId(ids[0]);
			String name = uid;
			if (userMan != null) {
				name = ids[1]; // nickname
				Map info = userMan.getUserInfo(name, uid);
				if (info == null) info = userMan.createUser(name, uid);
				name = (String) info.get(userMan.NAME);
			}

			User user = new UserImpl(this, roleMan, idProvider, name, uid);
			return new AuthResultImpl(this, AuthResult.PASS, user);
		} else {
			return new AuthResultImpl(this, AuthResult.MOOT, PAMFactory.getVisitUser());
		}
	}

	private String encodeId(String url) {
		try {
			url = url.replaceFirst("^https?://","");
			url = url.replaceFirst("/$","");
			url = url.replaceFirst("^[.]*","");
			url = url.replace('/','_');
			return URLEncoder.encode(url, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}
	}

}
