/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.auth.ldap;
import java.util.*;
import javax.servlet.http.*;
import org.kotemaru.auth.*;
import javax.naming.*;
import javax.naming.directory.*;
import javax.servlet.*;
import java.io.*;
import java.net.*;

/**
LDAP用RoleManager。

Ex:
class=auth.ldap.RoleManagerImpl
#LDAPがアクセスに認証を要求する場合。
java.naming.security.principal=uid=${user},ou=People,dc=exsample,dc=com
java.naming.security.credentials=${passwd}

※LDAPは標準のロールが無いので groupOfUniqueNames または posixGroup をサポートしておく。

# objectClass: groupOfUniqueNames の場合
role.contextName=ou=Roles,dc=exsample,dc=com
role.memberAttr=uniqueMember
role.nameAttr=cn

# objectClass: posixGroup の場合
role.contextName=ou=Group,dc=exsample,dc=com
role.memberAttr=memberUid
role.nameAttr=cn

*/

public class RoleManagerImpl implements RoleManager  {
	private final Map props;
	private final Hashtable enviroment = new Hashtable();
	private final Hashtable envAnonymous;

	private final String roleContextName;
	private final String roleMemberAttr;
	private final String roleNameAttr;

	public RoleManagerImpl(Map props) {
		this.props = props;
		roleContextName = (String)props.get("role.contextName");
		roleMemberAttr  = (String)props.get("role.memberAttr");
		roleNameAttr    = (String)props.get("role.nameAttr");
                                  
	    enviroment.put(Context.INITIAL_CONTEXT_FACTORY,
							"com.sun.jndi.ldap.LdapCtxFactory" );

		Iterator ite = props.entrySet().iterator();
		while(ite.hasNext()){
			Map.Entry ent = (Map.Entry) ite.next();
			String name = (String) ent.getKey();
			if (name.startsWith("java.naming.")) {
				enviroment.put(name, ent.getValue());
			}
		}

		envAnonymous = (Hashtable) enviroment.clone();
		envAnonymous.remove(Context.SECURITY_PRINCIPAL);
		envAnonymous.remove(Context.SECURITY_CREDENTIALS);
	}

	public int getRoleState(User user, String role) {
		return getRoles(user.getName()).contains(role) ? ALLOW : MOOT;
	}
	public Set getRoles(User user) {
		return getRoles(user.getName());
	}

	private HashSet getRoles(String uid) {
		HashSet roles = new HashSet();
		try {
			DirContext ctx = new InitialDirContext(envAnonymous);
			try {
				SearchControls cont = new SearchControls();
				cont.setSearchScope(SearchControls.SUBTREE_SCOPE);
				Attributes filter = new BasicAttributes(roleMemberAttr, uid);
				NamingEnumeration results = ctx.search(roleContextName, filter);
				while (results.hasMoreElements()) {
					SearchResult aResult = (SearchResult)results.nextElement();
					Attribute attr = aResult.getAttributes().get(roleNameAttr);
					if (attr != null) {
						roles.add(attr.get());
					}
				}
			} finally {
				ctx.close();
			}
		} catch(NamingException e) {
			throw new RuntimeException(e);
		}
		return roles;
	}

}
