/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.auth;
import java.util.Set;
import java.util.Map;

/**
 * ユーザのベース実装。
 */
public abstract class UserBase implements User {
	private static final String UNDEFINED_NAME = "-";
	private final PAM pam;
	private final RoleManager roleManager;
	private final IdProvider idProvider;
	private final String name;
	private final String id;
	private final Set roles;
	private final boolean isRegistered;

	/**
	 * コンストラクタ。
	 * @param pam 認証したPAMインスタンス
	 * @param name ユーザ名
	 */
	protected UserBase(PAM pam, RoleManager roleMan, IdProvider idp, String name, String id) {
		this.pam = pam;
		this.roleManager = roleMan;
		this.idProvider = idp;
		this.isRegistered = (name != null);
		this.name = (name != null) ? name : UNDEFINED_NAME;
		this.id = id;
		this.roles = (roleMan==null) ? null : roleMan.getRoles(this);
	}
	public UserBase(PAM pam, RoleManager roleMan, IdProvider idp, String name) {
		this(pam, roleMan, idp, name, name);
	}

	public PAM getPAM() {
		return pam;
	}
	public IdProvider getIdProvider() {
		return idProvider;
	}
	public RoleManager getRoleManager() {
		return roleManager;
	}

	public String getName() {
		return name;
	}
	public String getId() {
		return id;
	}

	public Map getAttributes() {
		Map info = getUserInfo();
		return (Map) info.get(UserManager.ATTRS);
	}
	public void setAttributes(Map attrs) {
		Map info = getUserInfo();
		info.put(UserManager.ATTRS, attrs);
		PAMFactory.getUserManager().putUserInfo(this, info);
	}
	private Map getUserInfo() {
		UserManager userMan = PAMFactory.getUserManager();
		if (userMan == null) {
			throw new RuntimeException("Not supported user attributes.");
		}
		Map info = userMan.getUserInfo(this);
		if (info == null) {
			throw new RuntimeException("Not found user.");
		}
		return info;
	}
	public boolean isRegistered() {
		return isRegistered;
	}

	public String toString() {
		return name;
	}

	/**
	 * ユーザ名のハッシュ値を返す。
	 */
	public int hashCode() {
		return name.hashCode();
	}
	/**
	 * ユーザ名が一致すれば一致。
	 */
	public boolean equals(Object o) {
		if (this == o) return true;
		if (!(o instanceof UserBase)) return false;
		if (name == null) {
			return ((UserBase)o).name == null;
		} else {
			return name.equals(((UserBase)o).name);
		}
	}

	/**
	 * visitorは常に無効。
	 */
	public boolean isVisitor() {
		return false;
	}

	public boolean isAdmin() {
		return hasRole(ADMIN);
	}
	public boolean isGuest() {
		return hasRole(GUEST);
	}
	public boolean hasRole(String role) {
		if (!isVisitor() && USERS.equals(role)) return true;
		//if (!ADMIN.equals(role) && getName().equals(role)) return true;
		if (roles != null) return roles.contains(role);
		return getRoleManager().getRoleState(this, role) == RoleManager.ALLOW;
	}

}




