/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.auth;
import javax.servlet.http.*;
import java.util.Map;
import org.kotemaru.util.LOG;
import java.lang.reflect.Constructor;
import javax.servlet.http.*;

/**
 * PAMのファクトリクラス。
 * <li>同時に１つのPAMインスタンスしか持てない。
 */
public class PAMFactory {
	public static final String AUTH_USER = "auth.user";

	private static final String CLASS = "class";
	private static final Class[] TYPES = {Map.class};
	private static PAM instance;
	private static RoleManager roleManager;
	private static UserManager userManager;
	private static User VISITOR;

	/**
	 * 初期化<ul>
	 * <li>PAM情報からPAMインスタンスを生成し利用可能な状態にする。
	 * </ul>
	 * @param pamInfo PAM情報	
	 * @param roleMans RoleManager情報	
	 */
	public static void init(PAM pam, RoleManager roleMan, UserManager userMan) throws Exception {
		instance = pam;
		roleManager = roleMan;
		userManager = userMan;
		VISITOR = new VisitUser(instance);

		LOG.info("PAM class for " + pam.getClass().getName());
		LOG.info("RoleManager class for " + roleMan.getClass().getName());
		LOG.info("UserManager class for " + userMan);
	}

	/**
	 * PAMインスタンスを返す。
	 * @return PAMインスタンス
	 */
	public static PAM getPAM() {
		return instance;
	}
	/**
	 * RoleManagerインスタンスを返す。
	 * @return RoleManagerインスタンス
	 */
	public static RoleManager getRoleManager() {
		return roleManager;
	}
	public static UserManager getUserManager() {
		return userManager;
	}

	public static String register(HttpServletRequest request, Map props)
			throws Exception
	{
		User user = (User) props.get("user");
		if (user == null || user.isVisitor()) return null;

		String name = (String) props.get("username");
		Map info = getUserManager().register(name, user.getId());
		if (info == null) return null;

		name = (String) info.get(UserManager.NAME);
		User newUser = new UserImpl(user.getPAM(), user.getRoleManager(),
									user.getIdProvider(), name, user.getId());
		request.getSession().setAttribute(AUTH_USER, newUser);
		return "/"; // TODO:
	}

	/**
	 * 認証の準備を行う。
	 * <li>認証の準備を行い次のステップとなるURLを返す。
	 * <li>複数回呼ばれる場合も有る。
	 * <li>実装は任意。SSOでは通常必要になる。
	 * @param request リクエスト情報
	 * @param props プロパティ
	 * @return 転送先URL。転送不要の場合はnull。
	 */
	public static String prepare(HttpServletRequest request, Map props)
		throws Exception
	{
		return getPAM().prepare(request, props);
	}

	/**
	 * ログイン処理を行う。
	 * @param request リクエスト情報
	 * @param props プロパティ
	 * @return 認証結果
	 */
	public static AuthResult login(HttpServletRequest request, Map props)
		throws Exception
	{
		AuthResult res = getPAM().authenticate(request, props);
		if (res.getUser() != null) {
			request.getSession().invalidate();
			request.getSession().setAttribute(AUTH_USER, res.getUser());
			LOG.info("Login "+res.getUser());
		}
		return res;
	}

	/**
	 * ログアウト処理を行う。
	 * <li>サーブレットセッションは無効になる。
	 * @param request リクエスト情報
	 * @return null
	 */
	public static User logout(HttpServletRequest request) throws Exception {
		LOG.info("Logout "+getUser(request));
		request.getSession().invalidate();
		request.getSession(true);
		return null;
		//User user =  new VisitUser(getPAM());
		//request.getSession().setAttribute(AUTH_USER, user);
		//return user;
	}


	/**
	 * リクエストからユーザを得る。
	 * <li>未ログイン状態ならリクエスト情報からログインする。
	 * <li>ログインできなければ {@link VisitUser} となる。
	 * @param request リクエスト情報
	 * @return ユーザ
	 */
	public static User getUser(HttpServletRequest request) 
		throws Exception
	{
		if (getPAM().isServletAuth()) {
			AuthResult res = getPAM().authForRequest(request);
			return res.getUser();
		} else {
			User user = (User) request.getSession().getAttribute(AUTH_USER);
			if (user == null || user.isVisitor()) {
				AuthResult res = getPAM().authForRequest(request);
				user = res.getUser();
				if (user != null && !user.isVisitor()) {
					request.getSession().invalidate();
					request.getSession().setAttribute(AUTH_USER, user);
					request.setAttribute(AUTH_USER, user);
					LOG.info("Login "+user);
				}
			}
			return user;
		}
	}

	/**
	 * 未ログインユーザを返す。
	 * @return {@link VisitUser}のインスタンス。
	 */
	public static User getVisitUser() {
		return VISITOR;
	}

	/**
	 * 認証済みユーザとしてユーザを返す。
	 * <li>内部利用のみ。スクリプトに対して機能を公開しないように。
	 * @return ユーザ。
	 */
	public static User getUser(String userName) {
		if (userManager != null) {
			Map info = userManager.getUserInfo(userName, null);
//TODO:GAEテスト
if (info == null) {
info = userManager.register(userName, userName);
}

			String uid = (String) info.get(UserManager.ID);
			return new UserImpl(getPAM(), getRoleManager(), null, userName, uid);
		} else {
			return new UserImpl(getPAM(), getRoleManager(), userName);
		}
	}
}
