/*******************************************************************************
 * Apache License, Version 2.0 (http://www.apache.org/licenses/LICENSE-2.0)
 * Copyright (c) 2011- kotemaru@kotemaru.org
 ******************************************************************************/
package org.kotemaru.auth;
import javax.servlet.http.*;
import javax.servlet.*;
import java.io.*;
import java.util.*;

/**
 * PAM(Pluggable Authentication Modules)のインターフェース。
 */
public interface PAM extends Serializable {
	/**
	 * PAMの名前を返す。
	 * @return PAMの名前
	 */
	public String getName();
	/**
	 * PAMの説明を返す。
	 * @return PAMの説明
	 */
	public String getDescription();
	/**
	 * ユーザを認証するプロバイダ情報を返す。
	 * <li>プロバイダが固定で無い場合はnullを返す。
	 * @return IdProvider
	 */
	public IdProvider getIdProvider();

	/**
	 * Servlet認証の場合のみtrueを返す。
	 * @return Servlet認証か否か。
	 */
	public boolean isServletAuth();

	/**
	 * リクエスト時の自動認証を行う。
	 * <li>ログインに必要な情報はリクエスト情報から取得。
	 * <li>セッション属性にユーザは保存しているのでクッキー等の処理は不要。
	 * <li>認証できない場合にはステータスを AuthResult.MOOT で返す。
	 * <li>Servlet認証以外での実装は通常不要。
	 * <li>認証失敗の場合には {@link VisitUser} を返す。
	 * @param request リクエスト情報
	 * @return 認証結果
	 */
	public AuthResult authForRequest(HttpServletRequest request) throws Exception;


	/**
	 * 認証の準備を行う。
	 * <li>認証の準備を行い次のステップとなるURLを返す。
	 * <li>複数回呼ばれる場合も有る。
	 * <li>実装は任意。SSOでは通常必要になる。
	 * <li>リクエストパラメータはプロトコルで決っていない限り使わない。
	 * @param request リクエスト情報
	 * @param props プロパティ
	 * @return 転送先URL。転送不要の場合はnull。
	 */
	public String prepare(HttpServletRequest request, Map props) throws Exception;

	/**
	 * 認証を行う。
	 * <li>認証失敗の場合には {@link VisitUser} を返す。
	 * <li>Servlet認証以外では実装必須。
	 * <li>プロティの内容は実装依存。例: {user:"ユーザ名", pass:"パスワード"}
	 * <li>リクエストパラメータはプロトコルで決っていない限り使わない。
	 * @param request リクエスト情報
	 * @param props プロパティ
	 * @return 認証結果
	 */
	public AuthResult authenticate(HttpServletRequest request, Map props) throws Exception;


	/**
	 * ログイン画面への転送先URLを得る。
	 * <li>{@link PAMFactory#login(HttpServletRequest,String,String)} から呼ばれる。
	 * <li>転送先URLに {@link PAMFactory#login(HttpServletRequest,String,String)}を呼ぶHTMLを用意して置く必要がある。
	 * <li>returnUrl はリクエストパラメータ returnUrl として転送先URLに渡す。
	 *     扱いは転送先URLのHTMLに任せる。
	 * @param request リクエスト情報
	 * @param returnUrl ログイン後転送するURL
	 * @param charset エンコードする場合の文字コード
	 */
	public String getLoginURL(HttpServletRequest request, String returnUrl, String charset);

	/**
	 * ログイン画面への転送先URLを得る。
	 * <li>セッションの切断以外に特別な処理が有る場合に記述する。
	 * @param request リクエスト情報
	 * @param returnUrl ログイン後転送するURL
	 * @param charset エンコードする場合の文字コード
	 */
	public String getLogoutURL(HttpServletRequest request, String returnUrl, String charset);

}
