<?php
/*
Plugin Name: AmazonLink
Plugin URI: http://wppluginsj.sourceforge.jp/amazonlink/
Description: Amazon.co.jp の商品情報を同一の設定で表示します。ブログのデザイン変更に簡単に対応できます。
Author: よしとも
Version: 2.0.0 beta
Author URI: http://blog.yoshitomo.org/
*/


// This file's encode is UTF-8. And language is Japanese.


/*  Copyright 2007  Yoshitomo  (email : See http://park15.wakwak.com/%7Eyoshitomo/aboutMail.html)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
使い方：
１．プラグインディレクトリにアップロード
２．管理画面で有効化
３．投稿本文に次の形式でコードを入力。[amazon]ASIN[/amazon]

更新履歴
2008/01/14 v0.9.2
　Windows サーバーで外部ファイルの読み込みができないバグの修正。
2007/11/10 v0.9.1
　アパレル＆シューズの検索に対応。
2007/09/02 v0.9.0
　投稿画面での Amazon.co.jp 検索機能追加。
2007/6/21 v0.6.0
　カラーピッカー導入。WordPress 付属のライブラリを使用。
2009/8/14 v2.0.0 beta
　WordPress 2.6・2.7に対応
　テンプレートによる表示機能追加
　Product Advertising API に対応
　PHP の対応バージョンを5.0以上に変更（5.1以上推奨）
*/

include_once(ABSPATH.'wp-includes/class-snoopy.php');

if( class_exists('WpAmazonLink') )
{
	$ywal = new WpAmazonLink();
}

// メインクラス
class WpAmazonLink
{
	// 基本
	var $version = '2.0.0 beta';						// バージョンナンバー
	var $i18nDomain = 'wp_AmazonLink.yoshitomo.org';	// 国際化リソースドメイン
	var $optionName = 'yo_amazon_link';					// オプション名
	// 環境
	var $pluginDirName = '';							// プラグインディレクトリ名（あとから初期化）
	var $pluginDirPath = '';							// プラグインディレクトリパス（あとから初期化）
	var $pluginDirUrl = '';								// プラグインディレクトリURL（あとから初期化）
	// アソシエイト
	var $trackingId = 'wpal-22';						// 作者のトラッキングID
	// 商品情報キャッシュ
	var $table_name = 'yo_amazon_cache';				// キャッシュ用テーブル名
	var $cache_limit = 86400;							// キャッシュの有効期限　24*60*60（秒）
	// 排他制御
	var $lockDir = 'lockdir';							// ロックフラグファイルを置くディレクトリ
	var $lockFile = 'lock';								// ロックフラグとするファイルの名前
	var $lockTimeout = 2;								// ロックの有効時間（秒）
	var $lockRetry = 1;									// ロック再実行までの時間
	var $lockHandle = array();							// ロックハンドル
	// テンプレート
	var $defaultAltImages = array(						// 商品代替画像の初期設定
		'large' => array(								// 大
			'file' => 'noImageLarge.png',
			'height' => 300,
			'width' => 431,
		),
		'medium' => array(								// 中
			'file' => 'noImageMedium.png',
			'height' => 160,
			'width' => 111,
		),
		'small' => array(								// 小
			'file' => 'noImageSmall.png',
			'height' => 75,
			'width' => 59,
		),
	);
	var $_ifCode = array(								// IF に対応したコード
					'AUTHOR',
					'PUBLISHER',
					'CREATOR',
					'RELEASE_DATE',
					'REVIEW',
					);
	var $_defaultTemplate = '<a href="[DETAIL_PAGE_URL]">[ITEM_NAME]</a><br />[POWERD_BY]';
														// デフォルトのテンプレート
	var $_template = '';								// テンプレート
	// 内部情報
	var $caller = '';									// プラグインの呼び出し元
	var $_options;										// オプションクラスインスタンス
	var $Snoopy = null;									// HTTP通信ライブラリのインスタンス

	/**
	 * コンストラクタ
	 *
	 * @param void
	 * @return object クラスオブジェクト
	 */
	function WpAmazonLink()
	{
		global $wpdb;
		global $table_prefix;
		global $wp_version;

		// i18n
		load_plugin_textdomain($this->i18nDomain);

		// オプションの初期化
		$this->_loadOption($this->optionName);
		if ( $this->_getOption('trackingId') )
		{
			$this->trackingId = $this->_getOption('trackingId');
		}

		// プラグインパス
		$commonPluginDir = '';							// WP のルートからプラグインディレクトリまでの相対パス
		if ( !defined('WP_PLUGIN_DIR') )				// WP 2.6以前
		{
			if ( defined('PLUGINDIR') )
			{
				$commonPluginDir = PLUGINDIR;
			}
			else
			{
				$commonPluginDir = 'wp-content/plugins';
			}
			$commonPluginDir = '/' . $commonPluginDir . '/';
		}
		$this->pluginDirName = array_pop(explode(DIRECTORY_SEPARATOR, dirname(__FILE__)));
		if ( defined('WP_PLUGIN_DIR') )
		{
			$this->pluginDirPath = WP_PLUGIN_DIR . '/' . $this->pluginDirName;
		}
		else
		{
			$this->pluginDirPath = rtrim(ABSPATH, '/') . $commonPluginDir . $this->pluginDirName;
		}
		if ( defined('WP_PLUGIN_URL') )
		{
			$this->pluginDirUrl = WP_PLUGIN_URL . '/' . $this->pluginDirName;
		}
		else
		{
			$this->pluginDirUrl = rtrim(get_bloginfo('wpurl'), '/') . $commonPluginDir . $this->pluginDirName;
		}

		// ロックハンドル初期化
		$this->lockHandle = array(
								'dir' => $this->pluginDirPath . '/'. $this->lockDir . '/',
								'basename' => $this->lockFile,
								'timeout' => $this->lockTimeout,
								'trytime' => $this->lockRetry
							);

		// プラグインの呼び出し元
		// 記事投稿
		if ( stristr($_SERVER['REQUEST_URI'], 'wp-admin/post.php') or stristr($_SERVER['REQUEST_URI'], 'wp-admin/post-new.php') )
		{
			$this->caller = 'post-entry';
		}
		// ページ投稿
		if ( stristr($_SERVER['REQUEST_URI'], 'wp-admin/page.php') or stristr($_SERVER['REQUEST_URI'], 'wp-admin/page-new.php') )
		{
			$this->caller = 'post-page';
		}
		elseif ( stristr($_SERVER['REQUEST_URI'], 'wp-admin/admin.php') or stristr($_SERVER['REQUEST_URI'], 'wp-admin/options-general.php'))
		{
			// オプション
			if ( $_GET['page'] == 'AmazonLink.php' )
			{
				$this->caller = 'option';
			}
		}

		// 代替画像URLの初期設定
		if ( $this->_getOption('altImageUrl') != ''  and  $this->_getOption('altImageWidth') != ''  and  $this->_getOption('altImageHeight') != '' )
		{
			foreach ( array_keys($this->defaultAltImages) as $key )
			{
				$this->defaultAltImages[$key]['url'] = $this->_getOption('altImageUrl');
				$this->defaultAltImages[$key]['width'] = $this->_getOption('altImageWidth');
				$this->defaultAltImages[$key]['height'] = $this->_getOption('altImageHeight');
			}
		}
		else
		{
			foreach ( array_keys($this->defaultAltImages) as $key )
			{
				$this->defaultAltImages[$key]['url'] = $this->pluginDirUrl.'/image/'.$this->defaultAltImages[$key]['file'];
			}
		}

		// テンプレートの決定
		if ( $this->_getOption('template') )
		{
			$this->_template = $this->_getOption('template');
		}
		else
		{
			$this->_template = $this->_defaultTemplate;
		}
		// テンプレートに POWERD_BY が入っていない場合は強制的に追加する
		if ( strpos($this->_template, '[POWERD_BY]') === false )
		{
			$this->_template .= '[POWERD_BY]';
		}

		// キャッシュ用データベーステーブル名
		if ( $wpdb->prefix )
		{
			$this->table_name = $wpdb->prefix.$this->table_name;
		}
		else
		{
			$this->table_name = $table_prefix.$this->table_name;
		}

		$this->Snoopy = new Snoopy();

		// フックの登録
		//	add_action('activate_'.trim($_GET['plugin']), array($MakeTableTest, '_createTable'));
		register_activation_hook( __FILE__, array(&$this, '_createTable') );
		add_action('admin_menu', array(&$this, '_addPage'));
		add_action('admin_head', array(&$this, '_addAdminHead'));
		if ( function_exists('wp_enqueue_script') )
		{
			add_action('admin_print_scripts', array(&$this, '_admin_print_scripts'));
		}
		add_action('the_content', array(&$this, '_replaceCode'));
		add_action('the_content_rss', array(&$this, '_replaceCode'));
		// スペシャルサンクス
		// 投稿画面の検索フォームを2.7に対応させる際に、うえちょこさんのブログ記事を参考にさせていただきました。
		// http://labs.uechoco.com/blog/2009/01/wordpressamazonlink%e3%82%92wordpress-27%e3%81%ab%e5%af%be%e5%bf%9c%e3%81%95%e3%81%9b%e3%82%8b.html
		if ( version_compare($wp_version, '2.5', '>=') )	// 2.5以上
		{
			add_action('edit_form_advanced',array(&$this, '_dbxPost'));
		}
		else												// 2.5未満
		{
			add_action('dbx_post_advanced',array(&$this, '_dbxPost'));
		}

	}

	/**
	 * Product Advertising API のキャッシュ保存用テーブルを必要に応じて作成する
	 *
	 * @return void
	 */
	function _createTable()
	{
		global $wpdb;

		if ( $wpdb->get_var("SHOW TABLES LIKE '".$this->table_name."'") != $this->table_name )
		{
			$sql  = 'CREATE TABLE IF NOT EXISTS `'.$this->table_name.'` ('."\n";
			$sql .= ' `asin` VARCHAR(25) NOT NULL,'."\n";							// Amazon ASIN
			$sql .= ' `item_info` LONGTEXT NOT NULL,'."\n";							// Item detail data
			$sql .= ' `udate` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,'."\n";	// Data update time
			$sql .= ' `cdate` DATETIME NOT NULL,'."\n";								// Data created time
			$sql .= ' PRIMARY KEY (`asin`)'."\n";
			$sql .= ')'."\n";
			$sql .= 'ENGINE = myisam';
			if ( $wpdb->query($sql) )
			{
			}
			else
			{
				$wpdb->print_error();
			}
		}
	}

	/**
	 * 記事本文中のコードを個別商品リンクHTMLコードに置換する
	 *
	 * @param string $content 記事本文データ
	 * @return string 置換処理の終わった記事本文データ
	 */
	function _replaceCode($content)
	{
		global $post;
		global $wpdb;

		$link_code_pattern = '/\[amazon\]([a-zA-Z0-9]+)\[\/amazon\]/';
														// 広告に置換されるパターン
		$match_count = preg_match_all($link_code_pattern, $content, $matches, PREG_PATTERN_ORDER);
		/*
		$match_count の構造
		Array
		(
			[0] => Array
				(
					[0] => [amazon]4829118792[/amazon]
					[1] => [amazon]4829124199[/amazon]
					[2] => [amazon]482912458X[/amazon]
				)
			[1] => Array
				(
					[0] => 4829118792
					[1] => 4829124199
					[2] => 482912458X
				)
		)
		*/
		if ( $match_count > 0 )
		{
			switch ( $this->_getOption('displayType') )
			{
				case 'original':						// バージョン2.0からの、テンプレートによる表示
					// データ取得 SQL
					$select_sql  = 'SELECT * FROM `'.$this->table_name."` WHERE `asin` = '%s';";

					// データ追加 SQL
					$sql_insert  = 'INSERT `'.$this->table_name."` SET\n";
					$sql_insert .= " `asin`='%s',\n";
					$sql_insert .= " `item_info`='%s',\n";
					$sql_insert .= " `udate`=NOW(),\n";
					$sql_insert .= " `cdate`=NOW();\n";

					// データ更新 SQL
					$sql_update  = 'UPDATE `'.$this->table_name."` SET\n";
					$sql_update .= " `item_info`='%s',\n";
					$sql_update .= " `udate`=NOW()\n";
					$sql_update .= " WHERE `asin`='%s';\n";

					$data_status = array();				// キャッシュのステータス
					$item_id_list = array();			// データの取得が必要なASIN
					$item_info = array();				// 表示する商品のデータ
/*
					print "<!-- \n";
					print_r($matches[1]);
					print " -->\n";
*/
					// 記事に含まれるコードからASINを抽出して1つずつ処理
					foreach ( array_unique($matches[1]) as $match )
					{
						$asin = mysql_real_escape_string(trim($match));
														// ASIN

						$sql = sprintf($select_sql, $asin);
														// ASINに対応するキャッシュがあるか調べるためのSQL文
						$result = $wpdb->get_row($sql, ARRAY_A);
						if ( $result )					// キャッシュがある
						{
							// キャッシュ期限チェック
							if ( $this->cache_limit < (time() - strtotime($result['udate'])) )
							{

								$data_status[$asin] = 'update';
														// 更新が必要
								$item_id_list[] = $asin;
							}
							else
							{
								$data_status[$asin] = 'none';
														// 更新は不要
								$item_info[$asin] = unserialize($result['item_info']);
														// キャッシュからデータを取得
							}
						}
						else
						{
							$data_status[$asin] = 'insert';
														// 追加が必要
							$item_id_list[] = $asin;
						}
					}
					// データの取得が必要な商品がある場合・・・
					if ( $item_id_list )
					{
						// 商品情報を一括で取得
						$url = $this->pluginDirUrl.'/awsAjax.php?Operation=ItemLookup&AssociateTag='.$this->trackingId.'&ItemId='.implode(',', $item_id_list);
						$this->lockHandle = $this->doLock($this->lockHandle);
						sleep(1);						// 1秒間に Amazon に複数回リクエストしないためのスリープ
						$this->Snoopy->fetch($url);
						$this->doUnlock($this->lockHandle);
						// 配列に変換
						$a2sResult = simplexml_load_string($this->Snoopy->results);
						// 取得時にエラーがなければ・・・
						if ( $a2sResult->Items->Request->IsValid == 'True' )
						{
							// 商品ごとに処理を実行
							foreach ( $a2sResult->Items->Item as $item )
							{
								$asin = (string)$item->ASIN;
/*
								print "<!-- iiii\n";
								var_dump($item);
								print " -->\n";
*/
								// キャッシュ用データの作成
								$ary = array();
								$ary['[ASIN]'] = $asin;
								$ary['[TYPE]'] = (string)$item->ProductGroup;
								$ary['[SALES_RANK]'] = number_format($item->SalesRank);
								$ary['[ITEM_NAME]'] = (string)$item->ItemAttributes->Title;
								$ary['[PUBLISHER]'] = (string)$item->ItemAttributes->Publisher;
								$ary['[RELEASE_DATE]'] = (string)$item->ItemAttributes->ReleaseDate;
								if ( $ary['[RELEASE_DATE]'] == '' )
								{
									$ary['[RELEASE_DATE]'] = (string)$item->ItemAttributes->PublicationDate;
								}
								$ary['[DETAIL_PAGE_URL]'] = (string)$item->DetailPageURL;
								$ary['[AUTHOR]'] = (string)$item->ItemAttributes->Author;
								if ( $creators = $item->ItemAttributes->Creator )
								{
									foreach ( $creators as $creator )
									{
										$ary['creator'][] = array(
																'name' => (string)$creator,
																'role' => (string)$creator['Role'],
															);
									}
								}
								$ary['[AVERAGE_RATING]'] = (string)$item->CustomerReviews->AverageRating;
								if ( $reviews = $item->CustomerReviews->Review )
								{
									foreach ( $reviews as $review )
									{
										$ary['review'][] = array(
																'rating' => (string)$review->Rating,
																'summary' => (string)$review->Summary,
															);
									}
								}
								$imageSet = array();
								if ( $item->ImageSets->ImageSet )
								{
									foreach ( $item->ImageSets->ImageSet as $images )
									{
										if ( $images['Category'] == 'primary' )
										{
											$imageSet = $images;
											break;
										}
									}
								}
								if ( count($imageSet) == 0 )
								{
									$imageSet = $item;
								}
								$ary['[LARGE_IMAGE_URL]'] = (string)$imageSet->LargeImage->URL;
								$ary['[LARGE_IMAGE_HEIGHT]'] = (string)$imageSet->LargeImage->Height;
								$ary['[LARGE_IMAGE_WIDTH]'] = (string)$imageSet->LargeImage->Width;
								$ary['[MEDIUM_IMAGE_URL]'] = (string)$imageSet->MediumImage->URL;
								$ary['[MEDIUM_IMAGE_HEIGHT]'] = (string)$imageSet->MediumImage->Height;
								$ary['[MEDIUM_IMAGE_WIDTH]'] = (string)$imageSet->MediumImage->Width;
								$ary['[SMALL_IMAGE_URL]'] = (string)$imageSet->SmallImage->URL;
								$ary['[SMALL_IMAGE_HEIGHT]'] = (string)$imageSet->SmallImage->Height;
								$ary['[SMALL_IMAGE_WIDTH]'] = (string)$imageSet->SmallImage->Width;
								$ary['[POWERD_BY]'] = 'Powerd by <a href="http://blog.yoshitomo.org/archives/category/computer/internet/blog/wordpress/plugin/amazonlink/">AmazonLink</a> '.$this->version.'.';
								// IF 対応処理
								foreach ( $this->_ifCode as $code )
								{
									if ( !$ary[$code] )
									{
										$ary['[IF_'.$code.'/]'] = '';
										$ary['[/IF_'.$code.']'] = '';
									}
								}
/*
								print "<!-- rrrr\n";
								var_dump($ary);
								print " -->\n";
*/
								$item_info[$asin] = $ary;
								// 追加・更新実行
								switch ( $data_status[$asin] )
								{
									case 'insert':		// 新規追加
										$sql = sprintf($sql_insert, $asin, mysql_real_escape_string(serialize($item_info[$asin])));
										break;
									case 'update':		// 更新
										$sql = sprintf($sql_update, mysql_real_escape_string(serialize($item_info[$asin])), $asin);
										break;
									case 'none':		// なにもしない
									default:
										$sql = '';
								}
								// 必要に応じてクエリー
								if ( $sql )
								{
									$wpdb->query($sql);
//									print "<!-- ssss $sql -->\n";
								}
							} // 商品ごとのループここまで
						}
						else
						{
//							print "<!-- ERROR -->\n";
						}
					} // データの取得処理ここまで
					// 商品画像がない場合に代替画像を設定
					foreach ( array_keys($item_info) as $key )
					{
						if ( $item_info[$key]['[LARGE_IMAGE_URL]'] == '' )
						{
							$item_info[$key]['[LARGE_IMAGE_URL]'] = $this->defaultAltImages['large']['url'];
							$item_info[$key]['[LARGE_IMAGE_HEIGHT]'] = $this->defaultAltImages['large']['height'];
							$item_info[$key]['[LARGE_IMAGE_WIDTH]'] = $this->defaultAltImages['large']['width'];
						}
						if ( $item_info[$key]['[MEDIUM_IMAGE_URL]'] == '' )
						{
							$item_info[$key]['[MEDIUM_IMAGE_URL]'] = $this->defaultAltImages['medium']['url'];
							$item_info[$key]['[MEDIUM_IMAGE_HEIGHT]'] = $this->defaultAltImages['medium']['height'];
							$item_info[$key]['[MEDIUM_IMAGE_WIDTH]'] = $this->defaultAltImages['medium']['width'];
						}
						if ( $item_info[$key]['[SMALL_IMAGE_URL]'] == '' )
						{
							$item_info[$key]['[SMALL_IMAGE_URL]'] = $this->defaultAltImages['large']['url'];
							$item_info[$key]['[SMALL_IMAGE_HEIGHT]'] = $this->defaultAltImages['large']['height'];
							$item_info[$key]['[SMALL_IMAGE_WIDTH]'] = $this->defaultAltImages['large']['width'];
						}
					}

					// 記事中のコードを置換
					foreach ( $item_info as $asin => $detail )
					{
/*
						print "<!-- ddd\n";
						print_r($detail);
						print " -->\n";
*/
						// テンプレート元データの取得
						$item_code = $this->_template;
						// テンプレート中の置換処理
						if ( preg_match('|\[CREATOR/\](.+?)\[/CREATOR\]|s', $this->_template, $template_matches) )
						{
							if ( $detail['creator'] )
							{
								for ( $i=0; $i<count($detail['creator']); $i++ )
								{
									$code = $template_matches[1];
									$code = str_replace('[CREATOR_NAME]', $detail['creator'][$i]['name'], $code);
									$code = str_replace('[CREATOR_ROLE]', $detail['creator'][$i]['role'], $code);
									$detail['[CREATOR]'] .= $code;
								}
								unset($detail['creator']);
							}
						}
						if ( preg_match('|\[REVIEW/\](.+?)\[/REVIEW\]|s', $this->_template, $template_matches) )
						{
							if ( $detail['review'] )
							{
								for ( $i=0; $i<count($detail['review']); $i++ )
								{
									$code = $template_matches[1];
									$code = str_replace('[REVIEW_RATING]', $detail['review'][$i]['rating'], $code);
									$code = str_replace('[REVIEW_SUMMARY]', $detail['review'][$i]['summary'], $code);
									$detail['[REVIEW]'] .= $code;
								}
								unset($detail['review']);
							}
/*
							print "<!-- vvv\n";
							print_r($detail['review']);
							print " -->\n";
*/
						}
						foreach ( $this->_ifCode as $code )
						{
							if ( $detail['['.$code.']'] != '' )
							{
								$item_code = preg_replace('|\['.$code.'/\].+?\[/'.$code.'\]|s', '['.$code.']', $item_code);
							}
							else
							{
								$item_code = preg_replace('|\[IF_'.$code.'/\].+?\[/IF_'.$code.'\]|s', '', $item_code);
							}
						}
						$item_code = str_replace(array_keys($detail), array_values($detail), $item_code);
						// コードをテンプレートで置換
						$content = str_replace('[amazon]'.$asin.'[/amazon]', $item_code, $content);
					}
					return $content;					// 置換済み記事データ
					break;
					// テンプレート方式　ここまで
				case 'amazon':							// 従来の、Amazon.co.jp 個別商品リンクによる表示
				default:
					// パラメーターによる上書き
//					$this->_optionOverwrite($param);

					$html_code = '<iframe src="http://rcm-jp.amazon.co.jp/e/cm?o=9&p=8&l=as1&asins=$1';

					// トラッキングID
					$html_code .= '&t='.$this->trackingId;

					// 商品イメージ
					// IS2	大きいイメージを利用する　オン：1　オフ：なし
					// IS1	通常のイメージを利用する　オン：なし　オフ：1
					switch ( $this->_getOption('image') )
					{
						case 'normal':
							$html_code .= '&IS1=1';
							break;
						default:
							$html_code .= '&IS2=1';
					}

					// テキストカラー
					// fc1	テキストカラー：色コード
					if ( $this->_getOption('fc') )
					{
						$html_code .= '&fc1='.$this->_getOption('fc');
					}
					else
					{
						$html_code .= '&fc1=000000';
					}

					// リンクカラー
					// lc1	リンクカラー：色コード
					if ( $this->_getOption('lc') )
					{
						$html_code .= '&lc1='.$this->_getOption('lc');
					}
					else
					{
						$html_code .= '&lc1=0000FF';
					}

					// 背景色
					// bg1	背景の色：色コード
					if ( $this->_getOption('bc') )
					{
						$bc .= $this->_getOption('bc');
					}
					else
					{
						$bc .= 'FFFFFF';
					}
					$html_code .= '&bg1='.$bc;

					// リンクターゲット
					// lt1	新しいウィンドウで開く　オン：_blank　オフ：_top
					switch ( $this->_getOption('target') )
					{
						case 'top':
							$html_code .= '&lt1=_top';
							break;
						default: // blank
							$html_code .= '&lt1=_blank';
					}

					// ボーダー
					// bc1	フレームボーダーを表示する　オン：000000　オフ：背景色色コード
					switch ( $this->_getOption('border') )
					{
						case 'none':
							$html_code .= '&bc1='.$bc;
							break;
						default: // display
							$html_code .= '&bc1=000000';
					}

					// 価格の表示
					// nou	価格表示オプション　すべての価格を表示する：なし　新品の価格のみを表示する：1　価格を非表示にする：なし
					// npa	価格表示オプション　すべての価格を表示する：なし　新品の価格のみを表示する：なし　価格を非表示にする：1
					switch ( $this->_getOption('plice') )
					{
						case 'new':
							$html_code .= '&nou=1';
							break;
						case 'none':
							$html_code .= '&npa=1';
							break;
					}

					$html_code .= '&f=ifr" style="width: 120px; height: 240px;" marginwidth="0" marginheight="0" frameborder="0" scrolling="no"></iframe>';


		//			$html_code = str_replace('%TRACKING_ID%', $trackingId, $html_code);
					$html_code = str_replace('&', '&amp;', $html_code);


					// ASINコードの置換
					$replacedContent = preg_replace($link_code_pattern, $html_code, $content);
					return $replacedContent;
					break;
					// 従来方式　ここまで
			}
		}
		else
		{
			return $content;
		}

	}


	/**
	 * 管理画面のヘッダ
	 */
	function _addAdminHead()
	{
?>
<link rel="stylesheet" type="text/css" href="<?php echo $this->pluginDirUrl; ?>/css/admin.css" />
<?php
		if ( !function_exists('wp_enqueue_script') )
		{
			switch ( $this->caller )
			{
				case 'post-entry':
?>
<script type="text/javascript" src="<?php echo $this->pluginDirUrl; ?>/js/prototype.js"></script>
<script type="text/javascript" src="<?php echo $this->pluginDirUrl; ?>/js/scriptaculous/scriptaculous.js?load=effects"></script>
<script type="text/javascript" src="<?php echo $this->pluginDirUrl; ?>/js/ObjTree.js"></script>
<script type="text/javascript" src="<?php echo $this->pluginDirUrl; ?>/js/jkl-dumper.js"></script>
<script type="text/javascript" src="<?php echo $this->pluginDirUrl; ?>/js/AmazonLink_post.js"></script>
<?php
					break;
				case 'option':
?>
<script type="text/javascript" src="<?php echo get_bloginfo('wpurl'); ?>/wp-includes/js/colorpicker.js"></script>
<script type="text/javascript" src="<?php echo $this->pluginDirUrl; ?>/js/AmazonLink_option.js"></script>
<?php
					break;
				default:
			}
		}
	}

	// 管理画面用スクリプトの登録をする
	function _admin_print_scripts()
	{
		switch ( $this->caller )
		{
			case 'post-entry':
				wp_enqueue_script('prototype');
				wp_enqueue_script('scriptaculous-effects');
				wp_enqueue_script('ObjTree', $this->pluginDirUrl.'/js/ObjTree.js', array('prototype'), '0.24');
				wp_enqueue_script('jkl-dumper', $this->pluginDirUrl.'/js/jkl-dumper.js', array('ObjTree'), false);
				wp_enqueue_script('AmazonLink-post', $this->pluginDirUrl.'/js/AmazonLink_post.js', array('prototype','ObjTree','jkl-dumper'), false);
				break;
			case 'option':
				wp_enqueue_script('colorpicker');
				wp_enqueue_script('AmazonLink-option', $this->pluginDirUrl.'/js/AmazonLink_option.js', array('colorpicker'), false);
				break;
			default:
		}
	}

	/**
	 * 管理画面にページを追加する
	 *
	 * @return void
	 */
	function _addPage()
	{
		// オプションページの追加
		if ( function_exists('add_options_page') )
		{
			add_options_page(__('AmazonLink オプション', $this->i18nDomain), __('AmazonLink', $this->i18nDomain), 8, basename(__FILE__), array(&$this, '_optionPage'));
		}
	}

	/**
	 * 管理画面メニューのオプションページ
	 *
	 * @return void
	 */

	function _optionPage()
	{
		switch ( $_POST['op'] )
		{
			case 'set_option':
				// 強制的にエスケープ処理が入るので、使用する直前にアンエスケープしてやる
				foreach ( $_POST as $web_post_name => $web_post_value )
				{
					$_POST[$web_post_name] = stripslashes($web_post_value);
				}

				$this->_setOption('trackingId', $_POST['trackingId']);
				$this->_setOption('accessKeyId', $_POST['accessKeyId']);
				$this->_setOption('secretAccessKey', $_POST['secretAccessKey']);
				$this->_setOption('displayType', $_POST['displayType']);
				$this->_setOption('altImageUrl', $_POST['altImageUrl']);
				$this->_setOption('altImageWidth', $_POST['altImageWidth']);
				$this->_setOption('altImageHeight', $_POST['altImageHeight']);
				$this->_setOption('template', $_POST['template']);
				$this->_setOption('target', $_POST['target']);
				$this->_setOption('border', $_POST['border']);
				$this->_setOption('image', $_POST['image']);
				$this->_setOption('plice', $_POST['plice']);
				$this->_setOption('lc', $_POST['lc']);
				$this->_setOption('fc', $_POST['fc']);
				$this->_setOption('bc', $_POST['bc']);
				if ( !($this->_getOption('accessKeyId')  and  $this->_getOption('secretAccessKey')) )
				{
					$this->_setOption('displayType', 'amazon');
				}
				$this->_saveOption($this->optionName);

				// テンプレートによるオリジナル方式を選択していた場合は、
				// 必要に応じてキャッシュ用のテーブルを作成する。
				if ( $this->_getOption('displayType') == 'original' )
				{
					$this->_createTable();
				}

				break;
			case 'reset_option':
				$this->_resetOption($this->optionName);
				break;
			default: // not submited
		}

		// 表示方式
		$select_displayType = array();
		switch ( $this->_getOption('displayType') )
		{
			case 'original':
				$select_displayType['original'] = ' checked="checked"';
				break;
			default: // amazon
				$select_displayType['amazon'] = ' checked="checked"';
		}

		// リンクターゲット
		$select_target = array();
		switch ( $this->_getOption('target') )
		{
			case 'top':
				$select_target['top'] = ' checked="checked"';
				break;
			default: // blank
				$select_target['blank'] = ' checked="checked"';
		}
		// ボーダー
		$select_border = array();
		switch ( $this->_getOption('border') )
		{
			case 'none':
				$select_border['none'] = ' checked="checked"';
			default: // show
				$select_border['show'] = ' checked="checked"';
		}

		// 商品イメージ
		$select_image = array();
		switch ( $this->_getOption('image') )
		{
			case 'normal':
				$select_image['normal'] = ' checked="checked"';
			default: // large
				$select_image['large'] = ' checked="checked"';
		}

		// 価格の表示
		$select_plice = array();
		switch ( $this->_getOption('plice') )
		{
			case 'new':
				$select_plice['new'] = ' checked="checked"';
				break;
			case 'none':
				$select_plice['none'] = ' checked="checked"';
				break;
			default: // all
				$select_plice['all'] = ' checked="checked"';
		}

		// 色
		if ( $this->_getOption('lc') == '' )
		{
			$this->_setOption('lc', '0000FF');
		}
		if ( $this->_getOption('bc') == '' )
		{
			$this->_setOption('bc', 'FFFFFF');
		}
		if ( $this->_getOption('fc') == '' )
		{
			$this->_setOption('fc', '000000');
		}


		// プラグインオプション画面のコード出力
?>
<div class="wrap" id="AmazonLink">
<h2><?php _e('設定', $this->i18nDomain); ?></h2>
<h3><?php _e('共通の設定', $this->i18nDomain); ?></h3>
<form action="<?php echo $_SERVER['REQUEST_URI']; ?>" method="post">
<fieldset>
<legend><?php _e('トラッキング ID', $this->i18nDomain); ?></legend>
<p><?php _e('Amazon アソシエイトのリンクで使用するトラッキング ID です。空欄の場合は作者のものが使用されます。', $this->i18nDomain); ?></p>
<dl>
	<dt><label for="trackingId"><?php _e('トラッキング ID', $this->i18nDomain); ?></label><dt>
	<dd><input type="text" name="trackingId" id="trackingId" value="<?php echo htmlspecialchars($this->_getOption('trackingId')); ?>" /></dd>
</dl>
</fieldset>

<fieldset>
<legend><?php _e('Product Advertising API アカウント', $this->i18nDomain); ?></legend>
<p>
<?php _e('Product Advertising API を利用するために必要なアカウント情報です。<br />この項目を設定しない場合はテンプレートによる表示を選択することができず、投稿画面での商品検索も利用することができません。', $this->i18nDomain); ?><br />
</p>
<p>
<?php _e('<strong>重要</strong>：他人のアカウントを利用することは、Amazon によって禁止されています。', $this->i18nDomain); ?>
</p>
<p><?php _e('参考：<a href="http://affiliate.amazon.co.jp/gp/associates/network/help/t126/a3/ref=amb_link_84042136_3?pf_rd_m=AN1VRQENFRJN5&pf_rd_s=center-1&pf_rd_r=&pf_rd_t=501&pf_rd_p=&pf_rd_i=assoc_help_t126_a1" target="_blank">Amazon.co.jp による参考訳：Product Advertising API アカウント</a>', $this->i18nDomain); ?></p>
<dl>
	<dt><label for="accessKeyId"><?php _e('アクセスキー ID', $this->i18nDomain); ?></label><dt>
	<dd><input type="text" name="accessKeyId" id="accessKeyId" value="<?php echo htmlspecialchars($this->_getOption('accessKeyId')); ?>" /></dd>
</dl>
<dl>
	<dt><label for="secretAccessKey"><?php _e('秘密キー', $this->i18nDomain); ?></label><dt>
	<dd><input type="text" name="secretAccessKey" id="secretAccessKey" value="<?php echo htmlspecialchars($this->_getOption('secretAccessKey')); ?>" /></dd>
</dl>
</fieldset>

<fieldset>
<legend><?php _e('表示方式', $this->i18nDomain); ?></legend>
<p><?php _e('広告を表示するときの方式を選択します。テンプレートによる表示では、自由にレイアウトすることが可能です。', $this->i18nDomain); ?></p>
<p><?php _e('<strong>重要</strong>：テンプレートによる表示を選択した場合は、<code><var>AmazonLinkプラグインディレクトリ</var>/lockdir</code> のパーミッションを606または666に変更してください。', $this->i18nDomain); ?></p>
<ul>
	<li><input type="radio" name="displayType" id="displayType_amazon" value="amazon"<?php echo $select_displayType['amazon']; ?> /><label for="displayType_amazon"><?php _e('Amazon.co.jpの個別商品リンク（デフォルト）', $this->i18nDomain); ?></label></li>
	<li><input type="radio" name="displayType" id="displayType_original" value="original"<?php echo $select_displayType['original']; ?> /><label for="displayType_original"><?php _e('テンプレートによる表示', $this->i18nDomain); ?></label></li>
</ul>
</fieldset>

<h3><?php _e('テンプレートによる表示をするときの設定', $this->i18nDomain); ?></h3>

<fieldset>
<legend><?php _e('代替画像', $this->i18nDomain); ?></legend>
<p><?php _e('商品写真がなかったときに表示される画像です。空欄の項目があると、プラグインに同梱されたものが使用されます。', $this->i18nDomain); ?></p>
<dl>
	<dt><label for="altImageUrl"><?php _e('アドレス', $this->i18nDomain); ?></label></dt>
	<dd><input type="text" name="altImageUrl" id="altImageUrl" class="url" value="<?php echo htmlspecialchars($this->_getOption('altImageUrl')); ?>" /></dd>
	<dt><label for="altImageWidth"><?php _e('横幅（ピクセル）', $this->i18nDomain); ?></label></dt>
	<dd><input type="text" name="altImageWidth" id="altImageWidth" class="imageSize" value="<?php echo htmlspecialchars($this->_getOption('altImageWidth')); ?>" /></dd>
	<dt><label for="altImageHeight"><?php _e('高さ（ピクセル）', $this->i18nDomain); ?></label></dt>
	<dd><input type="text" name="altImageHeight" id="altImageHeight" class="imageSize" value="<?php echo htmlspecialchars($this->_getOption('altImageHeight')); ?>" /></dd>
</dl>
</fieldset>

<fieldset>
<legend><?php _e('テンプレート', $this->i18nDomain); ?></legend>
<p><?php _e('表示するレイアウトを決定するテンプレートです。空欄にすると次のテンプレートが使用され、商品ページにリンクされた商品名が表示されます。', $this->i18nDomain); ?></p>
<p><code><?php echo htmlspecialchars($this->_defaultTemplate); ?></code></p>
<p><?php _e('<a href="'.$this->pluginDirUrl.'/html/code_ja.html" target="_blank" onclick="window.open(this.href, \'mywindow6\', \'width=800, height=600, menubar=no, toolbar=no, scrollbars=yes\'); return false;">コードリファレンス（ポップアップします）</a>', $this->i18nDomain); ?></p>
<textarea name="template" id="template" rows="10" cols="80"><?php echo htmlspecialchars($this->_getOption('template')); ?></textarea>
</fieldset>

<h3><?php _e('Amazon.co.jp の個別商品リンクで表示する時の設定', $this->i18nDomain); ?></h3>

<fieldset>
<legend><?php _e('リンクターゲット', $this->i18nDomain); ?></legend>
<p><?php _e('リンクをクリックしたときの開き先。', $this->i18nDomain); ?></p>
<ul>
	<li>
	<input type="radio" name="target" id="target_blank" value="blank"<?php echo $select_target['blank']; ?> /><label for="target_blank"><?php _e('新規ウィンドウ（デフォルト）', $this->i18nDomain); ?></label>
	</li>
	<li>
	<input type="radio" name="target" id="target_top" value="top"<?php echo $select_target['top']; ?> /><label for="target_top"><?php _e('同じウィンドウ', $this->i18nDomain); ?></label>
	</li>
</ul>
</fieldset>

<fieldset>
<legend><?php _e('ボーダー', $this->i18nDomain); ?></legend>
<p><?php _e('個別商品リンクにボーダー（枠線）をつけるかどうか。', $this->i18nDomain); ?></p>
<ul>
	<li>
		<input type="radio" name="border" id="border_show" value="show"<?php echo $select_border['show']; ?> />
		<label for="border_show"><?php _e('ボーダーあり（デフォルト）', $this->i18nDomain); ?></label>
	</li>
	<li>
		<input type="radio" name="border" id="border_none" value="none"<?php echo $select_border['none']; ?> />
		<label for="border_none"><?php _e('ボーダーなし', $this->i18nDomain); ?></label>
	</li>
</ul>
</fieldset>

<fieldset>
<legend><?php _e('商品イメージ', $this->i18nDomain); ?></legend>
<p><?php _e('表示する商品のイメージに大きなものを使用するかどうか。', $this->i18nDomain); ?></p>
<ul>
	<li>
		<input type="radio" name="image" id="image_large" value="large"<?php echo $select_image['large']; ?> />
		<label for="image_large"><?php _e('大きなイメージ（デフォルト）', $this->i18nDomain); ?></label>
	</li>
	<li>
		<input type="radio" name="image" id="image_normal" value="normal"<?php echo $select_image['normal']; ?> />
		<label for="image_normal"><?php _e('通常のサイズ', $this->i18nDomain); ?></label>
	</li>
</ul>
</fieldset>

<fieldset>
<legend><?php _e('価格の表示', $this->i18nDomain); ?></legend>
<p><?php _e('商品の価格表示方法です。', $this->i18nDomain); ?></p>
<ul>
	<li>
		<input type="radio" name="plice" id="plice_all" value="all"<?php echo $select_plice['all']; ?> />
		<label for="plice_all"><?php _e('すべての価格を表示する（デフォルト）', $this->i18nDomain); ?></label>
	</li>
	<li>
		<input type="radio" name="plice" id="plice_new" value="new"<?php echo $select_plice['new']; ?> />
		<label for="plice_new"><?php _e('新品の価格のみ表示', $this->i18nDomain); ?></label>
	</li>
	<li>
		<input type="radio" name="plice" id="plice_none" value="none"<?php echo $select_plice['none']; ?> />
		<label for="plice_none"><?php _e('価格は表示しない', $this->i18nDomain); ?></label>
	</li>
</ul>
</fieldset>

<fieldset>
<legend><?php _e('色', $this->i18nDomain); ?></legend>
<p><?php _e('個別商品リンク各部の色を、6桁の16進数表現で指定します。', $this->i18nDomain); ?></p>
<dl>
	<dt><label for="bc"><?php _e('背景色', $this->i18nDomain); ?></label></dt>
	<dd>#<input type="text" name="bc" id="bc" class="colorCode" value="<?php echo htmlspecialchars($this->_getOption('bc')); ?>" /><a href="#" onclick="yoColorPicker.select(document.forms[0].bc,'bcpick');return false;" name="bcpick" id="bcpick">カラーピッカー</a></dd>
	<dt><label for="fc"><?php _e('テキストカラー', $this->i18nDomain); ?></label></dt>
	<dd>#<input type="text" name="fc" id="fc" class="colorCode" value="<?php echo htmlspecialchars($this->_getOption('fc')); ?>" /><a href="#" onclick="yoColorPicker.select(document.forms[0].fc,'fcpick');return false;" name="fcpick" id="fcpick">カラーピッカー</a></dd>
	<dt><label for="lc"><?php _e('リンクカラー', $this->i18nDomain); ?></label></dt>
	<dd>#<input type="text" name="lc" id="lc" class="colorCode" value="<?php echo htmlspecialchars($this->_getOption('lc')); ?>" /><a href="#" onclick="yoColorPicker.select(document.forms[0].lc,'lcpick');return false;" name="lcpick" id="lcpick">カラーピッカー</a></dd>
</dl>
</fieldset>

<p class="submit">
<input type="submit" value="<?php _e('設定を更新する &raquo;', $this->i18nDomain); ?>" />
</p>
<input class="button" type="hidden" name="op" value="set_option" />
</form>
<form action="<?php echo $_SERVER['REQUEST_URI']; ?>" method="post">
<p class="submit">
<input class="button" id="deletepost" type="submit" value="<?php _e('設定をリセットする &raquo;', $this->i18nDomain); ?>" />
</p>
<input type="hidden" name="op" value="reset_option" />
</form>
</div>
<?php
	}

	// 記事作成画面のドッキングボックス
	/*
	 * @return none
	 */
	function _dbxPost()
	{
		global $post;
		global $wp_version;

		$categories = array(
						__('全商品', $this->i18nDomain) => 'Blended',
						__('本', $this->i18nDomain) => 'Books',
						__('洋書', $this->i18nDomain) => 'ForeignBooks',
						__('エレクトロニクス', $this->i18nDomain) => 'Electronics',
						__('ホーム＆キッチン', $this->i18nDomain) => 'Kitchen',
						__('ミュージック', $this->i18nDomain) => 'Music',
						__('ビデオ', $this->i18nDomain) => 'Video',
						__('ソフトウェア', $this->i18nDomain) => 'Software',
						__('ゲーム', $this->i18nDomain) => 'VideoGames',
						__('おもちゃ＆ホビー', $this->i18nDomain) => 'Toys',
						__('スポーツ＆アウトドア', $this->i18nDomain) => 'SportingGoods',
						__('ヘルス＆ビューティー', $this->i18nDomain) => 'HealthPersonalCare',
						__('時計', $this->i18nDomain) => 'Watches',
						__('ベビー＆マタニティー', $this->i18nDomain) => 'Baby',
						__('アパレル＆シューズ', $this->i18nDomain) => 'Apparel',
					);

		$box_title = 'Amazon.co.jp 商品検索';
		$box_id = 'yo_amazonLink_dbx';
		if ( version_compare($wp_version, '2.5', '>=') )	// 2.5以上
		{
?>
<div id="<?php echo $box_id; ?>" class="postbox <?php echo postbox_classes($box_id, 'post'); ?>">
<h3 class="hndle"><?php echo $box_title; ?></h3>
<div class="inside">
<?php
		}
		else												// 2.5未満
		{
?>
<div class="dbx-b-ox-wrapper">
<fieldset id="<?php echo $box_id; ?>" class="dbx-box">
<div class="dbx-h-andle-wrapper">
<h3 class="dbx-handle"><?php echo $box_title; ?></h3>
</div>
<div class="dbx-c-ontent-wrapper">
<div class="dbx-content">
<?php
		}
?>
<input type="hidden" name="yo_amazonLink_url" id="yo_amazonLink_url" value="<?php echo $this->pluginDirUrl; ?>" />
<input type="hidden" name="yo_amazonLink_totalPages" id="yo_amazonLink_totalPages" value="0" />
<input type="hidden" name="yo_amazonLink_currentPage" id="yo_amazonLink_currentPage" value="1" />
<input type="hidden" name="yo_amazonLink_lastType" id="yo_amazonLink_lastType" value="" />
<input type="hidden" name="yo_amazonLink_lastKeyword" id="yo_amazonLink_lastKeyword" value="" />
<select name="yo_amazonLink_type" id="yo_amazonLink_type">
<?php
foreach ( $categories as $key => $value )
{
	print "\t".'<option value="'.$value.'">'.$key."</option>\n";
}
?>
</select>
<input type="text" name="yo_amazonLink_keyword" id="yo_amazonLink_keyword" value="" title="検索するキーワードを入力します" />
<input type="button" name="yo_amazonLink_search" id="yo_amazonLink_search" class="button" value="検索" title="検索を実行します" />
<input type="button" name="yo_amazonLink_toPreviousPage" id="yo_amazonLink_toPreviousPage" class="button" value="前のページへ" />
<span id="yo_amazonLink_page">0/0</span>
<input type="button" name="yo_amazonLink_toNextPage" id="yo_amazonLink_toNextPage" class="button" value="次のページへ" />
<div id="yo_amazonLink_result"></div>
<?php
		if ( version_compare($wp_version, '2.5', '>=') )	// 2.5以上
		{
?>
</div>
</div>

<?php
		}
		else												// 2.5未満
		{
?>
</div>
</div>
</fieldset>
</div>
<?php
		}
	}

// ========== 共通オプション処理関数群　ここから ==========
/**
 * [前提]
 * 　オプションを保持する変数の名前は _options とする。
 * 　外部からのアクセスは拒否（private）とする。そのため、関数名は _ を頭につける（PHPの習慣）。
 * 　ほかのプラグインでも使用できるように、データ加工を目的とした処理は入れない。
 */

	/**
	 * オプションデータをデータベースから読み込む
	 *
	 * @param string $name オプション名
	 * @return int 初期化後のオプションの数
	 *
	 */
	function _loadOption($name)
	{
		$this->_options = get_option($name);
		if ( !is_array($this->_options) )
		{
			$this->_options = array();
		}
		$option_num = count($this->_options);
		if ( $option_num > 0 )
		{
			$this->_setOption('is_loaded', true);
			return $option_num;
		}
		else
		{
			$this->_setOption('is_loaded', false);
			return false;
		}
	}

	/**
	 * オプションデータをデータベースに保存する
	 *
	 * @param string $name オプション名
	 * @param bool $autoload  自動読み込みの設定
	 * @return int 初期化後のオプションの数
	 *
	 */
	function _saveOption($name, $autoload  = 'yes')
	{
		if ( get_option($name) )
		{
			update_option($name, $this->_options);
		}
		else
		{
			add_option($name, $this->_options, $autoload );
		}
	}

	/**
	 * 自分自身のプラグインオプション値を得る
	 *
	 * @param string $name オプション名
	 * @return mixed オプション値
	 *
	 */
	function _getOption($name)
	{
		return $this->_options[$name];
	}

	/**
	 * 自分自身のすべてのプラグインオプション値を得る
	 *
	 * @return mixed オプション値
	 *
	 */
	function _getAllOptions()
	{
		return $this->_options;
	}

	/**
	 * 自分自身のプラグインオプション値を設定する
	 *
	 * @param string $name オプション名
	 * @param mixed $value 値
	 * @return void
	 *
	 */
	function _setOption($name, $value)
	{
		$this->_options[$name] = $value;
	}

	/**
	 * 自分自身のプラグインオプション値をクリアする
	 *
	 * @return void
	 *
	 */
	function _resetOption($name)
	{
		delete_option($name);
		$this->_options = array();
	}


	/**
	 * パラメーターによる上書き
	 *
	 * @param string $param パラメーター
	 * @return void
	 *
	 */
	function _optionOverwrite($param = '')
	{
		// パラメーターオプションの初期化
		if ( $param != '' )
		{
			foreach ( explode('&', $param) as $option )
			{
				list($name, $value) = explode('=', $option);
				$this->_setOption($name, $value);
			}
		}
	}

// ========== 共通オプション処理関数群　ここまで ==========

// ========== 排他制御処理関数群　ここから ==========
	/**
	 * @see	http://d.hatena.ne.jp/IwamotoTakashi/20080602/p1
	 */
	function doLock($lfh)
	{
		$lfh['path'] = $lfh['dir'] . $lfh['basename'];
		$time = time();

		for ( $i = 0; $i < $lfh['trytime']; $i++, sleep(1) )
		{
			if ( rename($lfh['path'], $lfh['current']=$lfh['path'].$time) )
			{
				/*
				if ( $fh = fopen($lfh['current'], 'a') )
				{
					fwrite($fh, date('Y/m/d H:i:s', $time) . "\t");
					fclose($fh);
				}
				*/
				return $lfh;
			}
		}
		$filelist = scandir($lfh['dir']);
		foreach ( $filelist as $file )
		{
			if ( preg_match('/^'.$lfh['basename'].'(\d+)/', $file, $matches) )
			{
				if ( $time - $matches[1] > $lfh['timeout']
					and rename($lfh['dir'].$matches[0], $lfh['current']=$lfh['path'].$time )
				)
				{
					return $lfh;
				}
				break;
			}
		}
		return false;
	}

	function doUnlock($lfh)
	{
		/*
		if ( $fh = fopen($lfh['current'], 'a') )
		{
			fwrite($fh, date('Y/m/d H:i:s') . "\n");
			fclose($fh);
		}
		*/
		rename($lfh['current'], $lfh['path']);
	}
// ========== 排他制御処理関数群　ここまで ==========


} // class AmazonLink End

?>