<?php
/*
Plugin Name: Lightweight Google Maps
Plugin URI: http://wppluginsj.sourceforge.jp/lightweight-google-maps/
Description: Show google maps on your post and/or pages. Map and marker locations are read from "Lat_Long" custom fields (does not use Geo plugin.) At a static page, you can refine location markers by category or location name/address.
Version: 1.20
Author: IKEDA Yuriko
Author URI: http://www.yuriko.net/cat/wordpress
*/

/*  Copyright (c) 2007-2008 yuriko

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

define('LWGM_API_FILENAME', 'locations.php');
define('LWGM_FIXED_MAP_ID', 'google_maps');
define('LWGM_EACH_MAP_CLASS', 'each_map');
define('LWGM_FIXED_MAP_HEIGHT', 480);
define('LWGM_EACH_MAP_WIDTH', 300);
define('LWGM_EACH_MAP_HEIGHT', 150);
define('LWGM_MOBILE_MAP_WIDTH', 128);
define('LWGM_MOBILE_MAP_HEIGHT', 80);
define('LWGM_INFO_WINDOW_WIDTH', 200);
define('LWGM_ZOOM_OF_RECENT_LOCATIONS', 8);
define('LWGM_ZOOM_OF_ADDRESSED_MAP', 13);
define('LWGM_ZOOM_OF_EACH_MAP', 14);

$Lwgm = Lightweight_Google_Maps::factory();

/* ==================================================
 *   Lightweight_Google_Maps Class
   ================================================== */

class Lightweight_Google_Maps {
	var $plugin_path;

// ==================================================
//static
function factory() {
	if (function_exists('is_ktai') && is_ktai()) {
		if (function_exists('ks_is_flat_rate') && ks_is_flat_rate()) {
			return new LWGM_Mobile();
		}
	} elseif (! function_exists('is_mobile') || ! is_mobile()) {
		return new LWGM_PC();
	}
}

// ==================================================
//public
function Lightweight_Google_Maps() {
	$this->plugin_path = preg_replace('#^.*' . PLUGINDIR . '#', PLUGINDIR, dirname(__FILE__));
	if (DIRECTORY_SEPARATOR == '\\') {
		$this->plugin_path = str_replace('\\', '/', $this->plugin_path);
	}
	if (! preg_match('#^' . PLUGINDIR . '#', $this->plugin_path)) {
		$this->plugin_path = 'wp-content/plugins/lightweight-google-maps';
	}
	load_plugin_textdomain('lw_googlemaps', $this->plugin_path);
}

// ==================================================
//protected 
function get_zoom_level($field_name, $default_zoom) {
	$zoom = intval(get_option($field_name));
	if ($zoom < 1 || $zoom > 19) {
		$zoom = $default_zoom;
	}
	return $zoom;
}

// ===== End of class ====================
}

/* ==================================================
 *   LWGM_Mobile class
   ================================================== */

class LWGM_Mobile extends Lightweight_Google_Maps {

function LWGM_Mobile() {
	$map_type = get_option('lw_each_map_type');
	if ($map_type != 'LWGM_LINK_TO_MAP' && function_exists('ks_is_image_inline')) {
		add_filter('the_content', array($this, 'each_map'), 91);
	}
	parent::Lightweight_Google_Maps();
}

// ==================================================
//public 
function each_map($content) {
	require_once dirname(__FILE__) . '/Lat_Long.php';
	$latlongs = Lat_Long::get_LatLon();
	if ($latlongs) {
		global $post;
		$maps = array();
		foreach ($latlongs as $i => $l) {
			if (ks_is_image_inline()) {
				$maps[] = '<img src="http://maps.google.com/staticmap?markers=' . $l['lat'] . ',' . $l['lon'] . ',red&amp;zoom=' . LWGM_ZOOM_OF_EACH_MAP . '&amp;size=' . LWGM_MOBILE_MAP_WIDTH . 'x' . LWGM_MOBILE_MAP_HEIGHT . '&amp;maptype=mobile&amp;key=' . get_option('googlemaps_api_key') . '" />';
			} else {
				$maps[] = '[<img localsrc="94" alt="' . __('IMAGE:', 'ktai_style') . '" /><a href="http://maps.google.com/staticmap?markers=' . $l['lat'] . ',' . $l['lon'] . ',red&amp;zoom=' . LWGM_ZOOM_OF_EACH_MAP . '&amp;size=' . LWGM_MOBILE_MAP_WIDTH . 'x' . LWGM_MOBILE_MAP_HEIGHT . '&amp;maptype=mobile&amp;key=' . get_option('googlemaps_api_key') . '">' . __('Map of this location', 'lw_googlemaps') . '</a>]';
			}
			$content .= '<div align="center">' . implode('<br />', $maps) . '</div>';
			ks_added_image();
		}
	}
	return $content;
} 
 
// ===== End of class ====================
}

/* ==================================================
 *   LWGM_PC class
   ================================================== */

class LWGM_PC extends Lightweight_Google_Maps {
	var $fixed_map;
	var $each_maps;

function LWGM_PC() {
	require_once dirname(__FILE__) . '/prefpane.php';
	$admin = new Lightweight_Google_Maps_Admin;
	add_action('admin_menu',  array($admin, 'add_admin_page'));
	add_action('wp_head',     array($this, 'output_style'));
	parent::Lightweight_Google_Maps();
}

// ==================================================
//public 
function output_style() {
	$page_id = get_option('lw_fixed_map_page_id');
	if ($page_id && is_page($page_id)) {
		$this->fixed_map = $page_id;
		add_filter('the_content', array($this, 'fixed_map_menu'));
		add_action('wp_footer',   array($this, 'fixed_map_script'));
		$map_width = intval(get_option('lw_fixed_map_width'));
		$map_height = intval(get_option('lw_fixed_map_height'));
		if (! $map_height) {
			$map_height = LWGM_FIXED_MAP_HEIGHT;
		}
		$element = '#' . LWGM_FIXED_MAP_ID;
	} elseif (get_option('lw_each_map_type') != 'LWGM_LINK_TO_MAP') {
		add_filter('the_content', array($this, 'each_map'));
		add_action('wp_footer',   array($this, 'each_map_script'));
		$map_width = intval(get_option('lw_each_map_width'));
		if (! $map_width) {
			$map_width = LWGM_EACH_MAP_WIDTH;
		}
		$map_height = intval(get_option('lw_each_map_height'));
		if (! $map_height) {
			$map_height = LWGM_EACH_MAP_HEIGHT;
		}
		$element = '.' . LWGM_EACH_MAP_CLASS;
	} else {
		return;
	}
?>
<style type="text/css" media="screen,tv,print,handheld">
<?php echo $element; ?> {
<?php if ($map_width) {echo "	width:{$map_width}px;\n";} ?>
	height:<?php echo $map_height; ?>px;
	clear:both;
	margin:1em auto;
	border:1px solid #999;
	line-height:1.25em;
	text-align:center;
}
<?php echo $element; ?> img {
	margin:0;
	padding:0;
	border:0 none;
}
<?php echo $element; ?> .infowindow {
	text-align:left;
	font-size:0.88em;
}
<?php echo $element; ?> p {
	margin:0;
	text-indent:0;
	text-align:left;
	font-size:0.75em;
}
</style>
<script src="http://maps.google.com/maps?file=api&amp;v=2&amp;key=<?php echo  get_option('googlemaps_api_key');?>" type="text/javascript" charset="utf-8"></script>
<?php
}

// ==================================================
//public 
function fixed_map_menu($content) {
	$menu = '<div id="gmap_menu">
<form onsubmit="refine_by_category(this.cat.value); return false;" id="cat_refine" action="#"><div>
<label>' . __('Refine by category:', 'lw_googlemaps') . wp_dropdown_categories('orderby=name&show_count=1&echo=0&selected=' . intval($_GET['cat'])) . '
</label>
<input type="submit" value="' . __('Show', 'lw_googlemaps') . '" />
</div></form>
<form onsubmit="move_to_place(this.place.value); return false;" id="geocoding" action="#"><div>
' . __('Move to an address or a landmark:', 'lw_googlemaps') . '<input type="text" size="32" id="place" /> <input type="submit" value="' . __('Show', 'lw_googlemaps') . '" />
</div></form>
</div>';
	return str_replace('<div id="gmap_menu"></div>', $menu, $content);
}

// ==================================================
//public 
function each_map($content) {
	require_once dirname(__FILE__) . '/Lat_Long.php';
	$latlongs = Lat_Long::get_LatLon();
	if (! $latlongs) {
		return $content;
	}
	$map_type = get_option('lw_each_map_type');
	$page_id  = get_option('lw_fixed_map_page_id');
	if ($page_id && is_numeric($page_id)) {
		$page_link = get_permalink($page_id);
		if (strpos($page_link, '?') === FALSE) {
			$page_link .= '?';
		} else {
			$page_link .= '&';
		}
	} else {
		$page_link = '';
	}
	if ($page_link && $map_type == 'LWGM_LINK_TO_MAP') {
		foreach ($latlongs as $l) {
			$content .= '<p class="map_link"><a href="' . "{$page_link}lat={$l['lat']}&amp;lon={$l['lon']}" . '">' . __('View this location on my large map &raquo;', 'lw_googlemaps') . "</a></p>\n";
		}
	} elseif ($map_type != 'LWGM_LINK_TO_MAP') {
		global $post;
		foreach ($latlongs as $i => $l) {
			$this->each_maps[$post->ID][$i + 1] = array($l['lat'], $l['lon'], ($page_link ? "{$page_link}lat={$l['lat']}&lon={$l['lon']}" : ''));
			$seq = count($latlongs) >= 2 ?  '_' . ($i + 1) : '';
			$content .= '<div class="each_map" id="map_' . $post->ID . $seq  . '"></div>' . "\n";
		}
	}
	return $content;
}

// ==================================================
//public
function fixed_map_script() {
	$addressed_zoom = $this->get_zoom_level('lw_addressed_map_zoom', LWGM_ZOOM_OF_ADDRESSED_MAP);
	$recent_zoom = $this->get_zoom_level('lw_recent_map_zoom', LWGM_ZOOM_OF_RECENT_LOCATIONS);
	$map_type = get_option('lw_fixed_map_type');
	if (! $map_type) {
		$map_type = 'G_HYBRID_MAP';
	}
	$api_url = $this->plugin_path;
	$url = get_bloginfo('wpurl') . "/$api_url/" . LWGM_API_FILENAME . '?format=xml&';
?>
<script type="text/javascript">
//<![CDATA[
function parse_loc(response) {
	var xmldoc  = GXml.parse(response);
	var lat     = xmldoc.documentElement.getElementsByTagName('lat');
	var lon     = xmldoc.documentElement.getElementsByTagName('lon');
	var title   = xmldoc.documentElement.getElementsByTagName('title');
	var link    = xmldoc.documentElement.getElementsByTagName('link');
	var date    = xmldoc.documentElement.getElementsByTagName('date');
	var excerpt = xmldoc.documentElement.getElementsByTagName('excerpt');
	if (lat.length < 1) {
		return null;
	}
	var locs = new Array();
	for (var i = 0 ; i < lat.length ; i++) {
		locs[i] = {
			'latlng' : new GLatLng(GXml.value(lat[i]), GXml.value(lon[i])),
			'title'  : GXml.value(title[i]),
			'link'   : GXml.value(link[i]),
			'date'   : GXml.value(date[i]),
			'excerpt': GXml.value(excerpt[i])
		};
	}
	return locs;
}
// --------------------
function create_marker(loc) {
	var marker = new GMarker(loc.latlng);
	var html = '<div class="infowindow"><a href="'+loc.link+'"><strong>'+loc.title+'</strong></a><br />'+loc.date+'</div><p>'+loc.excerpt+'</p>';
	GEvent.addListener(marker, 'click', function() {
		marker.openInfoWindowHtml(html, {'maxWidth':<?php echo LWGM_INFO_WINDOW_WIDTH; ?>});
	});
	return marker;
}
// --------------------
function put_markers(map, locs) {
	if (locs) {
		map.clearOverlays();
		for (var i = 0 ; i < locs.length ; i++) {
			var marker = create_marker(locs[i]);
			map.addOverlay(marker);
		}
	}
}
// --------------------
function recent_locations(num) {
	var query = 'recent=' + num;
	GDownloadUrl(baseurl + query, function(response) {
		var locs = parse_loc(response);
		put_markers(map, locs);
	});
}
// --------------------
function move_to_latlong(latlng) {
	map.setCenter(latlng, <?php echo "$addressed_zoom, $map_type"; ?>);
	var bounds = map.getBounds();
	var query = 'bounds=' + bounds.getSouthWest().lat() + ',' + bounds.getSouthWest().lng() + ',' + bounds.getNorthEast().lat() + ',' + bounds.getNorthEast().lng();
	GDownloadUrl(baseurl + query, function(response) {
		var locs = parse_loc(response);
		put_markers(map, locs);
	});
}
// --------------------
function move_to_place(place) {
	geocoder.getLatLng(place, function(latlng) {
		if (latlng) {
			move_to_latlong(latlng);
		} else {
			alert(place + '<?php _e(': Could not find the latitude/longitude of this place.', 'lw_googlemaps'); ?>');
		}
	});
}
// --------------------
function get_outline(locs) {
	var outline = new GLatLngBounds(locs[0].latlng, locs[0].latlng);
	for (var i = 0 ; i < locs.length ; i++) {
		outline.extend(locs[i].latlng);
	}
	return outline;
}
// --------------------
function refine_by_category(cat_id) {
	var query = 'category=' + cat_id;
	GDownloadUrl(baseurl + query, function(response) {
		var locs  = parse_loc(response);
		if (! locs) {
			map.setCenter(new GLatLng(38.0,137.5), 4, <?php echo $map_type; ?>);
			map.clearOverlays();
			return;	
		}
		var outline = get_outline(locs);
		map.setCenter(outline.getCenter(), map.getBoundsZoomLevel(outline), <?php echo $map_type; ?>);
		put_markers(map, locs);
	});
}
// --------------------
function main() {
	map = new GMap2(document.getElementById('<?php echo LWGM_FIXED_MAP_ID; ?>'));
	map.addControl(new GLargeMapControl());
	map.addControl(new GMapTypeControl());
	map.addControl(new GScaleControl());
	map.addControl(new GOverviewMapControl());
	geocoder = new GClientGeocoder();
<?php
	if (isset($_GET['lat']) && isset($_GET['lon'])) {
?>
	move_to_latlong(new GLatLng(<?php echo floatval($_GET['lat']) . ',' . floatval($_GET['lon']); ?>));
<?php
	} elseif (isset($_GET['place'])) {
?>
	move_to_place('<?php echo wp_specialchars($_GET['place']); ?>');
<?php
	} elseif (isset($_GET['cat'])) {
?>
	refine_by_category(<?php echo intval($_GET['cat']); ?>);
<?php
	} else {
		$last = $this->last_location();
		if ($last) {
			$num = intval(get_option('lw_num_recent_markers'));
			echo <<< E__O__T
	map.setCenter(new GLatLng({$last['lat']}, {$last['lon']}), $recent_zoom, $map_type);
	recent_locations($num);

E__O__T;
		} else {
			echo <<< E__O__T
	map.setCenter(new GLatLng(38.0,137.5), 4, $map_type);

E__O__T;
		}
	}
?>
}
// --------------------
if (GBrowserIsCompatible()) {
	var map;
	var geocoder;
	var baseurl = '<?php echo $url; ?>';
	onload = main;
	onunload = GUnload;
}
//]]>
</script>
<?php
}

// ==================================================
// private
function last_location() {
	require_once dirname(__FILE__) . '/Lat_Long.php';
	$loc = NULL;
	for ($page = 1 ; $page <= 100000 ; $page++) {
		$posts = new WP_Query("paged=$page");
		if (! $posts->have_posts() || $page > 100000) {
			break;
		}
		while ($posts->have_posts()) : $posts->the_post();
			$latlongs = Lat_Long::get_LatLon();
			if ($latlongs) {
				$loc = $latlongs[0];
				break 2;
			}
		endwhile;
	}
	return $loc;
}

// ==================================================
// public
function each_map_script() {
	if (! isset($this->each_maps) || count($this->each_maps) < 1) {
		return;
	}
	$page_id = get_option('lw_fixed_map_page_id');
	$map_type = get_option('lw_each_map_type');
	if (! $map_type) {
		$map_type = 'G_NORMAL_MAP';
	}
	$zoom = $this->get_zoom_level('lw_each_map_zoom', LWGM_ZOOM_OF_EACH_MAP);
?>
<script type="text/javascript" charset="utf-8">
	//<![CDATA[
	function main() {
		var markeropt = new Object();
		markeropt.title = "<?php echo ($page_id ? __('Click to view my large map.', 'lw_googlemaps') : ''); ?>";
<?php
	foreach ($this->each_maps as $id => $points) {
		foreach ($points as $c => $latlng) {
			$count   = count($points) >= 2 ? "_$c" : '';
			echo <<<E__O__T
		var map$id$count = new GMap2(document.getElementById('map_$id$count'));
		map$id$count.addControl(new GSmallZoomControl());
		map$id$count.setCenter(new GLatLng($latlng[0], $latlng[1]), $zoom, $map_type);
		var marker$id$count = new GMarker(map$id$count.getCenter(), markeropt);

E__O__T;
			if ($latlng[2]) {
				echo <<<E__O__T
		GEvent.addListener(marker$id$count, 'click', function() {location.href = '$latlng[2]'; });

E__O__T;
			}
			echo <<<E__O__T
		map$id$count.addOverlay(marker$id$count);

E__O__T;
		}
	}
?>
	}
	if (GBrowserIsCompatible()) {
		onload = main;
		onunload = GUnload;
	}
//]]>
</script>
<?php
}

// ===== End of class ====================
}
?>