<?php
/*
Plugin Name: Ktai Style
Plugin URI: http://wppluginsj.sourceforge.jp/ktai_style/
Description: Provides lightweight pages for mobile phones.
Author: IKEDA Yuriko
Version: 0.92
Author URI: http://www.yuriko.net/cat/wordpress
*/

/*  Copyright (c) 2007 yuriko

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

$Ktai_Style = new Ktai_Style;
$KS_Config  = new Ktai_Style_Config;
add_action('admin_menu',  array($KS_Config, 'add_page'));

/* ==================================================
 * @param	none
 * @return	boolean $is_mobile
 */
if (! function_exists('is_mobile')) :
function is_mobile() {
	global $Ktai_Style;
	return $Ktai_Style->is_mobile();
}
endif;

/* ==================================================
 *   Ktai_Style class
   ================================================== */

class Ktai_Style {
	private $plugin_dir;
	private $template_dir;
	public  $ktai;

/* ==================================================
 * @param	none
 * @return	object  $this
 */
public function __construct() {
	require_once dirname(__FILE__) . '/services.php';
	$this->set_plugin_dir();
	load_plugin_textdomain('ktai_style', $this->plugin_dir);
	$this->ktai = Ktai_Services::factory(TRUE);
	if (! $this->ktai) {
		return;
	}

	add_action('template_redirect', array($this, 'output'), 9);
	if (isset($_GET['ks'])) {
		$_GET['s'] = mb_convert_encoding($_GET['ks'], get_bloginfo('charset'), $this->detect_encoding(@$_GET['Submit']));
	}
	return;
}

/* ==================================================
 * @param	string  $key
 * @return	boolean $charset
 */
public function get_param($key) {
	switch ($key) {
	case 'plugin_dir':
		return $this->plugin_dir;
	case 'template_dir':
		return $this->template_dir;
	default:
		if (! $this->ktai) {
			return Ktai_Services::get_param($key);
		}
		return $this->ktai->get_param($key);
	}
}

/* ==================================================
 * @param	none
 * @return	boolean $is_mobile
 */
public function is_mobile() {
	if (! $this->ktai) {
		return FALSE;
	}
	return $this->ktai->get_param('operator');
}

/* ==================================================
 * @param	none
 * @return	none
 */
private function set_plugin_dir() {
	$this->plugin_dir = preg_replace('#^.*' . PLUGINDIR . '#' , PLUGINDIR, dirname(__FILE__));
	if (DIRECTORY_SEPARATOR == '\\') {
		$this->plugin_dir = str_replace('\\', '/', $this->plugin_dir);
	}
	if (! preg_match('#^' . PLUGINDIR . '#', $this->plugin_dir)) {
		$this->plugin_dir = PLUGINDIR . '/ktai_style';
	}
}

/* ==================================================
 * @param	none
 * @return	none
 */
public function output() {
	if (is_robots() || is_feed() || is_trackback()) {
		return;
	}
	if (! $template = $this->load_template()) {
		$this->ks_die(__('Can\'t display pages. Bacause mobile phone templates are collapsed.', 'ktai_style'));
	}

	add_filter('comments_template', array($this, 'comments_template'), 90);
	add_filter('wp_list_categories', array($this, 'filter_tags'), 90);
	add_filter('wp_list_pages', array($this, 'filter_tags'), 90);
	require_once dirname(__FILE__) . '/tags.php';
	$charset      = $this->ktai->get_param('charset');
	$iana_charset = $this->ktai->get_param('iana_charset');
	$buffer = $this->ktai->get_param('preamble') . "\n";
	ob_start();
	include $template;
	$buffer .= ob_get_contents();
	ob_end_clean();
	$buffer = mb_convert_encoding($buffer, $charset, get_bloginfo('charset'));
	$buffer = $this->ktai->shrink_pre_split($buffer);
	$buffer = $this->ktai->replace_smiley($buffer);
	$buffer = $this->ktai->convert_char($buffer);
	$buffer = $this->ktai->split_page($buffer, $this->get_page_num());
	$buffer = $this->ktai->shrink_post_split($buffer);
	header ("Content-Type: text/html; charset=$iana_charset");
	echo $buffer;
	exit;
}

/* ==================================================
 * @param	string  $input
 * @return	string  $charset
 */
public function detect_encoding($input) {
	if (empty($input)) {
		$charset = 'auto';
	} else {
		$charset = mb_detect_encoding($input, array('JIS', 'UTF-8', 'SJIS-win', 'EUC-JP'));
		if (! $charset) {
			$charset = 'auto';
		}
	}
	return $charset;
}

/* ==================================================
 * @param	none
 * @return	string  $template
 * based on wp-includes/template-loader.php of WP 2.2.3
 */
private function load_template() {
	$this->template_dir = dirname(__FILE__) . '/templates/';
	if ( is_404() && $template = $this->query_template('404')) {
		return $template;
	} elseif (isset($_GET['menu'])) {
		if ($_GET['menu'] == 'months' && $template = $this->query_template('menu_months')) {
			return $template;
		} elseif ($_GET['menu'] == 'cats' && $template = $this->query_template('menu_cats')) {
			return $template;
		} elseif ($_GET['menu'] == 'links' && $template = $this->query_template('menu_links')) {
			return $template;
		} elseif ($_GET['menu'] == 'pages' && $template = $this->query_template('menu_pages')) {
			return $template;
		} elseif ($_GET['menu'] == 'comments' && $template = $this->query_template('menu_comments')) {
			return $template;
		}
	} elseif (is_search() && $template = $this->query_template('search')) {
		return $template;
	} elseif (is_home() && $template = $this->get_home_template()) {
		return $template;
	} elseif (is_attachment() && $template = $this->get_attachment_template()) {
		return $template;
	} elseif (is_single() && $template = $this->query_template('single')) {
		if (is_attachment()) {
			add_filter('the_content', 'prepend_attachment');
		}
		return $template;
	} elseif (is_page() && $template = $this->query_template('page')) {
		if (is_attachment()) {
			add_filter('the_content', 'prepend_attachment');
		}
		return $template;
	} elseif (is_category() && $template = $this->get_category_template()) {
		return $template;
	} elseif (is_author() && $template = $this->query_template('author')) {
		return $template;
	} elseif (is_date() && $template = $this->query_template('date')) {
		return $template;
	} elseif (is_archive() && $template = $this->query_template('archive')) {
		return $template;
	} elseif (is_paged() && $template = $this->query_template('paged')) {
		return $template;
	} elseif (file_exists($this->template_dir . 'index.php')) {
		if (is_attachment()) {
			add_filter('the_content', 'prepend_attachment');
		}
		return $this->template_dir . 'index.php';
	}
	return NULL;
}

/* ==================================================
 * @param	string  $type
 * @return	string  $template
 * based on get_query_template() at wp-includes/theme.php of WP 2.2.3
 */
private function query_template($type) {
	$template = '';
	if (file_exists($this->template_dir . "{$type}.php")) {
		$template = $this->template_dir . "{$type}.php";
	}
	return apply_filters("{$type}_template", $template);
}

/* ==================================================
 * @param	none
 * @return	string  $template
 * based on get_category_template() at wp-includes/theme.php of WP 2.2.3
 */
private function get_category_template() {
	$template = '';
	if (file_exists($this->template_dir . 'category-' . get_query_var('cat') . '.php')) {
		$template = $this->template_dir . 'category-' . get_query_var('cat') . '.php';
	} elseif (file_exists($this->template_dir . 'category.php')) {
		$template = $this->template_dir . 'category.php';
	}
	return apply_filters('category_template', $template);
}

/* ==================================================
 * @param	none
 * @return	string
 * based on get_home_template() at wp-includes/theme.php of WP 2.2.3
 */
private function get_home_template() {
	$template = '';
	if (file_exists($this->template_dir . 'home.php')) {
		$template = $this->template_dir . 'home.php';
	} elseif (file_exists($this->template_dir . 'index.php')) {
		$template = $this->template_dir . 'index.php';
	}
	return apply_filters('home_template', $template);
}

/* ==================================================
 * @param	none
 * @return	string  $template
 * based on get_attachment_template() at wp-includes/theme.php of WP 2.2.3
 */
private function get_attachment_template() {
	global $posts;
	$type = explode('/', $posts[0]->post_mime_type);
	if ( $template =  $this->query_template($type[0]) )
		return $template;
	elseif ( $template =  $this->query_template($type[1]) )
		return $template;
	elseif ( $template =  $this->query_template("$type[0]_$type[1]") )
		return $template;
	else
		return $this->query_template('attachment');
}

/* ==================================================
 * @param	none
 * @return	none
 */
public function get_header() {
	do_action('get_header');
	if (file_exists($this->template_dir . 'header.php')) {
		load_template($this->template_dir . 'header.php');
	}
	return;
}

/* ==================================================
 * @param	none
 * @return	none
 */
public function get_footer() {
	do_action('get_footer');
	if (file_exists($this->template_dir . 'footer.php')) {
		load_template($this->template_dir . 'footer.php');
	}
	return;
}

/* ==================================================
 * @param	string  $path
 * @return	string  $path
 */
function comments_template($path) {
	return str_replace(TEMPLATEPATH . '/', $this->template_dir, $path);
}

/* ==================================================
 * @param	none
 * @return	int    $page_num
 */
private function get_page_num() {
	$page_num = 0;
	if (isset($_GET['kp']) && is_numeric($_GET['kp'])) {
		$page_num = intval($_GET['kp']);
	} elseif (isset($_POST['kp']) && is_numeric($_POST['kp'])) {
		$page_num = intval($_POST['kp']);
	}
	return $page_num;
}

/* ==================================================
 * @param	string  $html
 * @return	string  $html
 */
public function filter_tags($html) {
	$html = Ktai_HTML_Filter::kses($html, $this->get_param('allowedtags'));
/*****
	$html = preg_replace('!<([^>]*[^ >])/>!', '<$1 />', $html); // hack for bad HTML of UTW
	// ----- save alt strings to prevent that wp_kses() destroys them -----
	preg_match_all('/alt=([\'"])([^\\1\\\\]|\\\\\\1)*?\\1/', $html, $alt); // "
	if ($alt) {
		$alt_pat  = array();
		$save     = array();
		foreach ($alt[0] as $a) {
			$alt_pat[]  = '/' . preg_quote($a, '/') . '/';
			$save[]     =  'alt="' . count($alt_pat) . '"';
		}
		$html = preg_replace($alt_pat, $save, $html, 1);
	}
	// ----- filter HTML by wp_kses() ----- 
	$html = wp_kses($html, $this->get_param('allowedtags'), array('http', 'https', 'mailto', 'tel', 'tel-av', 'vtel'));
	// ----- restore alt strings -----
	if ($alt) {
		$html = str_replace($save, $alt[0], $html);
	}
*****/
	return $html;
}

/* ==================================================
 * @param	string $message
 * @param	string $title
 * @return	none
 * based on wp_die() at wp-includes/functions() of WP 2.2.3
 */
public function ks_die($message, $title = '') {
	if ( function_exists( 'is_wp_error' ) && is_wp_error( $message ) ) {
		if ( empty($title) ) {
			$error_data = $message->get_error_data();
			if ( is_array($error_data) && isset($error_data['title']) )
				$title = $error_data['title'];
		}
		$errors = $message->get_error_messages();
		switch ( count($errors) ) :
		case 0 :
			$message = '';
			break;
		case 1 :
			$message = "<p>{$errors[0]}</p>";
			break;
		default :
			$message = "<ul>\n\t\t<li>" . join( "</li>\n\t\t<li>", $errors ) . "</li>\n\t</ul>";
			break;
		endswitch;
	} elseif ( is_string($message) ) {
		$message = "<p>$message</p>";
	}

	if ( !did_action('admin_head') ) :
		global $Ktai_Style;
		if ($Ktai_Style->ktai) {
			$charset      = $Ktai_Style->ktai->get_param('charset');
			$iana_charset = $Ktai_Style->ktai->get_param('iana_charset');
		} else {
			$charset      = Ktai_Services::get_param('charset');
			$iana_charset = Ktai_Services::get_param('iana_charset');
		}
		header ("Content-Type: text/html; charset=$iana_charset");

		if ( empty($title) )
			$title = mb_convert_encoding(__('WordPress &rsaquo; Error'), $charset, get_bloginfo('charset'));

		echo '<?xml version="1.0" encoding="' . $iana_charset .'" ?>' . "\n"; ?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML Basic 1.0//EN" "http://www.w3.org/TR/xhtml-basic/xhtml-basic10.dtd">
<html>
<head>
<meta http-equiv="Content-Type" content="text/html; charset=<?php echo $iana_charset; ?>" />
<title><?php echo $title ?></title>
</head>
<body>
<?php endif; ?>
<div><h1 id="logo"><img alt="WordPress" src="<?php echo get_bloginfo('wpurl'); ?>/wp-admin/images/wordpress-logo.png" /></h1></div>
<?php echo mb_convert_encoding($message, $charset, get_bloginfo('charset')); ?>
</body>
</html>
<?php
	die();
}

// ===== End of class ====================
}

/* ==================================================
 *   Ktai_Style_Config class
   ================================================== */

class Ktai_Style_Config {
	private $nonce = -1;

/* ==================================================
 * @param	string  $name
 * @return	mix     $value
 */
public function get_option($name, $return_default = FALSE) {
	if (! $return_default) {
		$value = get_option($name);
		if ($value) {
			return $value;
		}
		if ($name == 'ks_separate_comments') {
			return get_option('ks_separate_recent_comments');
		}
	}
	switch ($name) {
	case 'ks_date_color':
		return '#00aa33';
	case 'ks_author_color':
		return '#808080';
	case 'ks_comment_type_color':
		return '#808080';
	case 'ks_external_link_color':
		return '#660099';
	case 'ks_year_format':
		return 'Y-m-d';
	case 'ks_month_date_format':
		return 'n/j';
	case 'ks_time_format':
		return 'H:i';
	default:
		return NULL;
	}
}

/* ==================================================
 * @param	none
 * @return	none
 */
public function add_page() {
	add_options_page('Ktai Style Configuration', __('Mobile Output', 'ktai_style'), 'manage_options', basename(__FILE__), array($this, 'option_page'));
	if ( !function_exists('wp_nonce_field') ) {
		$this->nonce = -1;
	} else {
		$this->nonce = 'ktai-style-config';
	}
}

/* ==================================================
 * @param	none
 * @return	none
 */
public function option_page() {
	global $Ktai_Style;

	if (isset($_POST['update_option'])) {
		check_admin_referer($this->nonce);
		$this->upate_options();
	}
	if ($this->get_option('ks_title_only')) {
		$title_only[0] = '';
		$title_only[1] = ' checked="checked"';
	} else {
		$title_only[0] = ' checked="checked"';
		$title_only[1] = '';
	}
	if ($this->get_option('ks_separate_comments')) {
		$separate_comments[0] = '';
		$separate_comments[1] = ' checked="checked"';
	} else {
		$separate_comments[0] = ' checked="checked"';
		$separate_comments[1] = '';
	}
	if ($this->get_option('ks_external_link')) {
		$external_link[0] = '';
		$external_link[1] = ' checked="checked"';
	} else {
		$external_link[0] = ' checked="checked"';
		$external_link[1] = '';
	}
	$treat_as_internal   = $this->get_option('ks_treat_as_internal');
	$date_color          = $this->get_option('ks_date_color');
	$author_color        = $this->get_option('ks_author_color');
	$comment_type_color  = $this->get_option('ks_comment_type_color');
	$external_link_color = $this->get_option('ks_external_link_color');
	$year_format         = $this->get_option('ks_year_format');
	$month_date_format   = $this->get_option('ks_month_date_format');
	$time_format         = $this->get_option('ks_time_format');
?>
<div class="wrap">
<h2><?php _e('Ktai Style Options', 'ktai_style'); ?></h2>
<p><?php _e('Note: To revert values to default, just empty of the field.', 'ktai_style'); ?></p>
<form method="post">
<?php $this->make_nonce_field($this->nonce); ?>
<fieldset name="design" class="options">
<legend><?php _e('Design', 'ktai_style'); ?></legend>
<dl>
<dt><label for="title_only"><?php _e('Front Page, Search Results', 'ktai_style'); ?></label></dt>
  <dd>
    <label><input type="radio" name="title_only" id="title_only" value="1"<?php echo $title_only[0]; ?> /> <?php _e('Display the content of the first post.', 'ktai_style'); ?></label>
    <label><input type="radio" name="title_only" id="title_only" value="2"<?php echo $title_only[1]; ?> /> <?php _e('Title only for all posts.', 'ktai_style'); ?></label>
  </dd>
<dt><label for="separate_comments"><?php _e('Comments and Trackbacks/Pingbacks', 'ktai_style'); ?></label></dt> 
  <dd>
    <label><input type="radio" name="separate_comments" id="separate_comments" value="1"<?php echo $separate_comments[0]; ?> /> <?php _e('Mix them at each posts and/or recent comments.', 'ktai_style'); ?></label><br />
    <label><input type="radio" name="separate_comments" id="separate_comments" value="2"<?php echo $separate_comments[1]; ?> /> <?php _e('Separate comments and trackbacks/pingbacks.', 'ktai_style'); ?></label>
  </dd>
<dt><label for="external_link"><?php _e('Hyperlinks for external websites', 'ktai_style'); ?></label></dt> 
  <dd>
    <label><input type="radio" name="external_link" id="external_link" value="1"<?php echo $external_link[0]; ?> /> <?php _e('Delete', 'ktai_style'); ?></label><br />
    <label><input type="radio" name="external_link" id="external_link" value="2"<?php echo $external_link[1]; ?> /> <?php _e('Shift action by the phone operator: Delete (SoftBank) / Convert links to view with Full Browser, PC Site Browser (DoCoMo, au) / Leave (WILLCOM, Other)', 'ktai_style'); ?></label>
  </dd>
  <dd>
    <label for="treat_as_internal"><?php _e('Leave links for below websites (space separated)', 'ktai_style'); ?></label><br />
    <textarea name="treat_as_internal" id="treat_as_internal" cols="80%" rows="1"><?php echo $treat_as_internal; ?></textarea>
  </dd>
</dl>
</fieldset>
<fieldset name="colors" class="options">
<legend><?php _e('Text Color', 'ktai_style'); ?></legend>
<dl>
<dt><label for="date_color"><?php _e('Date/time after post titles', 'ktai_style'); ?></label></dt>
<dd><input type="text" value="<?php echo $date_color ?>" name="date_color" id="date_color" /> (<?php _e('#rrggbb as hex format', 'ktai_style'); ?>)</dd>
<dt><label for="author_color"><?php _e('Author, Date with a post content', 'ktai_style'); ?></label></dt>
<dd><input type="text" value="<?php echo $author_color ?>" name="author_color" id="author_color" /> (<?php _e('#rrggbb as hex format', 'ktai_style'); ?>)</dd>
<dt><label for="comment_type_color"><?php _e('Comment types (comment/trackback/pingback)', 'ktai_style'); ?></label></dt>
<dd><input type="text" value="<?php echo $comment_type_color ?>" name="comment_type_color" id="comment_type_color" /> (<?php _e('#rrggbb as hex format', 'ktai_style'); ?>)</dd>
<dt><label for="external_link_color"><?php _e('Hyperlinks for external websites (Available when \'Shift action by the phone operator\' is enabled.)', 'ktai_style'); ?></label></dt>
<dd><input type="text" value="<?php echo $external_link_color ?>" name="external_link_color" id="external_link_color" /> (<?php _e('#rrggbb as hex format', 'ktai_style'); ?>)</dd>
</dl>
<p><?php _e('Note: To configure background color/normal text color/hyperlink color/visited link color, edit &lt;body&gt; element at templates/header.php', 'ktai_style'); ?> <a href="templates.php?file=<?php echo str_replace('%2F', '/', $Ktai_Style->get_param('plugin_dir') . '/templates/header.php'); ?>"><?php _e('&raquo; Edit', 'ktai_style'); ?></a></p>
</fieldset>
<fieldset name="date_format" class="options">
<legend><?php _e('Date format of posts/comments', 'ktai_style'); ?></legend>
<dl>
<dt><label for="year_format"><?php _e('In case of displaying year, month, date (for last year and before)', 'ktai_style'); ?></label></dt>
<dd><input type="text" value="<?php echo $year_format ?>" name="year_format" id="year_format" /></dd>
<dt><label for="month_date_format"><?php _e('In case of displaying month, date only (for this year)', 'ktai_style'); ?></label></dt>
<dd><input type="text" value="<?php echo $month_date_format ?>" name="month_date_format" id="month_date_format" /></dd>
<dt><label for="time_format"><?php _e('Time', 'ktai_style'); ?></label></dt>
<dd><input type="text" value="<?php echo $time_format ?>" name="time_format" id="time_format" /></dd>
</dl>
<p><?php _e('Note: About date format, refer to <a href="http://codex.wordpress.org/Formatting_Date_and_Time">Codex</a> or <a href="http://www.php.net/date">date() function manual</a> of PHP.', 'ktai_style'); ?></p>
</fieldset>
<div class="submit">
<input type="hidden" name="action" value="update" />
<input type="submit" name="update_option" value="<?php _e('Update Options &raquo;'); ?> &raquo;" />
</div>
</form>
</div>
<?php
} 

/* ==================================================
 * @param	mix   $action
 * @return	none
 */
private function make_nonce_field($action = -1) {
	if ( !function_exists('wp_nonce_field') ) {
		return;
	} else {
		return wp_nonce_field($action);
	}
}

/* ==================================================
 * @param	none
 * @return	none
 */
private function upate_options() {
	if (! empty($_POST['title_only'])) {
		if (is_numeric($_POST['title_only'])) {
			update_option('ks_title_only', ($_POST['title_only'] == 2));
		}
	} else {
		delete_option('ks_title_only');
	}

	if (! empty($_POST['separate_comments'])) {
		if (is_numeric($_POST['separate_comments'])) {
			update_option('ks_separate_comments', ($_POST['separate_comments'] == 2));
		}
		if (get_option('ks_separate_recent_comments')) {
			delete_option('ks_separate_recent_comments');
		}
	} else {
		delete_option('ks_separate_comments');
		delete_option('ks_separate_recent_comments');
	}

	if (! empty($_POST['external_link'])) {
		if (is_numeric($_POST['external_link'])) {
			update_option('ks_external_link', ($_POST['external_link'] == 2));
		}
	} else {
		delete_option('ks_external_link');
	}

	if (! empty($_POST['treat_as_internal'])) {
		$sites = preg_split('/\\s+/', $_POST['treat_as_internal'], -1, PREG_SPLIT_NO_EMPTY);
		$sites = array_map('clean_url', $sites);
		$sites = preg_replace('#/$#', '', $sites);
		$sites_join = implode(' ', $sites);
		if (! preg_match('/^\\s*$/', $sites_join)) {
			update_option('ks_treat_as_internal', $sites_join);
		} else {
			delete_option('ks_treat_as_internal');
		}
	} else {
		delete_option('ks_treat_as_internal');
	}

	if (! empty($_POST['author_color'])) {
		if (preg_match('/^#[0-9a-fA-F]{6}$/', $_POST['author_color'])) {
			update_option('ks_author_color', $_POST['author_color']);
		}
	} else {
		delete_option('ks_author_color');
	}

	if (! empty($_POST['date_color'])) {
		if (preg_match('/^#[0-9a-fA-F]{6}$/', $_POST['date_color'])) {
			update_option('ks_date_color', $_POST['date_color']);
		}
	} else {
		delete_option('ks_date_color');
	}

	if (! empty($_POST['comment_type_color'])) {
		if (preg_match('/^#[0-9a-fA-F]{6}$/', $_POST['comment_type_color'])) {
			update_option('ks_comment_type_color', $_POST['comment_type_color']);
		}
	} else {
		delete_option('ks_comment_type_color');
	}

	if (! empty($_POST['external_link_color'])) {
		if (preg_match('/^#[0-9a-fA-F]{6}$/', $_POST['external_link_color'])) {
			update_option('ks_external_link_color', $_POST['external_link_color']);
		}
	} else {
		delete_option('ks_external_link_color');
	}

	if (! empty($_POST['year_format'])) {
		if (preg_match('/[Yyo]/', $_POST['year_format']) && preg_match('/[mnMF]/', $_POST['year_format']) && preg_match('/[djz]/', $_POST['year_format'])) {
			update_option('ks_year_format', $_POST['year_format']);
		}
	} else {
		delete_option('ks_year_format');
	}

	if (! empty($_POST['month_date_format'])) {
		if (preg_match('/[mnMF]/', $_POST['month_date_format']) && preg_match('/[djz]/', $_POST['month_date_format'])) {
			update_option('ks_month_date_format', $_POST['month_date_format']);
		}
	} else {
		delete_option('ks_month_date_format');
	}

	if (! empty($_POST['time_format'])) {
		if (preg_match('/[BgGhH]/', $_POST['time_format'])) {
			update_option('ks_time_format', $_POST['time_format']);
		}
	} else {
		delete_option('ks_time_format');
	}
?>
<div class="updated fade"><p><strong><?php _e('Options saved.'); ?></strong></p></div>
<?php
	return;
}

// ===== End of class ====================
}

?>
