<?php
/*
Plugin Name: Referer
Plugin URI: http://kitty0.org/blog/referer_plugin/
Description: サイドバーなどにブログへの参照元のリスト(検索語や参照元のサイトの URI など)を表示させるプラグインです。(日本語化: <a href="plasticdreams.org/">aka</a> from <a href="http://wppluginsj.sourceforge.jp/">WordPress Plugins J-Series</a>)
Author: Hamish Morgan
Version: 0.30
Author URI: http://kitty0.org/
Requirements: *Nix Server with PHP >= 4.3 MySQL >= 3.23
Tested On: Ubuntu PHP/5.0.5 MySQL/4.1.12, Ferdora Code 3 PHP/4.2.1 MySQL/3.23

Licence
==============================
	Copyright (C) 2006  Hamish Morgan (referer [at] kitty0 [dot] org)

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
	MA 02110-1301, USA.

Change Log 0.21 -> 0.3
==============================
	* Added rel="nofollow" to all generated links, so referer spammers dont get any pagerank for their trouble.
	* Added "Clear Logged Data" button to empty the referers data table.
	* Added GZip (.gz) compressed log file upload. (Not .tar.gz though.)
	* Added error checking for client-side form post failures.
	* Added disable on submit to options forms so it doesn't get borked by users epileptic-clicking.
	* Added ability to import logs from a server path.
    * Hitting [Enter] key in the Options page now runs "Update Options" not "Restore Defaults".
	* Max-upload in label for "Log File Upload" is now in human readable units.
	* Simplified the default display templates.
	* Fixed some HTML bugs in Options page.
    * Fixed bug where limit argument in topReferers() functions doesn't update %*-limit% tag.
	* Fixed multilingual character support in search words and phrases.
	* Fixed bug where uploading files larger than max upload file size sometimes just reloads the page with no error.

Change Log 0.20 -> 0.21
==============================
	 + Fixed bug where Host Blacklist didn't blocks urls
	 + Fixed/Enabled Searchengine Host Blacklisting
	 + Improved default output format templates.
	 + Added %snip% element to output format. This is is the normal element (e.g host, word) but cropped to fit in a column.
	 + Fixed problem where stats titles could say "Top 0 Referers" If the limit for a stat is 0 (ie no limit) then format element %*-limit% becomes an empty string.
	 + Fixed problem with submitting forms containing HTML entities (eg &nbsp;)
	 + Fixed typo in function referer_fetchPhrases()
	 + parseReferer now return an associative array of referer info.
	 + Add migration support
	 + Fixed bug where empty lines in searchengine regex caused an error

Change Log 0.10 –> 0.20
===============================
    * Added Search Phrases.
    * Added complete URL logging.
    * Changed database structure to use one table for all data.
    * Added search word blacklist
    * Improved default format strings for all data types.
    * Improved configuration options sub-panel layout.
    * Re-wrote almost everything to better accommodate new and upcoming features.
    * Removed all fatal errors.. so if something goes wrong it should at least still be possible to disable the plugin.

TODO - Bugs
==============================
	 * Fuzzy match domain names in URLs (e.g www.example.com and example.com should be grouped.)
	 * Form submition validation is secure but brain damaged
	 * Search matching seems to have problems with some URLs.
	 * do something about cache:blah search terms (wtf are they anyway?)
	 * Rewrite all string/regex functions to be mulltibyte safe.
	 *
TODO - Short term:
==============================
	 + Improved and easier searchengine matching
	 + Log local searches
	 + Add "Empty Database" button
	 + Add "Re-Proccess Database against Blacklists"
	 + Improve Options page layout with individual Help and Restore Default for each field.
	 + Install and Mirgate should happen automatically
	* Admin page to set rows visibly/hidden.

TODO - Long term
==============================
	 + Variable logFormat parsing
	 + Referers page template (like links template)
	 + Fuzzy host matching (eg: example.com, www.example.com)
	 + Localization
	 + Run as a Wordpress Widget
	 + Current HTTP_REFERER Referer search word highlighting.
	 + Detailed stats breakdown in admin section
		#  Referers Origin
			# Direct address / Bookmarks
			# Links from a NewsGroup
			# Links from an Internet Search Engine
			# Links from an external page (other web sites except search engines)
			# Unknown Origin
		# Referring search engines
			# Google
			# Yahoo
			# Msn
			# etc...
		# Referring sites
			# basicly host breakdown.
		# Search Keyphrases
		# Search Keywords
		# TLD's

*/


/*
 * Function wrappers
 *
 * To display stats about your top referers in your blog use one or more of the
 * function below.
 *
 * I have created these wrappers in superglobal namespace so I dont have to mess
 * about with scope resolution and users don't have to mess about with class
 * methods.
 *
 */

/*
 * string referer_topReferers([integer limit [, boolean echo]])
 *
 * This is a wrappers for a function in the Referer class. Use it in your
 * template to print a formated listst of top referers by catagory. This will
 * display all usefull information in a box and should be the only function you
 * have to include in your template. However if you wish to display only certain
 * parts of the data or format it yourself use the referer_top*() and
 * referer_fetch*() function below.
 *
 * They take the results of the all the other top*(function) functions (see
 * below) and format it using the "referer" XHTMl snippets in the plugin options
 * panel. It then outputs all the data into the page.
 *
 * Optional paramater $limit determins tha maximum number of results to return.
 * If it is not given then it will return the number set in the plugin options
 * panel. If limit is given as 0 (zero) the ALL results will be returned.
 *
 * If you set option paramater $echo as FALSE then instead of print the results
 * they will be returned so you can perform additional formatting or concatonate
 * them with other data.
 */
function referer_topReferers($limit = false, $echo = true)
{
	return $GLOBALS[REFERER_OBJECT_NAME]->topReferers($limit, $echo);
}


/*
 * string referer_topUrls([integer limit [, boolean echo]])
 * string referer_topHosts([integer limit [, boolean echo]])
 * string referer_topWords([integer limit [, boolean echo]])
 * string referer_topPhrases([integer limit [, boolean echo]])
 *
 * These are wrappers for function in the Referer class. Use them in you
 * template print a formated lists of top referers.
 *
 * They take the results of the fetch*() functions (see below) and format it
 * using the XHTMl snippets in the plugin options panel. Then they output all
 * the data into the page.
 *
 * Optional paramater $limit determins tha maximum number of results to return.
 * If it is not given then it will return the number set in the plugin options
 * panel. If limit is given as 0 (zero) the ALL results will be returned.
 *
 * If you set option paramater $echo as flase then instead of print the results
 * they will be returned so you can perform additional formatting or concatonate
 * them with other data.
 */
function referer_topUrls($limit = false, $echo = true)
{
	return $GLOBALS[REFERER_OBJECT_NAME]->topUrls($limit, $echo);
}
function referer_topHosts($limit = false, $echo = true)
{
	return $GLOBALS[REFERER_OBJECT_NAME]->topHosts($limit, $echo);
}
function referer_topWords($limit = false, $echo = true)
{
	return $GLOBALS[REFERER_OBJECT_NAME]->topWords($limit, $echo);
}
function referer_topPhrases($limit = false, $echo = true)
{
	return $GLOBALS[REFERER_OBJECT_NAME]->topPhrases($limit, $echo);
}

/*
 * array referer_fetchUrls([int limit])
 * array referer_fetchHosts([int limit])
 * array referer_fetchWords([int limit])
 * array referer_fetchPhrases([int limit])
 *
 * These are wrappers for function in the Referer class. Use them in you
 * template to get an unformatted array of results.
 *
 * They query the database for stats about top referers. They return an array of
 * urls, hosts, search words and searchphrases respectively. If there was a
 * problem or if there are no results they return NULL.
 *
 * Optional paramater $limit determins tha maximum number of results to return.
 * If it is not given then it will return the number set in the plugin options
 * panel. If limit is given as 0 (zero) the ALL results will be returned.
 */
function referer_fetchUrls($limit = false) {
	return $GLOBALS[REFERER_OBJECT_NAME]->db->fetchUrls($limit);
}
function referer_fetchHosts($limit = false) {
	return $GLOBALS[REFERER_OBJECT_NAME]->db->fetchHosts($limit);
}
function referer_fetchWords($limit = false) {
	return $GLOBALS[REFERER_OBJECT_NAME]->db->fetchWords($limit);
}
function referer_fetchPhrases($limit = false) {
	return $GLOBALS[REFERER_OBJECT_NAME]->db->fetchPhrases($limit);
}



/* ========================================================================
 * CONSTANTS -- If you wish to change these then you MUST first "Uninstall" the
 * plugin via the options panel, then change these values, then re-"Install".
 */

define('REFERER_VERSION', 0.30);

/*
 * Database table name will be this with the WordPress prefix (default is wp_)
 */
define('REFERER_TABLE_NAME', 'referers');

/*
 * All plugin options set in `wp_options` table will have this prefix
 * If you change this then version migration will be broken.
 */
define('REFERER_OPTIONS_PREFIX', 'referer_');

/*
 * If PHP has problems (i.e runs out of memory) while parsing a log file try
 * lowering this value... though higher is generally faster.
 */
define('REFERER_FILE_READ_BUFFER', 1048576); // 1MB

/*
 * Variable name of the object instance. If for whatever reason you wish to call
 * a method within the instance use ${REFERER_OBJECT_NAME}->method()
 * Yes I know this is complicated but it's a work around so when a callback
 * method is called staticly it can call an instance of itself.
 */
define('REFERER_OBJECT_NAME', 'reeeeFurHer');

/*
 * Show a bunch of debug information on the admin page.
 */
define('REFERER_DEBUG', false);

/*
 * ==========================================================================
 * CLASS RefererDB - Class for handling all Database communication, thus
 * removing all SQL from the rest of the code.
 * NOTE: Not sure if this should extend or wrap wpdb.
 */
class Referer_DataBase extends wpdb {

	// Tables
	var $referers;

	// Cached result values
	var $urls_cache;
	var $hosts_cache;
	var $words_cache;
	var $phrases_cache;

	/*
	 * constructor
	 */
	function Referer_DataBase()
	{
		parent::wpdb(DB_USER, DB_PASSWORD, DB_NAME, DB_HOST);

		$this->referers = $GLOBALS['table_prefix'] . REFERER_TABLE_NAME;

		$this->_flushCache();

	}

	function _flushCache()
	{
		$this->urls_cache = array();
		$this->hosts_cache = array();
		$this->words_cache = array();
		$this->phrases_cache = array();
	}

	function createReferersTable()
	{

		/*
		 * Yes I know Coding Standards says "If you don't use a variable more
		 * than once, don't create it. This includes queries." But that makes
		 * statements like this totally unreadable so it's worth 1 nanosecond of
		 * cpu time.
		 */
		$sql = 'CREATE TABLE `'.$this->referers.'` (
			`id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
			`count` INT(11) UNSIGNED NOT NULL DEFAULT 1,
 			`date` DATETIME  NOT NULL,
 			`url` VARCHAR (255)  NOT NULL,
			`visibility` ENUM("visible","hidden") NOT NULL DEFAULT "visible",
			`host` VARCHAR (71) ,
			`words` VARCHAR (127) ,
			`phrases` VARCHAR (127) ,
			PRIMARY KEY(`id`),
			INDEX `url`(`url`)
		)';
		return $this->query($sql);

	}

	/*
	 * Drop a given table (default is current referer table.)
	 */
	function dropTable($table = false)
	{
		if(false === $table)
			$table = '`'.$this->referers.'`';
		else
			$table = '`' . $GLOBALS['table_prefix'] . $table . '`';

		$sql = 'DROP TABLE '.$table.'';
		return $this->query($sql);
	}

	function emptyTable($table = false)
	{
		if(false === $table)
			$table = $this->referers;
		else
			$table = $GLOBALS['table_prefix'] . $table;

		$sql = 'TRUNCATE TABLE `' . $table . '`';
		return $this->query($sql);

	}

	/*
	 * bool addReferer(array ref [, int count])
	 *
	 * Updates the database with the new referer stats. If a row with the same
	 * url already exists then the count will be increments, otherwise it will
	 * add a new row. Paramater ref is an array of all the element of the update
	 * created by parseReferer() eg: ('url'=>blah, 'host'=>=foo, 'words'=>'bar',
	 * etc..) Optional paramater count says how many referers with this data
	 * should be added.
	 *
	 */
	function addReferer(&$ref, $count = 1)
	{
		$this->_flushCache();

		/*
		 * Idealy this should use a single query: INSERT... ON DUPLICATE KEY UPDATE
		 * But this does not seem to work in mysql ver 3.23, dispite what the
		 * documentation says, so uses 2 queries.
		 */

		$update = 'UPDATE `'.$this->referers.'` SET `count` = `count` + "'.$count.'", `date` = NOW()
			WHERE `url` = "'.strtolower($ref['url']).'"';

		$insert = 'INSERT INTO `'.$this->referers.'` SET
			`url` = "'.strtolower($ref['url']).'", '
			. '`host` = "'.strtolower($ref['host']).'", '
			. '`date` = NOW()'
			. ($count ? ', `count` =  "'.$count.'"' : '')
			. ($ref['words'] ? ', `words` =  "'.strtolower(implode(',', $ref['words'])).'"' : '')
			. ($ref['phrases'] ? ', `phrases` =  "'.strtolower(implode(',', $ref['phrases'])).'"' : '')
			. ($ref['visibility'] ? '' : ', `visibility` = "hidden"');

		if(1 === $this->query($update) or 1 === $this->query($insert))
			return true;
		else
			return false;

	}

	function hideReferer($id)
	{
		$sql = 'UPDATE  `'.$this->referers.'` SET `visibility` = "hidden" WHERE `id` = "'.$id.'"';
		if(1 !== $this->query($sql) )
			return false;
		else
			return true;
	}

	function showReferer($id)
	{
		$sql = 'UPDATE  `'.$this->referers.'` SET `visibility` = "visible" WHERE `id` = "'.$id.'"';
		if(1 !== $this->query($sql) )
			return false;
		else
			return true;
	}

	/*
	 * array fetchUrls([int limit])
	 *
	 * Queries the database for a list of urls from which you have had the most
	 * refers. If limit is given the it will return that many, otherwise it will
	 * you the configuration option.
	 *
	 * Returns an array of results on success or NULL on failure.
	 */
	function fetchUrls($limit = false)
	{
		if($limit and sizeof($this->urls_cache))
		{
			if(sizeof($this->urls_cache) == $limit)
				return $this->urls_cache;
			elseif(sizeof($this->urls_cache) > $limit)
				return array_slice($this->urls_cache, 0, $limit);
		}

		$sql = 'SELECT `url`, `count` FROM `'.$this->referers.'`
			WHERE `visibility`="visible"
			ORDER BY `count` DESC'.($limit ? " LIMIT $limit" : '');

		if($this->query($sql))
		{
			$this->urls_cache = $this->get_results(null, ARRAY_A);
			return $this->urls_cache;
		}
		else
			return false;
	}

	/*
	 * array fetchHosts([int limit])
	 *
	 * Queries the database for a list of hosts from which you have had the most
	 * refers. If limit is given the it will return that many, otherwise it will
	 * you the configuration option.
	 *
	 * Returns an array of results on success or NULL on failure.
	 */
	function fetchHosts($limit = false)
	{
		if($limit and sizeof($this->hosts_cache))
		{
			if(sizeof($this->hosts_cache) == $limit)
				return $this->hosts_cache;
			elseif(sizeof($this->hosts_cache) > $limit)
				return array_slice($this->hosts_cache, 0, $limit);
		}

		/*
		 * TODO: Add some way to fuzzy match the domain name so it doesnt return
		 * multiple of the same referer eg. ww.example.com, example.com,
		 * example.co.uk
		 */

		$sql = 'SELECT host, SUM(count) AS count FROM '.$this->referers.'
			WHERE visibility="visible" GROUP BY host
			ORDER BY count DESC'.($limit ? " LIMIT $limit" : '');;

		if($this->query($sql))
		{
			$this->hosts_cache = $this->get_results(null, ARRAY_A);
			return $this->hosts_cache;
		}
		else
			return false;
	}


	/*
	 * array fetchWords([int limit])
	 *
	 * Queries the database for a list of words from which you have had the most
	 * refers. If limit is given the it will return that many, otherwise it will
	 * you the configuration option.
	 *
	 * Returns an array of results on success or NULL on failure.
	 */
	function fetchWords($limit = false)
	{
		if($limit and sizeof($this->words_cache))
		{
			if(sizeof($this->words_cache) == $limit)
				return $this->words_cache;
			elseif(sizeof($this->words_cache) > $limit)
				return array_slice($this->words_cache, 0, $limit);
		}

		$sql = 'SELECT words, SUM(count) AS  count FROM '.$this->referers.'
			WHERE words != "" GROUP BY words';

		if(! $this->query($sql) )
			return false;

		// TODO2 : Optimise this method

		$results = $this->get_results(null, ARRAY_A);


		$words = array();
		foreach($results as $res)
		{
			foreach(explode(',', $res['words']) as $word)
			{
				if(isset($words[$word]))
					$words[$word] += $res['count'];
				else
					$words[$word] = $res['count'];
			}
		}
		arsort($words, SORT_NUMERIC);

		$out = array();
		while(($limit-- >0) and (list($key, $count) = each($words)) )
			$out[] = array ('word'=>$key, 'count'=>$count);

		$this->words_cache = $out;
		return $this->words_cache;
	}


	/*
	 * array fetchPhrases([int limit])
	 *
	 * Queries the database for a list of phrases from which you have had the
	 * most refers. If limit is given the it will return that many, otherwise it
	 * will you the configuration option.
	 *
	 * Returns an array of results on success or NULL on failure.
	 */
	function fetchPhrases($limit = false)
	{
		if($limit and sizeof($this->phrases_cache))
		{
			if(sizeof($this->phrases_cache) == $limit)
				return $this->phrases_cache;
			elseif(sizeof($this->phrases_cache) > $limit)
				return array_slice($this->phrases_cache, 0, $limit);
		}

		$sql = 'SELECT phrases, SUM(count) AS count FROM '.$this->referers.'
			WHERE phrases != "" GROUP BY phrases';

		if(! $this->query($sql) )
			return false;

		// TODO2 : Optimise this method

		$results = $this->get_results(null, ARRAY_N);

		while( list($key,list($data, $count)) = each($results))
		{
			foreach(explode(',', $data) as $str)
			{
				if(strlen(trim($str)))
				{
					if(isset($phrases[$str]))
						$phrases[$str] += $count;
					else
						$phrases[$str] = $count;
				}
			}
		}
		arsort($phrases, SORT_NUMERIC);

		$out = array();
		while(($limit-- >0) and (list($key, $count) = each($phrases)) )
			$out[] = array ('phrase'=>$key, 'count'=>$count);

		$this->phrases_cache = $out;
		return $this->phrases_cache;
	}

}

define('REFERER_ERR_FATAL', 1);
define('REFERER_ERR_ERROR', 2);
define('REFERER_ERR_WARNING', 4);
define('REFERER_ERR_INFO', 8);
define('REFERER_ERR_DEBUG', 16);

class Referer_ErrorHandler
{
	var $messages;
	var $message_id;
	var $type_index;

	function Referer_ErrorHandler()
	{
		$this->messages = array();
		$this->messages_id = 0;
		$this->type_index = array();
	}


	function _get_type_index($type)
	{
		if(isset($this->type_index[$type]))
		{
			return $this->type_index[$type];
		}

		$this->type_index[$type] = array();
		foreach($this->messages as $msg)
		{
			if($msg['type'] & $type)
			{
				$this->type_index[$type] =& $msg;
			}
		}
		return $this->type_index[$type];
	}

	function add($type, $text)
	{
		$this->message_id ++;

		$this->messages[$this->message_id] = array(
			'id' => $this->message_id,
			'type' => $type,
			'text' => $text,
			'date' => time(),
		);
		if(!isset($this->type_index[$type]))
		{
			$this->type_index[$type] = array();
		}
		foreach($this->type_index as $key => $poptarts_are_yummy)
		{
			if($key & $type)
			{
				$this->type_index[$key][] =& $this->messages[$this->message_id];
			}
		}

		return $this->message_id;
	}

	function get($type = false)
	{
		if(false === $type)
		{
			return $this->messages;
		}
		else
		{
			return $this->_get_type_index($type);
		}
	}

	function count($type = false)
	{

		if(false === $type)
		{
			return sizeof($this->messages);
		}
		else
		{
			return sizeof($this->_get_type_index($type));
		}
	}

	function dump($type = false, $textf = false, $datef = false, $return = false)
	{
		$out = '';

		if(!$textf) $textf = "<strong>%s</strong> &nbsp; &nbsp; <em>%s</em> <br />\n";
		if(!$datef) $datef = 'F j, Y, g:i a';

		foreach($this->get($type) as $msg)
		{
			$out .= sprintf($textf, $msg['text'], date($datef, $msg['date']));
		}

		if($return)
		{
			return $out;
		}
		else
		{
			echo $out;
			return;
		}
	}

	function error($text)
	{
		$this->add(REFERER_ERR_ERROR, $text);
	}

	function info($text)
	{
		$this->add(REFERER_ERR_INFO, $text);
	}

}

/* ========================================================
 * The main referer class
 */

class Referer {

	var $db; 			// Wordpress database class to use.

	var $errors;		// List of any problems that happened
	var $messages;		// List of any status reports

	var $charset;

	var $options;
	var $option_defaults;

	/* Options */
	var $version;

	var $url_limit;
	var $host_limit;
	var $word_limit;
	var $phrase_limit;

	var $urls_format;
	var $hosts_format;
	var $words_format;
	var $phrases_format;
	var $referers_format;

	var $blacklist_localhost;
	var $blacklist_searchhosts;
	var $searchengines;
	var $host_blacklist;
	var $word_blacklist;

	var $logformat;

	/*
	 * void Referer(void)
	 * The constructor function.
	 */
	function Referer()
	{
		// Wordpress database class.
		$this->db = new Referer_DataBase();

		$this->errors = array();
		$this->messages = array();

		$this->msg = new Referer_ErrorHandler();

		$this->charset = get_option('blog_charset');

		$this->options = array (
			'version', 'url_limit', 'host_limit', 'word_limit', 'phrase_limit',
			'urls_format', 'hosts_format', 'words_format', 'phrases_format',
			'referers_format', 'blacklist_localhost', 'blacklist_searchhosts',
			'searchengines', 'host_blacklist', 'word_blacklist', 'logformat'
		);
		$this->option_defaults = array (
			'version' =>  REFERER_VERSION,
			'url_limit' => 10,
			'host_limit' => 10,
			'word_limit' => 10,
			'phrase_limit' => 10,
			'urls_format' => '<li class="referers"><strong>%n%</strong> <a href="%url%" target="_blank" rel="nofollow">%snip%</a> (%count%)</li>',
			'hosts_format' => '<li class="referers"><strong>%n% </strong><a href="http://%host%/" target="_blank" rel="nofollow">%snip%</a> (%count%)</li>',
			'words_format' => '<li class="referers"><strong>%n% </strong>%snip% (%count%) <a href="http://www.google.com/search?q=%word%" title="Google for \'%word%\'" class="google" rel="nofollow">G</a></li>',
			'phrases_format' => '<li class="referers"> <strong>%n% </strong>%snip% (%count%) <a href="http://www.google.com/search?q=%phrase%" title="Google for \'%phrase%\'" class="google" rel="nofollow">G</a></li>',
			'referers_format' => '<style type="text/css" media="screen"> a.google {font-weight:bold; font-family:serif; color:#8888ff; text-decoration:none;} a.google:hover {color:#3333cc;} abbr.snip {border:none;} h3.referers {margin:4px; font-size:1em;} li.referers {white-space:nowrap;} </style> <h2>参照元ランク</h2> <h3 class="referers">上位 %url-limit% Uri</h3> <ul>%top-urls%</ul> <h3 class="referers">上位 %host-limit% ホスト</h3> <ul>%top-hosts%</ul> <h3 class="referers">上位 %word-limit% の検索語</h3> <ul>%top-words%</ul> <h3 class="referers">上位 %phrase-limit% の検索語句</h3> <ul>%top-phrases%</ul>',
			'blacklist_localhost' =>  1 ,
			'blacklist_searchhosts' => 1,
			'searchengines' => array('/^http:\/\/www\.google\.[^\?]*\?.*q=([^&]*)&.*$/i', '/^http:\/\/search\.msn\.[^\?]*\?.*q=([^&]*)&.*$/i', '/^http:\/\/.*teoma\.com[^\?]*\?.*q=([^&]*)&.*$/i', '/^http:\/\/.*ask\.com[^\?]*\?.*q=([^&]*)&.*$/i', '/^http:\/\/.*\.yahoo\..*[^\?]*\?.*p=([^&]*)&.*$/i'),
			'host_blacklist' => array('example.com', 'example.co.uk'),
			'word_blacklist' =>  array('and', 'or', 'if', 'not', 'at', 'the'),
			'logformat' => '%h %l %u %t \"%r\" %>s %b \"%{Referer}i\" \"%{User-Agent}i\"',
		);
		$this->_loadOptions();

		// Install options panel.
		add_action('admin_menu', array('referer', '__adminMenu'));

		// Parse the current HTTP_REFERER
		if($this->_installedVersion() == REFERER_VERSION)
			$this->_doReferer();


	} // end function referer


	function _loadOptions()
	{
		foreach($this->options as $op)
		{
			$this->{$op} = get_option(REFERER_OPTIONS_PREFIX . $op);
		}

		/*
		$this->url_limit = get_option(REFERER_OPTIONS_PREFIX . 'url_limit');
		$this->host_limit = get_option(REFERER_OPTIONS_PREFIX . 'host_limit');
		$this->word_limit = get_option(REFERER_OPTIONS_PREFIX . 'word_limit');
		$this->phrase_limit = get_option(REFERER_OPTIONS_PREFIX . 'phrases_limit');
		$this->urls_format = get_option(REFERER_OPTIONS_PREFIX . 'urls_format');
		$this->hosts_format = get_option(REFERER_OPTIONS_PREFIX . 'hosts_format');
		$this->words_format = get_option(REFERER_OPTIONS_PREFIX . 'words_format');
		$this->phrases_format = get_option(REFERER_OPTIONS_PREFIX . 'phrases_format');
		$this->referers_format = get_option(REFERER_OPTIONS_PREFIX . 'referers_format');
		$this->blacklist_localhost = get_option(REFERER_OPTIONS_PREFIX . 'blacklist_localhost');
		$this->blacklist_searchhosts = get_option(REFERER_OPTIONS_PREFIX . 'blacklist_searchhosts');
		$this->searchengines = get_option(REFERER_OPTIONS_PREFIX . 'searchengines');
		$this->host_blacklist = get_option(REFERER_OPTIONS_PREFIX . 'host_blacklist');
		$this->word_blacklist = get_option(REFERER_OPTIONS_PREFIX . 'word_blacklist');
		$this->logformat = get_option(REFERER_OPTIONS_PREFIX . 'logformat');
		*/

		return true;
	}

	function _saveOptions()
	{
		foreach($this->options as $op)
		{
			update_option(REFERER_OPTIONS_PREFIX . $op, $this->{$op});
		}

		/*
		update_option(REFERER_OPTIONS_PREFIX . 'url_limit', $this->url_limit);
		update_option(REFERER_OPTIONS_PREFIX . 'host_limit', $this->host_limit);
		update_option(REFERER_OPTIONS_PREFIX . 'word_limit', $this->word_limit);
		update_option(REFERER_OPTIONS_PREFIX . 'phrases_limit', $this->phrase_limit);
		update_option(REFERER_OPTIONS_PREFIX . 'urls_format', $this->urls_format);
		update_option(REFERER_OPTIONS_PREFIX . 'hosts_format', $this->hosts_format);
		update_option(REFERER_OPTIONS_PREFIX . 'words_format', $this->words_format);
		update_option(REFERER_OPTIONS_PREFIX . 'phrases_format', $this->phrases_format);
		update_option(REFERER_OPTIONS_PREFIX . 'referers_format', $this->referers_format);
		update_option(REFERER_OPTIONS_PREFIX . 'blacklist_localhost', $this->blacklist_localhost);
		update_option(REFERER_OPTIONS_PREFIX . 'blacklist_searchhosts', $this->blacklist_searchhosts);
		update_option(REFERER_OPTIONS_PREFIX . 'searchengines', $this->searchengines);
		update_option(REFERER_OPTIONS_PREFIX . 'host_blacklist', $this->host_blacklist);
		update_option(REFERER_OPTIONS_PREFIX . 'word_blacklist', $this->word_blacklist);
		update_option(REFERER_OPTIONS_PREFIX . 'logformat', $this->logformat);
		*/
		return true;
	}

	function _restoreOptions()
	{

		$defaults = $this->option_defaults;
		$failed = array();
		$changed = array();
		foreach($this->options as $op)
		{
			if(get_option(REFERER_OPTIONS_PREFIX . $op) != $defaults[$op])
			{
				if(update_option(REFERER_OPTIONS_PREFIX . $op, $defaults[$op]) === false)
				{
					$failed[] = $op;
				}
				else
				{
					$changed[] = $op;
					$this->{$op} = $defaults[$op];
				}
			}
		}

		if(sizeof($failed) > 0)
		{
			$this->msg->error( 'The following options could not be restored to their default values:' . implode(', ', $failed));
			return false;
		}
		elseif(sizeof($changed) == 0)
		{
			$this->msg->error( 'Options are the same as their default values so no change is required.');
			return false;
		}
		else
		{
			return true;
		}
	}



	/*
	 * array update (void)
	 *
	 * Checks $_POST array for submitted options and process them. Returns a
	 * complete list of all submitted elements whether updated or not.
	 */
	function _updateOptions()
	{
		$nl = array("\n", "\r"); //new-lines characters.

		// check posted data types.
		foreach($this->options as $op)
		{
			//$name = $this->_option($option, 'name');
			//$type = $this->_option($option, 'type');
			switch( $op )
			{
			case 'url_limit':
			case 'host_limit':
			case 'word_limit':
			case 'phrase_limit':
				if(isset($_POST[$op]))
				{
					$value = sprintf('%d', $_POST[$op]);
					if($value < -1)
						$this->msg->error( 'Field '.$op.' must not be less than -1.');
					$this->{$op} = $value;
				}
				break;

			case 'referers_format':
			case 'urls_format':
			case 'hosts_format':
			case 'words_format':
			case 'phrases_format':
			case 'logformat':
				if(isset($_POST[$op]))
				{
					$value = stripcslashes($_POST[$op]);
					if(strlen($value) < 1)
						$this->msg->error( 'Field '.$op.' is empty.');
					$this->{$op} = $value;
				}
				break;

			case 'searchengines':
				if(isset($_POST[$op]))
				{
					$value = explode("\n", stripcslashes($_POST[$op]));
					foreach($value as $k => $v)
						$value[$k] = trim($v);
					if(sizeof($value) <= 0)
						$this->msg->error( 'Field '.$op.' is empty.');
					$this->{$op} = $value;
				}
				break;

			case 'host_blacklist':
			case 'word_blacklist':
				if(isset($_POST[$op]))
				{
					$value = explode(",", stripcslashes($_POST[$op]));
					if(sizeof($value) <= 0)
						$this->msg->error( 'Field '.$op.' is empty.');
					foreach($value as $k => $v)
						$value[$k] = trim($v);
					$this->{$op} = $value;
				}
				break;

			case 'blacklist_localhost':
			case 'blacklist_searchhosts':
				if(isset($_POST[$op]))
				{
					$value =  sprintf('%d', $_POST[$op]);
					if($value != 1)
						$this->msg->error( 'Field '.$op.' should be boolean true if submitted.');
					$this->{$op} = $value;
				}
				elseif ($this->{$op} == 1)
				{
					/*
					 * Boolean checkboxs dont submit when they aren't checked so if they
					 * aren't here we have to check against current settings.
					 */
					$this->{$op} = 0;
				}
				break;

			case 'version':
				break;

			default:
				$this->msg->add(REFERER_ERR_FATAL, __CLASS__.':'.__FUNCTION__.' ERROR Bad option: '.$op);
			}// end switch
		}  //end if

		/*
		 * TODO: Some fields should be allowed to be blank
		 *
		 * TODO: Extra checking on form data here:
		 *
		 * 	host_output_string is valid html and tags
		 * 	word_output_string is calid html and tags
		 * 	referers_output_string is vlaid html and tags
		 * 	searchengines are valid regexp
		 * 	host_blacklist are valid regexp
		 */

		// process the changes.
		if(!$this->msg->count(REFERER_ERR_ERROR) and $this->_saveOptions())
		{
			$this->msg->info( 'Options saved.');
			return true;
		}
		else
			return false;

	}


	/* ====================================================
	 * PRIVATE METHODS
	 */


	/*
	 * mixed option(string name [, string action [, mixed value]])
	 *
	 * Get/set options and option meta data.
	 *
	 * TODO option() is messy... think of something better. The main advantages
	 * of this is first that the class-wide names if option dont have to be the
	 * same as the names in the database. Secondy I dont have to have 8 little
	 * functions.
	 */
/*	function _option($name, $action = false, $value = false)
	{
		static $cache = array();
		$meta = $this->options_meta[$name];
		$default = $GLOBALS['_REFERER_option_defaults'][$name];

		switch($action)
		{
			case false;
			case 'get':
				if(isset($cache[$name]))
					return $cache[$name];
			case 'get_nocache':
				$cache[$name] = get_option(REFERER_OPTIONS_PREFIX.$name);
				if(!isset($this->options_meta[$name]))
					echo __CLASS__.':'.__FUNCTION__." ERROR get_option failed with: $name";
				return $cache[$name];

			case 'add':
				if(! $this->_option($name) )
				{
					$cache[$name] = ($value ? $value : $default);
					return add_option(REFERER_OPTIONS_PREFIX.$name,
						($value ? $value : $default),
						'',
						$meta['autoload']);
				}

			case 'delete': case 'del':
				unset($cache[$name]);
				return delete_option(REFERER_OPTIONS_PREFIX.$name);

			case 'reset': case 'restore':
				$cache[$name] = $default;
				return update_option(REFERER_OPTIONS_PREFIX.$name, $default);

			case 'set': case 'update':
				$cache[$name] = $value;
				return update_option(REFERER_OPTIONS_PREFIX.$name, $value);

			case 'type':
				return $meta['type'];

			case 'list':
				$list = array();
				foreach($this->options_meta as $k => $val)
					$list[] = $k;
				return $list;

			case 'title':
				return $meta['title'];

			case 'name':
				return REFERER_OPTIONS_PREFIX.$name;

			case 'clearcache':
				$cache = array();
				return true;

			default:
				 echo __CLASS__.':'.__FUNCTION__." ERROR Bad options passed: ($name, $action, $value).";
		}

	}
*/

	/*
	 * boolean isInstalled(void)
	 *
	 * Returns true if this version of the plugin has it's data installed,
	 * false otherwise.
	 */
	function _isInstalled()
	{
		if((float)sprintf('%.2f', REFERER_VERSION) == (float)sprintf('%.2f', $this->_installedVersion()))
			return true;
		else
			return false;
	}

	function _emptyData()
	{
		if(false === $this->db->emptyTable())
		{
			$this->msg->error('Failed to clear referers data.');
			return false;
		}
		else
		{
			$this->msg->info('Referers data has been cleared.');
			return true;
		}
	}

	/*
	 * boolean install(void)
	 *
	 * Installs the required database tables and options.
	 * Returns true on success, false on failure.
	 */
	function _install()
	{
		/*
		 *
		 foreach($this->_option(null, 'list') as $o)
		{
			$this->db->show_errors();

			//$this->messages[] = "Adding option $o...";
			$this->_option($o, 'add');

			/*
			 * Use of mysql_affected_rows() which isn't ideal, it would be nice
			 * if the wpdb class supported it or add_option() returned status.
			 */
		/*	if(mysql_affected_rows() != 1)
				$this->errors[] = 'Failed to add option ' . $this->_option($o, 'title');
		}
		*/

		$this->_restoreOptions();

		if(false === $this->db->createReferersTable())
			$this->msg->error('Unable to create referers data table.');

		if($this->msg->count(REFERER_ERR_ERROR))
		{
			return false;
		}
		else
		{
			return true;
		}

	}

	/*
	 * boolean uninstall(void)
	 *
	 * Removes the plugins database tables and options.
	 * Returns true on success, false on failure.
	 */
	function _uninstall()
	{

		foreach($this->options  as $o)
		{
			if(!delete_option(REFERER_OPTIONS_PREFIX . $o))
				$this->msg->error('Failed to delete option: '.$o);
		}

		if(false === $this->db->dropTable())
			$this->msg->error('Failed to drop referer data table.');


		if(0 == $this->msg->count(REFERER_ERR_ERROR))
		{
			return true;
		}
		else
		{
			return false;
		}


	}

	/* something funny going on here... dont think this is working entirely as planned. */
	function _installedVersion()
	{
		$ver = null;
		if( ($ver = get_option(REFERER_OPTIONS_PREFIX . 'version')))
			/* nada */;
		//else if( ($ver = get_option('referer_version')) and (float)0.21 == (float)$ver)
		//	/* nada */echo $ver;
		else if( ($ver = get_option('referer020_version')) and 0.20 == $ver)
			/* nada */;
		elseif( ($ver = get_option('referer_version')) and  0.10 == $ver)
			/* nada */;

		if($ver)
			return (float)sprintf('%.2f', $ver);
		else
			return false;

	}

	function _migrate()
	{
		if(! ($ver = $this->_installedVersion()) )
		{
			$this->msg->error('Plugin not installed.');
			return false;
		}
		elseif($ver == REFERER_VERSION)
		{
			$this->msg->error('No migration nessesary: Installed version is '.sprintf('%.2f',REFERER_VERSION));
			return false;
		}
		elseif($ver > REFERER_VERSION)
		{
			$this->msg->error('Installed version is '.sprintf('%.2f',REFERER_VERSION).' which is greater than this version so no migration is possible.');
			return false;
		}

		$old_prefix = '';
		$delete = array();
		$restore = array();
		$prefix_change = array();

		switch ($ver)
		{
			case 0.10:
				if(false === $this->db->dropTable('referer_hosts')
						or false === $this->db->dropTable('referer_words') )
					$this->msg->error('Unable to drop tables: referer_hosts and referer_words.');
					// TODO Move this error to DB class

				$this->db->createReferersTable();

				$old_prefix = 'referer_';
				$prefix_change = array ('host_limit','word_limit');
				$delete = array ('version',	'blacklist_local_hosts',
					'blacklist_search_hosts', 'searchengines', 'host_blacklist',
					'host_output_string', 'word_output_string',
					'referers_output_string');
				$restore = array ('url_limit', 'phrase_limit', 'urls_format',
					'hosts_format', 'words_format', 'phrases_format',
					'referers_format', 'blacklist_localhost', 'blacklist_searchhosts',
					'searchengines', 'host_blacklist', 'word_blacklist', 'logformat');

				break;

			case 0.20:

				$old_prefix = 'referer020_';
				$prefix_change = array ('url_limit','host_limit','word_limit','phrase_limit',
					'urls_format','blacklist_localhost','blacklist_searchhosts',
					'searchengines','host_blacklist','word_blacklist','logformat', 'version');
				$restore = array('hosts_format', 'words_format', 'phrases_format',
					'urls_format', 'referers_format');

				break;

			case 0.21:
				break;
			case 0.22: // never released :p
				break;

			default:
				$this->msg->error('Installed version is '.sprintf('%.2f',REFERER_VERSION).' which doesn\'t exist as far as I know!');
				return false;

		}

		foreach($delete as $name)
		{
			delete_option($old_prefix . $name);
		}

		foreach($prefix_change as $name)
		{
			$val = get_option($old_prefix . $name);
			delete_option($old_prefix . $name);
			update_option(REFERER_OPTIONS_PREFIX . $name, $val);
		}

		foreach($restore as $name)
		{
			delete_option($old_prefix . $name);
			update_option(REFERER_OPTIONS_PREFIX.$name, $this->option_defaults[$name]);
		}

		update_option(REFERER_OPTIONS_PREFIX.'version', REFERER_VERSION);

		$this->msg->info(  'Successfuly migrated version ' . $ver . ' -> ' . REFERER_VERSION);

		$this->_loadOptions();
		return true;


	}

	/*
	 * boolean restore (void)
	 *
	 * Reverts all options data to their defaults.
	 * Returns true on success, false if there was a problem
	 */
	 /*
	function _restore()
	{
		foreach($this->_option(null, 'list') as $o)
		{
			$this->_option($o, 'reset');
			if(mysql_affected_rows() != 1)
				$this->errors[] = 'Failed to add option ' .$this->_option($o, 'del');
		}
		if(! sizeof($this->errors))
			$this->messages[] = 'Plugin options installed.';

		if(sizeof($this->errors))
			return false;
		return true;
	}*/


	/*
	 * bool parseReferer(string refererUrl)
	 *
	 * Processes the referer url given in $refererUrl. Returns an array containing
	 * all the relevant commonents in an array (eg: host, words, phrases, etc..)
	 * If there url is invalid, malformed or blacklisted it returns false.
	 *
	 */
	function _parseReferer($refererUrl)
	{
		$refererUrl = trim($refererUrl);
		if('' == $refererUrl or '-' == $refererUrl)
			return false;
		if(! ($url = parse_url($refererUrl)))
			return false;
		if(! strlen($url['host']))
			return false;
		$url['url'] = $refererUrl;
		$url['search'] = '';
		$url['words'] = array();
		$url['phrases'] = array();
		$url['visibility'] = true;

		foreach($this->host_blacklist as $badhost)
		{
			if(stristr($url['host'], $badhost))
			{
				$url['visibility'] = false;
			}
		}

		if($url['visibility'] and $this->blacklist_localhost
			and	stristr($url['host'], $_SERVER['HTTP_HOST']))
		{
			$url['visibility'] = false;
		}


		// find a search query in the url
		/*
		 * TODO: Add to default search engines:  excite, hotbot, dmoz, aol, a9,
		 * technorati, altavista, images.google, dogpile, beta.search.msn,
		 * lycos, metacrawler, alltheweb, vivisimo, wisenut, gigablast, mamma,
		 * looksmart, search. netscape, clusty, about.com, iwom.ask, webcrawler
		 *
		 * TODO: searchengine regex should probably be defined as host + query
		 * identifier e.g: host: "/www.google\..+\/search/", query: "/[\?&]q=
		 * ([^&])&/" or better yet host: "www.google", query: "q" --- this
		 * should make it easiery for users to add their own.
		 */

		$searchengines = $this->searchengines;
		if(strlen($url['query']) and is_array($searchengines) and sizeof($searchengines))
		{
			foreach($searchengines as $regex)
			{
				if(strlen(trim($regex)) and preg_match($regex, $url['url'], $matches))
				{
					if(isset($matches[1]))
					{
						echo $url['search'];
						$url['search'] = strtolower(urldecode($matches[1]));
						if( $this->blacklist_searchhosts )
							$url['visibility'] = false;
						break;
					}
				}
			}
		}

		if(strlen($url['search']))
		{
			foreach(explode(' ', $url['search']) as $k => $q)
			{

				//$part = preg_replace('/[^a-z0-9"]/i', '', $q);
				//$part = preg_replace('/[+\-]/i', '', $q);


				$part = str_replace(array('+', '-'), '', $q);

				$word = trim($part, '"');
				if(!in_array($word, $this->word_blacklist))
				{
					if($word and !in_array($word, $url['words']))
						$url['words'][] = $word;
					$url['phrases'][] = $part;
				}
			}
			$parts = explode('"', implode(' ', $url['phrases']));
			$url['phrases'] = array();
			foreach($parts as $p)
				if(! in_array($p, $url['phrases']))
					$url['phrases'][] = $p;
		}

		return $url;
	} // end function parseReferer()

	/*
	 * bool doReferer(void)
	 *
	 * If HTTP_REFERER is set in the $_SERVER array then parse the url and
	 * add the host and searchword to the database.
	 *
	 * Returns true on success, false on failure.
	 */
	function _doReferer()
	{

		if(!isset($_SERVER['HTTP_REFERER']) or !$_SERVER['HTTP_REFERER'])
			return false;

		$url = trim($_SERVER['HTTP_REFERER']);

		if(! ($ref = $this->_parseReferer($url)))
			return false;
		if(! strlen($ref['host']))
			return false;

		if(!sizeof($ref['words']))
			$ref['words'] = false;
		if(!sizeof($ref['phrases']))
			$ref['phrases'] = false;

		if(! $this->db->addReferer($ref, 1))
		{
			$this->msg->error('Failed to update/insert referer.<br />');
			return false;
		}
		return true;

	}

	/*
	 * string getUpload(string name)
	 *
	 * Find a file given in an <input type=file name=$name> and for any
	 * problems. Returns false on failure or the filename on success.
	 *
	 * TODO: Support compressed file uploads. (gzip, bzip2, zip,...)
	 */

	/*
	 * string _snip(string inString [, integer maxLength])
	 *
	 * Attempts to crop paramater inString to a given length based on the likely
	 * width of characters in a 10px Sans Sherif font. If the string is too long
	 * it returns the the cropped string with a '...' suffixed. Otherwise it
	 * returns the whole string. Optional paramater maxLength sets the desired
	 * width in pixels. Optional paramater fontSize sets the size of the font in
	 * pixels.
	 *
	 * This is not very accurate but is better than nothing, it would be nice if
	 * CSS could do something like this.
	 */
	function _snip($inString, $maxLength=120, $fontSize = 10.0)
	{
		$maxLength = (int)((float)$maxLength * (10.0 / (float)$fontSize));

		static $charSize = array (
			'2' => 2,'i' => 2,'l' => 2,':' => 2,'.' => 2,'I' => 2,
			'J' => 3,
			'c' => 4,'f' => 4,'r' => 4,'t' => 4,'v' => 4,'/' => 4,'?' => 4,
			/* 5 is default so dont need these:
			'a' => 5,'b' => 5,'d' => 5,'e' => 5,'g' => 5,'h' => 5,'k' => 5,'n' => 5,
			'o' => 5,'p' => 5,'q' => 5,'s' => 5,'u' => 5,'x' => 5,'y' => 5,'2' => 5,
			'3' => 5,'4' => 5,'5' => 5,'6' => 5,'7' => 5,'8' => 5,'9' => 5,'0' => 5,
			'B' => 5,'F' => 5,'H' => 5,'N' => 5,'P' => 5,'U' => 5,'S' => 5,'1' => 4,
			'K' => 5,'L' => 5,'z' => 5,
			*/
			'A' => 6,'C' => 6,'D' => 6,'E' => 6,'G' => 6,'M' => 6,'O' => 6,
			'Q' => 6,'R' => 6,'T' => 6,'V' => 6,'X' => 6,'Y' => 6,'Z' => 6,
			'#' => 6,'&' => 6,'w' => 7,'W' => 7,'m' => 9,'%' => 8,'='=>8,'_'=>8,
		);
		$pos = 0;
		$length = 0;
		$outString = '';
		while($length < $maxLength and $pos < strlen($inString) )
		{
			$char = substr($inString, $pos, 1);
			if(isset($charSize[$char]))
				$length += $charSize[$char];
			else /* default*/
				$length += 5 ;
			$outString .= $char;
			$pos ++;
		}
		if($pos < strlen($inString))
		{
			$outString = "<abbr title='$inString' class='snip'>".substr($outString, 0, -2).'…</abbr>';

		}
		return $outString;
	}


	function _gunzip($path)
	{
		$output = null;
		$return_var = null;

		if(!file_exists($path))
		{
			$this->msg->error('File does not exist: ' . $path);
		}
		elseif('file' != filetype($path))
		{
			$this->msg->error('File does not appear to be a file (eg its a directory):' . $path);
		}
		elseif(!is_readable($path))
		{
			$this->msg->error('Unable to open file for reading: ' . $path);
		}
		elseif(ini_get('safe_mode'))
		{
			$this->msg->error( 'Server is running in safe mode; Cannot handle compressed uploads.');
		}
		elseif(! is_executable(($gunzip = trim(shell_exec('which gunzip')))))
		{
			$this->msg->error( 'gunzip is not available on this server.');
		}
		elseif(! ($outfile = tempnam('/tmp', '')) )
		{
			$this->msg->error( 'Failed to create temporary file for gunzip output.');
		}
		elseif(false === exec("$gunzip -dc ".escapeshellarg($path)." > $outfile", $output, $return_var) or $return_var !== 0)
		{
			$this->msg->error( 'Failed to gunzip file '.escapeshellarg($path).' (exit code: '.$return_var.')');
		}
		if($this->msg->count(REFERER_ERR_ERROR) > 0)
		{
			return false;
		}
		else
		{
			return $outfile; //substr($path, 0, -3);
		}

	}


	/*
	 * mixed _MaxFilesize(void)
	 *
	 * Look at PHP's configuration and return the maximum possible upload
	 * filesize is in bytes. If there is a problem it returns null.
	 *
	 * If $reformat is true then is outpus a human-readable string.
	 */
	function _maxFilesize($reformat = false)
	{
		$val['upload_max_filesize'] = ini_get('upload_max_filesize');
		$val['post_max_size'] = ini_get('post_max_size');
		$matches = array();
		foreach($val as $k => $filesize)
			if($filesize and preg_match('/^([0-9]+)([kKmMgG]?)$/', $filesize, $matches))
			{
				$val[$k] = $matches[1];
				switch (strtolower($matches[2])) {
					case 'g': $val[$k] *=  1024;
					case 'm': $val[$k] *=  1024;
					case 'k': $val[$k] *=  1024;
					break;
					default: return null;
				}
			}
		if($val['upload_max_filesize'] < $val['post_max_size'])
			$maxFilesize = $val['upload_max_filesize'];
		else
			$maxFilesize = $val['post_max_size'];

		if($reformat)
		{
			$count = 0;
			$format = array('byte', 'kilobyte', 'megabyte', 'gigabyte',
				'terabyte', 'petabyte', 'exabyte', 'zettabyte', 'yottabyte');

			while(($maxFilesize / 1024) > 1 && $count < 8)
			{
				$maxFilesize /= 1024;
				$count ++;
			}

			$maxFilesize = number_format($maxFilesize, 0 , '', '.') . ' '
				. $format[$count] . ($maxFilesize > 1 ? 's' : '' );
		}

		return $maxFilesize;
	}


	function _getLogUpload($id)
	{
		if(!ini_get('file_uploads'))
		{
			$this->msg->error( 'File uploads are disabled. Check file_uploads setting in your php.ini.');
			return false;
		}

		if(!isset($_FILES) or 0 == sizeof($_FILES))
		{
			$this->msg->error( 'No file(s) uploaded.');
			return false;
		}

		$file = null;
		foreach($_FILES as $name => $file)
		{
			if($name == $id)
			{
				$file = $file;
				break;
			}
		}

		if($file == null)
		{
			$this->msg->error( 'File upload ' . $id . ' not found.');
			return false;
		}

		$err = null;


		if(UPLOAD_ERR_OK != $file['error'])
		{
			switch($file['error'])
			{
				case UPLOAD_ERR_NO_FILE:
					$err = 'No file uploaded.';
					break;
				case UPLOAD_ERR_FORM_SIZE:
				case UPLOAD_ERR_INI_SIZE:
					$err = 'The uploaded file exceeds the upload_max_filesize directive in php.ini.';
					break;
				case UPLOAD_ERR_PARTIAL:
					$err = 'The uploaded file was only partially uploaded. Aboring.';
					break;
				case UPLOAD_ERR_NO_TMP_DIR:
					$err = 'Missing a temporary folder.';
					break;
				case UPLOAD_ERR_CANT_WRITE: //php5 only... but it won't get here in php4 so it isnt a problem.
					$err = 'Failed to write file to disk.';
					break;
				default:
					$err = 'Unknown error code: '.$file['error'];
			}
		}
		elseif('file' != filetype($file['tmp_name']))
		{
			$err = 'Upload does not appear to be a file.';
		}
		elseif(!is_uploaded_file($file['tmp_name']))
		{
			$err = 'A file upload attack is anoying but easy to detect.';
		}
		elseif($file['size'] != filesize($file['tmp_name']))
		{
			$err = 'Declared file size mismatch.';
		}
		elseif(!is_readable($file['tmp_name']))
		{
			$err = 'Unable to open file for reading';
		}

		$extension = '.' . pathinfo($file['name'], PATHINFO_EXTENSION);

		if(! (rename($file['tmp_name'], $file['tmp_name'] . $extension))
			or !is_file($file['tmp_name'] . $extension)
			or !is_writable($file['tmp_name'] . $extension))
		{
			$err =  'Unable to rename unploaded gzip file.';
		}
		else
		{
			$file['tmp_name'] .= $extension;
		}

		if(null != $err)
		{
			$this->msg->error( 'File Upload failed'.(isset($file['name']) ? " '$file[name]'" : '').': '.$err);
			return false;
		}
		else
		{
			return $file['tmp_name'];
		}
	}


	function _parseApacheLog($filename)
	{
		$matches = array();
		$referer_list = array();

		$handle = @fopen ($filename, 'r');
		if($handle == FALSE)
		{
			$this->msg->error( "Unable to open log file '$filename' for reading.");
			return false;
		}

		$regex = '!^(?:[^ ]+) (?:[^ ]+) (?:[^ ]+) \[(?:[^\]]+)\] "(?:[^ ]+) (?:[^ ]+) (?:[^/]+)/(?:[^"]+)" (?:[^ ]+) (?:[^ ]+) \"([^ ]*)\" (?:.+)!';

		$buffer = '';
		while ( !feof($handle) )
		{
			$buffer = $buffer . fread($handle, REFERER_FILE_READ_BUFFER);
			$start = 0;

			while (false != ($end = strpos($buffer, "\n", $start)))
			{
				if(!preg_match($regex, substr($buffer, $start, $end-$start), $matches) and !isset($matches[1]))
					$this->msg->info( 'Warning: Regex failed with: "' . substr($buffer, $start, $end-$start).'"');
				else
				{
					$referer = $matches[1];
					// TODO Probably not a good idea using a url as a key
					if(!isset($referer_list[$referer]))
					{
						$referer_list[$referer] =  1;
					}
					else
						$referer_list[$referer] ++;

				}
				$start = $end + 1;
			}
			// TODO: Note sure if PHP will free the memory here.. need to investigate
			$buffer =  substr($buffer, $start);
		}
		unset($buffer);

		$this->msg->info(  'Found '.sizeof($referer_list).' unique urls.');
	 	fclose($handle);

		$word_count = 0;
		$phrase_count = 0;

		foreach($referer_list as $url => $count)
		{
			if( ($ref = $this->_parseReferer($url)) )
			{
				$word_count += sizeof($ref['words']) * $count;
				$phrase_count += sizeof($ref['phrases']) * $count;
				if(! $this->db->addReferer($ref, $count) )
					$this->msg->error( 'Failed to update/insert referer.<br /><blockquote>');

			}
		}

		$this->msg->info(  'Found '.$word_count." search words. \n");
		$this->msg->info(  'Found '.$phrase_count." search phrases. \n");

		return true;

	} // end function parseApacheLog


/*	function _tempDir($dir=false, $prefix=false, $mode=false)
	{
		if(!$dir) $dir = '/tmp';
		if(!$prefix) $prefix = 'tempDir_';
		if(!$mode) $mode = 0770;
		if(! ($path = tempnam($dir, $prefix)) )
			return false;
		if(! unlink($path))
			return false;
		if(! mkdir($path, $mode))
			return false;
		return $path;
	}
*/
/*
	function _getDirContents($dir)
	{
		$files = array();
		$d = new dir($dir);
		while (false !== ($entry = $d->read()))
		{
			if(is_file($entry))
				$files[] = $dir.'/'.$entry;
			elseif (is_dir($entry) and '.' != substr($entry,1))
				$files = array_merge($files, $this->_getDirContents($dir.'/'.$entry));
		}
		return $files;
	}
*/

	/* ====================================================
	 * PUBLIC METHODS
	 */




	/*
	 * string topUrls([integer limit [, boolean echo]])
	 *
	 * They take the results of the fetchUrls() functions (see above) and format
	 * it using the XHTMl snippet in the plugin options panel. Then they output
	 * all the data into the page.
	 *
	 * Optional paramater $limit determins tha maximum number of results to return.
	 * If it is not given then it will return the number set in the plugin options
	 * panel. If limit is given as 0 (zero) the ALL results will be returned.
	 *
	 * If you set option paramater $echo as flase then instead of print the results
	 * they will be returned so you can perform additional formatting or concatonate
	 * them with other data.
	 */
 	function topUrls($limit = false, $echo = true)
	{
		if(!$this->_isInstalled())
			return false;

		if($limit === false)
			$limit = $this->url_limit;

		//$urls = $this->fetchUrls($limit);

		$urls = $this->db->fetchUrls($limit);

		$find = array ('%url%', '%snip%', '%count%', '%n%');
		if(sizeof($urls))
		{
			$format = $this->urls_format;
			$str = '';
			for($n = 0; $n < sizeof($urls); $n++)
			{

				$replace = array(
					$urls[$n]['url'],
					$this->_snip(substr($urls[$n]['url'], 3+ strpos($urls[$n]['url'], '://'))),
					$urls[$n]['count'], $n+1);
				$str .= str_replace($find, $replace, $format);
			}
		}
		else
			$str = 'No logged referers.';

		if($echo) echo $str;
		else return $str;

	}

	/*
	 * string topHosts([integer limit [, boolean echo]])
	 *
	 * They take the results of the fetchHosts() functions (see above) and
	 * format it using the XHTMl snippet in the plugin options panel. Then they
	 * output all the data into the page.
	 *
	 * Optional paramater $limit determins tha maximum number of results to return.
	 * If it is not given then it will return the number set in the plugin options
	 * panel. If limit is given as 0 (zero) the ALL results will be returned.
	 *
	 * If you set option paramater $echo as flase then instead of print the results
	 * they will be returned so you can perform additional formatting or concatonate
	 * them with other data.
	 */
 	function topHosts($limit = false, $echo = true)
	{
		if(!$this->_isInstalled())
			return false;

		if($limit === false)
			$limit = $this->host_limit;

		$hosts = $this->db->fetchHosts($limit);

		$find = array ('%host%','%snip%', '%count%', '%n%');
		if(sizeof($hosts))
		{
			$format = $this->hosts_format;
			$str = '';
			for($n = 0; $n < sizeof($hosts); $n++)
			{
				$str .= str_replace(
					$find,
					array($hosts[$n]['host'], $this->_snip($hosts[$n]['host']), $hosts[$n]['count'], $n+1),
					$format);
			}
		}
		else
			$str = 'No logged referers.';

		if($echo) echo $str;
		else return $str;
	}

	/*
	 * string topWords([integer limit [, boolean echo]])
	 *
	 * They take the results of the fetchWords() functions (see above) and
	 * format it using the XHTMl snippet in the plugin options panel. Then they
	 * output all the data into the page.
	 *
	 * Optional paramater $limit determins tha maximum number of results to return.
	 * If it is not given then it will return the number set in the plugin options
	 * panel. If limit is given as 0 (zero) the ALL results will be returned.
	 *
	 * If you set option paramater $echo as flase then instead of print the results
	 * they will be returned so you can perform additional formatting or concatonate
	 * them with other data.
	 */
 	function topWords($limit = false, $echo = true)
	{
		if(!$this->_isInstalled())
			return false;

		if($limit === false)
			$limit = $this->word_limit;

		if($limit == 0) // XXX ???wtf???
			$limit = PHP_INT_MAX;

		$words = $this->db->fetchWords($limit);

		$find = array ('%word%', '%snip%', '%count%', '%n%');
		if($words and sizeof($words))
		{
			$format = $this->words_format;
			$str = '';
			for($n = 0; $n < sizeof($words); $n++)
			{
				$str .= str_replace($find, array(
					$words[$n]['word'],
					$this->_snip($words[$n]['word']),
					$words[$n]['count'], $n+1), $format);
			}
		}
		else
			$str = 'No logged referers.';

		if($echo) echo $str;
		else return $str;
	}

	/*
	 * string topPhrases([integer limit [, boolean echo]])
	 *
	 * They take the results of the fetchPhrases() functions (see above) and
	 * format it using the XHTMl snippet in the plugin options panel. Then they
	 * output all the data into the page.
	 *
	 * Optional paramater $limit determins tha maximum number of results to return.
	 * If it is not given then it will return the number set in the plugin options
	 * panel. If limit is given as 0 (zero) the ALL results will be returned.
	 *
	 * If you set option paramater $echo as flase then instead of print the results
	 * they will be returned so you can perform additional formatting or concatonate
	 * them with other data.
	 */
 	function topPhrases($limit = false, $echo = true)
	{
		if(!$this->_isInstalled())
			return false;

		if($limit === false)
			$limit = $this->phrase_limit;

		if($limit == 0)
			$limit = PHP_INT_MAX;

		$phrases = $this->db->fetchPhrases($limit);

		$find = array ('%phrase%', '%snip%', '%count%', '%n%');
		if($phrases and sizeof($phrases))
		{
			$format = $this->phrases_format;
			$str = '';
			for($n = 0; $n < sizeof($phrases); $n++)
			{
				$str .= str_replace($find, array(
					$phrases[$n]['phrase'],
					$this->_snip($phrases[$n]['phrase']),
					$phrases[$n]['count'],
					$n+1), $format);
			}
		}
		else
			$str = 'No logged referers.';

		if($echo) echo $str;
		else return $str;
	}

	/*
	 * string topReferers([integer limit [, boolean echo]])
	 *
	 * They take the results of the top*() functions (e.g: topUrls, topHosts --
	 * see above) and format it using the XHTMl snippets in the plugin options
	 * panel. Then they output all the data into the page.
	 *
	 * Optional paramater $limit determins tha maximum number of results to return.
	 * If it is not given then it will return the number set in the plugin options
	 * panel. If limit is given as 0 (zero) the ALL results will be returned.
	 *
	 * If you set option paramater $echo as flase then instead of print the results
	 * they will be returned so you can perform additional formatting or concatonate
	 * them with other data.
	 */
 	function topReferers($limit = false, $echo = true)
	{
		if(!$this->_isInstalled())
			return false;

		// TODO: Shouldn't show limit higher than results
		$find = array ('%url-limit%','%host-limit%','%phrase-limit%','%word-limit%');
		$replace = array ();
		if($limit)
		{
			$replace = array($limit, $limit, $limit, $limit);
		}
		elseif($limit === 0)
		{
			$replace = array('', '', '', '');
		}
		else
		{
			$replace = array (
				($this->url_limit ? $this->url_limit : ''),
				($this->host_limit ? $this->host_limit : ''),
				($this->phrase_limit ? $this->phrase_limit : ''),
				($this->word_limit ? $this->word_limit : ''));
		}
		$str = str_replace($find, $replace, $this->referers_format);

		if(strstr($str, '%top-urls%'))
			$str = str_replace('%top-urls%', $this->topUrls($limit, false), $str);
		if(strstr($str, '%top-hosts%'))
			$str = str_replace('%top-hosts%', $this->topHosts($limit, false), $str);
		if(strstr($str, '%top-words%'))
			$str = str_replace('%top-words%', $this->topWords($limit, false), $str);
		if(strstr($str, '%top-phrases%'))
			$str = str_replace('%top-phrases%', $this->topPhrases($limit, false), $str);

		if($echo) echo $str;
		else return $str;

	}

	/* ====================================================
	 * CALLBACK FUNCTIONS
	 */

	/*
	 * void __adminMenu(void)
	 *
	 * Call-back function used my WordPress to insert the plugins options panel
	 * in the admin section.
	 */
	function __adminMenu() {
	    if (function_exists('add_options_page')) {
			add_options_page('Referer Plugin Options', 'Referer', 10, basename(__FILE__), array('Referer', '__optionsSubpanel'));
	    }
	    else
	    	echo __CLASS__.':'.__FUNCTION__.' ERROR Oh Oh!';
 	}

	/*
	 * void optionsSubpanel(void)
	 *
	 * Call-back function used by WordPress to draw the options panel in the
	 * admin section.
	 *
	 * TODO2 Implement localisation with _e() ...note to self: remeber to update
	 * the submit button match strings.
	 */
	function __optionsSubpanel()
	{
		/* Gets called by call_user_func_array() thus is run outside the instance scope.
		   So in this case re-call itself as an object method. */
		if(!is_object($this) and $this !== $GLOBALS[REFERER_OBJECT_NAME])
		{
		 	if(isset($GLOBALS[REFERER_OBJECT_NAME]))
			 	return $GLOBALS[REFERER_OBJECT_NAME]->{__FUNCTION__}();
			else  echo __CLASS__.':'.__FUNCTION__.' ERROR No instance of referer Class.';
		}

		$p = array();

		if(isset($_POST['form_id']) and strlen($_POST['form_id']))
		{
			switch($_POST['form_id'])
			{
				case 'referer_restore':
					if($this->_restoreOptions())
						$this->msg->info( 'Plugin options have been restored to their default values.');
					break;

				case 'referer_empty':
					$this->_emptyData();
					break;

				case 'referer_uninstall':
					// XXX TODO where has uninstall gone??
					if($this->_uninstall())
						$this->msg->info( 'Plugin data has been removed.');
					else
						$this->msg->error('There was a problem removing this plugins data.');
					break;

				case 'referer_install':
					if($this->_install())
						$this->msg->info( 'Plugin data has been installed.');
					else
						$this->msg->error('There was a problem installing this plugin.');
					break;

				case 'referer_migrate':
					if($this->_migrate())
						$this->msg->info(  'Plugin data has been migrated succesfully.');
					else
						$this->msg->error('There was a problem migrating the plugin data.');
					break;

				case 'referer_update':
					$this->_updateOptions();
					break;

				case 'referer_uploadlog':

					if(isset($_POST['inputmethod']) and strlen($_POST['inputmethod']))
					{
						if($_POST['inputmethod'] == 'upload')
						{
							$file = $this->_getLogUpload('fileupload');
						}
						elseif($_POST['inputmethod'] == 'path')
						{
							$file = $_POST['filepath'];
						}
						else
						{
							$this->msg->error('Unknown input method: '.$_POST['inputmethod']);
						}
					}

					if($this->msg->count(REFERER_ERR_ERROR) == 0
						and pathinfo($file, PATHINFO_EXTENSION) == 'gz'
						and false == ($file = $this->_gunzip($file) ))
					{
						/* errors set inside _gunzip() */;
					}
					elseif(!file_exists($file))
					{
						$this->msg->error('File does not exist: ' . $file);
					}
					elseif('file' != filetype($file))
					{
						$this->msg->error('File does not appear to be a file (eg its a directory):' . $file);
					}
					elseif(!is_readable($file))
					{
						$this->msg->error('Unable to open file for reading: ' . $file);
					}
					else
					{
						$this->_parseApacheLog($file);
					}

					break;

				default:
					$this->msg->error( 'Internal error: bad form ID "'.$_POST['form_id'].'".');
			}
		}

		// check for client side fail with form post.
		if ( $_SERVER['REQUEST_METHOD'] == 'POST' and !sizeof($_POST))
		{
			if(substr($_SERVER['CONTENT_TYPE'], 0, 20) == 'multipart/form-data;')
			{
				if($_SERVER['CONTENT_LENGTH'] > $this->_getUploadMaxFilesize())
				{
					$this->msg->error('File Upload failed: The uploaded file size exceeds the upload_max_filesize directive in php.ini.');
				}
				else
				{
					$this->msg->error( 'File Upload failed mysteriously: It was probably intercepted by the NSA, either that or it was abducted by aliens.');
				}
			}
			else
			{
				$this->msg->error( 'Form submission failed for no good reason.... This is one of those error messages you shouldn\'t see.');
			}
		}


		$this->__optionsSubpanel_head();

		if(!$this->_isInstalled())
		{
			if( ($ver = $this->_installedVersion()) )
			{
				$this->__optionsSubpanel_migrate();
			}
			else
			{
				$this->__optionsSubpanel_install();
			}
		}
		else
		{

			$this->__optionsSubpanel_options();

		}

		$this->__optionsSubpanel_foot();

	} // end function __optionsSubpanel()

	function __optionsSubpanel_options()
	{
		?>
		<form method="post" action="" onsubmit="return referer_disableElement('options_form_submit');">
		<input type="hidden" name="form_id" value="referer_update" />

		<fieldset class="options" style="border: solid #999 1px; margin-bottom: 2em;">
		<legend>リファラの件数</legend>
		<p>リストに表示するリファラの件数を指定します。このオプションはディスプレイ関数のパラメータでオーバーライドされます。例えば: <code>topReferers(10)</code> や <code>topHosts(23)</code>。defaultでは5です。0に設定すると全てのリファラを表示します。</p>
		 <p style="margin-left: 4em;">
			<label for="url_limit"> <strong>Urls:</strong>
			<input type="text" id="url_limit" name="url_limit" size="2" value="<?php echo $this->url_limit ?>" /> &nbsp; &nbsp; </label>
			<label for="host_limit"> <strong>ホスト:</strong>
			<input type="text" id="host_limit" name="host_limit" size="2" value="<?php echo $this->host_limit ?>" /> &nbsp; &nbsp; </label>
			<label for="word_limit"> <strong>検索語:</strong>
			<input type="text" id="word_limit" name="word_limit" size="2" value="<?php echo $this->word_limit ?>" /> &nbsp; &nbsp; </label>
			<label for="phrase_limit"> <strong>検索語句:</strong>
			<input type="text" id="phrase_limit" name="phrase_limit" size="2" value="<?php echo $this->phrase_limit ?>" /> &nbsp; &nbsp; </label>
		</p>
		</fieldset>

		<fieldset class="options" style="border: solid #999 1px; margin-bottom: 2em;">
		<legend>表示フォーマットのテンプレート</legend>
		<table class="optiontable" summary="Display format templates layout table">
			<caption style="text-align:left">リファラリストには <abbr title="eXtesible HyperText Markup Language">XHTML</abbr> のタグが使えます。Each one has a number of special tags that will be replaced with dynamic
				content when the page is generated. Technically they format the output of
				calls to display functions: <code>topReferers()</code>,
				<code>topWords()</code>, etc...
			</caption>

			<tr valign="top"><th scope="row">Top Referers</th>
			<td><label for="referers_format">
			<textarea class="code" id="referers_format" name="referers_format" rows="5" cols="60"><?php echo htmlspecialchars($this->referers_format, ENT_QUOTES, $this->charset) ?></textarea><br />
			<em><code>%url-limit%</code></em> - 表示される URI の長さ<br />
			<em><code>%host-limit%</code></em> - 表示されるホストの長さ<br />
			<em><code>%word-limit%</code></em> - 表示される検索語の長さ<br />
			<em><code>%phrase-limit%</code></em> - 表示される検索語句の長さ<br />
			<em><code>%top-urls%</code></em> - <code>getUrls()</code> で呼び出すURIの数。<br />
			<em><code>%top-hosts%</code></em> - <code>getHosts()</code> で呼び出すホスト名の数<br />
			<em><code>%top-words%</code></em> - <code>getWords()</code> で呼び出す検索語の長さ<br />
			<em><code>%top-phrases%</code></em> - <code>getPhrases()</code> で呼び出す検索語句の長さ<br />
			</label></td>
			</tr>

			<tr valign="top"><th scope="row">Uri</th>
			<td><label for="urls_format">
			<textarea class="code" id="urls_format" name="urls_format" rows="3" cols="60"><?php echo htmlspecialchars($this->urls_format, ENT_QUOTES, $this->charset) ?></textarea><br />
			<em><code>%url%</code></em> - 実際の参照元 URI <br />
			<em><code>%snip%</code></em> - 表示される URI<br />
			<em><code>%count%</code></em> - 参照回数<br />
			<em><code>%n%</code></em> - ランキング<br />
			</label></td>
			</tr>

			<tr valign="top"><th scope="row">ホスト名</th>
			<td><label for="hosts_format">
			<textarea class="code" id="hosts_format" name="hosts_format" rows="3" cols="60"><?php echo htmlspecialchars($this->hosts_format, ENT_QUOTES, $this->charset) ?></textarea><br />
			<em><code>%host%</code></em> - 実際の参照元のホスト名<br />
			<em><code>%snip%</code></em> - 表示されるホスト名<br />
			<em><code>%count%</code></em> - 参照回数<br />
			<em><code>%n%</code></em> - ランキング<br />
			</label></td>
			</tr>

			<tr valign="top"><th scope="row">検索語</th>
			<td><label for="words_format">
			<textarea class="code" id="words_format" name="words_format" rows="3" cols="60"><?php echo htmlspecialchars($this->words_format, ENT_QUOTES, $this->charset) ?></textarea><br />
			<em><code>%word%</code></em> - 実際に検索された単語<br />
			<em><code>%snip%</code></em> - 表示される検索語<br />
			<em><code>%count%</code></em> - 同一語で検索された回数<br />
			<em><code>%n%</code></em> - ランキング<br />
			</label></td>
			</tr>

			<tr valign="top"><th scope="row">検索語句</th>
			<td><label for="phrases_format">
			<textarea class="code" id="phrases_format" name="phrases_format" rows="3" cols="60"><?php echo htmlspecialchars($this->phrases_format, ENT_QUOTES, $this->charset) ?></textarea><br />
			<em><code>%phrase%</code></em> - 実際の検索語<br />
			<em><code>%snip%</code></em> - 表示される検索語句<br />
			<em><code>%count%</code></em> - 同一語句での検索回数<br />
			<em><code>%n%</code></em> - ランキング<br />
			</label></td>
			</tr>
		</table>
		</fieldset>

		<fieldset class="options" style="border: solid #999 1px; margin-bottom: 2em;">
		<legend>ブラックリスト</legend>
		<table class="optiontable" summary="Blacklisting options layout table">
			<caption style="text-align:left">このオプションでは記録するリファラのコントロールをします。プラグインの有効化/設定保存の後からのリファラ<em>のみ</em>を適用/記録されます。</caption>

			<tr valign="top"><th scope="row"><?php echo $_SERVER[HTTP_HOST]; ?> からのリファラを除外する</th>
			<td><label for="blacklist_localhost">
			<input type="checkbox" name="blacklist_localhost" id="blacklist_localhost" value="1"<?php if($this->blacklist_localhost) echo ' checked="checked"' ?> /><?php echo $_SERVER[HTTP_HOST]; ?> からのリファラを記録しないようにするにはチェックを入れてください。</label></td></tr>

			<tr valign="top"><th scope="row">検索サイトからのリファラを除外する</th>
			<td><label for="blacklist_searchhosts">
			<input type="checkbox" name="blacklist_searchhosts" id="blacklist_searchhosts" value="1"<?php if($this->blacklist_searchhosts) echo ' checked="checked"' ?> />検索サイトのホスト名や URI を表示させないようにするにはチェックを入れて下さい。ここの設定は検索語や検索語句にも適用されます。</label></td></tr>

			<tr valign="top"><th scope="row">除外するホスト</th>
			<td><label for="host_blacklist">
			<textarea class="code" id="host_blacklist" name="host_blacklist" rows="3" cols="60"><?php echo htmlspecialchars(implode(', ', $this->host_blacklist), ENT_QUOTES, $this->charset) ?></textarea><br />コンマ区切りで語句/単語を入力して下さい。特定の語句/単語を含む URI を記録させない場合使います。</label></td></tr>

			<tr valign="top">
			<th scope="row">検索語の除外</th>
			<td><label for="word_blacklist">
			<textarea class="code" id="word_blacklist" name="word_blacklist" rows="3" cols="60"><?php echo htmlspecialchars(implode(', ', $this->word_blacklist), ENT_QUOTES, $this->charset) ?></textarea><br />記録させたくない検索語をコンマ区切りで入力して下さい。</label></td></tr>
		</table>
		</fieldset>

		<fieldset class="options" style="border: solid #999 1px; margin-bottom: 1em;">
		<legend>検索サイト</legend>
		<table class="optiontable" summary="search engine options layout table">
			<caption style="text-align:left">このオプションではどの検索サイトからのリファラを記録するかを設定します。	</caption>

			<tr valign="top"><th scope="row">検索サイトの設定</th>
			<td><label for="searchengines">
			<textarea class="code" id="searchengines" name="searchengines" rows="4" cols="60"><?php echo htmlspecialchars(implode("\n", $this->searchengines), ENT_QUOTES, $this->charset) ?></textarea><br />記録する検索サイトを設定します。URI の指定にはPerlの正規表現を使えます。1行につき1つの正規表現を入力して下さい。(ユーザフレンドリじゃないので色々いじくり中)</label></td></tr>

		</table>
		</fieldset>

		<div class="submit" style="margin-bottom: 2em">
		<input type="reset" name="options_form_reset" value="リセット «" />
		<input id="options_form_submit" type="submit" name="options_form_submit" value="オプションを保存 »" />
		</div>
		</form>


		<?php $this->__optionsSubpanel_apachelog(); ?>


		<table>
		<tr><td style="padding-right:1em; vertical-align:top">

			<form method="post" action="" onsubmit="return referer_disableElement('restore_form_submit');">
			<input type="hidden" name="form_id" value="referer_restore" />
			<fieldset class="options" style="border: solid #999 1px; margin-bottom: 2em;">
			<legend>デフォルトに戻す</legend>

			<p><label for="referer_form_submit">このプラグインの設定を全てデフォルトの値に戻します。今までの変更は全て失われます。</label></p>

			<div class="submit">
			<input id="restore_form_submit" type="submit" name="restore_form_submit" value="デフォルトに戻す »" onclick="return confirm('本当にデフォルトの設定に戻しますか?');" />
			</div>

			</fieldset>
			</form>

		</td><td style="padding:0em 1em; vertical-align:top">

			<form method="post" action="" onsubmit="return referer_disableElement('clear_form_submit');">
			<input type="hidden" name="form_id" value="referer_empty" />
			<fieldset class="options" style="border: solid #999 1px; margin-bottom: 2em;">
			<legend>ログを消去する</legend>

			<p><label for="clear_form_submit">記録されたリファラをきれいサッパリ消去します。取り消しはできませんのでよ〜く考えてからクリックしてください。</label></p>

			<div class="submit">
			<input id="clear_form_submit" type="submit" name="clear_form_submit" value="ログを消去 »" onclick="return confirm('本当にデータを消去しますか?');" />
			</div>
			</fieldset>
			</form>

		</td><td style="padding-left:1em; vertical-align:top">

			<form method="post" action="" onsubmit="return referer_disableElement('uninstall_form_submit');">
			<input type="hidden" name="form_id" value="referer_uninstall" />
			<fieldset class="options" style="border: solid #999 1px; margin-bottom: 2em;">
			<legend>アンインストール</legend>

			<p><label for="uninstall_form_submit">データテーブルとオプションをデータベースから完全に削除します。</label></p>

			<div class="submit">
			<input id="uninstall_form_submit" type="submit" name="uninstall_form_submit" value="アンインストール »" onclick="return confirm('本当にプラグインを削除しますか?');" />
			</div>
			</fieldset>
			</form>

		</td></tr>
		</table>
		<?php
	}

	function __optionsSubpanel_apachelog()
	{
		?>
		<form enctype="multipart/form-data" id="uploadForm" method="post" action=""  onsubmit="return referer_disableElement('uploadlog_form_submit');">
		<input type="hidden" name="form_id" value="referer_uploadlog" />

		<fieldset class="options" style="border: solid #999 1px; margin-bottom: 3em;">
		<legend>Apache のログをインポート</legend>

		<p>パソコンに保存している Apache のログファイルをアップロードするか，サーバ上のログファイルのパス入力して下さい。</p>
		<ul>

		<li>ログのフォーマットは以下のようでなくてはなりません:
		<span style="white-space: nowrap;"><code>"%h %l %u %t \"%r\" %>s %b \"%{Referer}i\" \"%{User-Agent}i\""</code></span>
		これは Apache 2系のデフォルトのフォーマットです。<code>httpd.conf</code> に記述されている <code>LogFormat</code>の行でどのフォーマットでログが記録されいるかをチェックしてください。</li>

		<li>アップロードできるファイルの最大サイズは <?php echo $this->_maxFilesize(true); ?> です。</p></li>

		<li>ファイルはプレーンテキストか gzip で圧縮されてなくてはいけません(<code>access_log.20060705.gz</code>)。 <code>.tar.gz</code> の圧縮ではダメです。</li>

		</ul>

		<table>
			<tr><th style="text-align:right">
				<label for="inputmethod_upload"  onclick="referer_logSelectUpload()"> <strong>ログをアップロード:</strong>
				<input type="radio" name="inputmethod" id="inputmethod_upload" value="upload" checked="checked" onclick="referer_logSelectUpload()">
				</label>
			</td><td>
				<input type="hidden" name="MAX_FILE_SIZE" value="<?php echo $this->_maxFilesize(); ?>" />
				<input class="fileupload"  size="70" type="file" id="fileupload" name="fileupload" onclick="referer_logSelectUpload()" />
			</td></tr>
			<tr><th style="text-align:right">
				<label for="inputmethod_path"  onclick="referer_logSelectPath()"> <strong>ログのパスを入力:</strong>
				<input type="radio" name="inputmethod" id="inputmethod_path" value="path" onclick="referer_logSelectPath()">
				</label>
			</td><td>
				<input type="text" name="filepath" size="70" id="filepath" value="<?php echo (isset($_POST['filepath']) ? $_POST['filepath'] : dirname($_SERVER['SCRIPT_FILENAME'])); ?>"  onclick="referer_logSelectPath()" />
			</td></tr>
		</table>

		<script type="text/javascript" > //<![CDATA[

			var filepath = referer_findObj('filepath');
			var fileupload = referer_findObj('fileupload');
			var inputmethod_path = referer_findObj('inputmethod_path');
			var inputmethod_upload = referer_findObj('inputmethod_upload');

			fileupload.disabled = false;
			filepath.disabled = true;
			inputmethod_upload.checked = true;

			function referer_logSelectUpload()
			{
				fileupload.disabled = false;
				filepath.disabled = true;
				inputmethod_upload.checked = true;
				fileupload.focus();
			}

			function referer_logSelectPath()
			{
				filepath.disabled = false;
				fileupload.disabled = true;
				inputmethod_path.checked = true;
				filepath.focus();
			}


		//]]>  </script><noscript></noscript>

		<div class="submit">
		<input id="uploadlog_form_submit" type="submit" name="uploadlog_form_submit" value="ログのインポート »" />
		</div>
		</fieldset>
		</form>
		<?php
	}

	function __optionsSubpanel_head()
	{
		?>

		<script type="text/javascript" > //<![CDATA[

			function referer_findObj(n, d)
			{
				var p, i, x;
				if(!d) d = document;
				if((p = n.indexOf("?")) > 0 && parent.frames.length)
				{
					d = parent.frames[n.substring(p + 1)].document;
					n = n.substring(0, p);
				}
				if(!(x = d[n]) && d.all) x = d.all[n];
				for (i = 0; !x && i < d.forms.length; i++)
					x = d.forms[i][n];
				for(i = 0; !x && d.layers && i < d.layers.length; i++)
					x = findObj(n, d.layers[i].document);
				if(!x && document.getElementById)
					x=document.getElementById(n);
				return x;
			}

			function referer_disableElement(element)
			{
				var obj = referer_findObj(element);
				obj.disabled = true;
				return true;
			}

		//]]> </script><noscript></noscript>

		<div class="wrap">
		<h2>リファラオプション</h2>
		<?php
		if($this->msg->count(REFERER_ERR_ERROR))
		{
			?>
				<div class="error">
				<div><strong>設定の反映時にトラブルが起こりました。</strong></div>
				<p style="margin-left: 40px;">
				<?php
					$this->msg->dump(REFERER_ERR_ERROR);
				?>
				</p></div>
			<?php
		}

		if($this->msg->count(REFERER_ERR_INFO))
		{
			?>
				<div class="updated"><p>
				<?php
					$this->msg->dump(REFERER_ERR_INFO);
				?>
				</p></div>
			<?php
		}

	}

	function __optionsSubpanel_foot()
	{
		if(REFERER_DEBUG == true): ?>
		<h3>$_POST</h3><blockquote><pre><code><?php echo htmlspecialchars(print_r($_POST, true)); ?></code></pre></blockquote>
		<h3>$_FILES</h3><blockquote><pre><code><?php echo htmlspecialchars(print_r($_FILES, true)); ?></code></pre></blockquote>
		<h3>Object</h3><blockquote><pre><code><?php htmlspecialchars(print_r($this)); ?></code></pre></blockquote>
		<?php endif;

		?>
		</div>
		<?php
	}

	function __optionsSubpanel_migrate()
	{
		?>
			<h3>プラグインの移行</h3>
			<form method="post" action="" onsubmit="return referer_disableElement('migrate_form_submit');">
			<input type="hidden" name="form_id" value="referer_migrate" />
			<fieldset class="options">
			<p>新しいバージョンのプラグインがインストールされました。メッセージを読んで"移行"ボタンをクリックして新しいバージョンのプラグインを使えるようにして下さい。</p>
			<p><strong>
			使用中のバージョン: <?php echo $this->_installedVersion(); ?> <br />
			新バージョン: <?php echo REFERER_VERSION; ?>
			</strong></p><ul>
			<?php
				switch ($this->_installedVersion())
				{
				case 0.10:
					echo '<li><strong>バージョン 0.10</strong>: 残念ですが， バージョン 0.1.0 は他のバージョンとの互換性がありません。設定したオプションやデータは移行時に失われます。ご不便をおかけして大変申し訳ありません。</li>';
					break;
				case 0.20:
					echo '<li><strong>バージョン 0.20</strong>: 新しい出力フォーマットは' .
							'今使っているテンプレートを上書きします。</li>';
				}
			?>
			</ul>
			<div class="submit">
			<input type="submit" id="migrate_form_submit" name="migrate_form_submit" value="Migrate »" />
			</div>
			</fieldset>
			</form>
		<?php
	}

	function __optionsSubpanel_install()
	{
		?>
			<h3>Install</h3>
			<form method="post" action=""  onsubmit="return referer_disableElement('install_form_submit');">
			<input type="hidden" name="form_id" value="referer_install" />
			<fieldset class="options">
			<p><label for="install_form_submit">プラグインに必要なテーブルの作成とデフォルトオプションの設定をするには下にあるインストールボタンを押して下さい。</label></p>
			<div class="submit">
			<input type="submit" id="install_form_submit" name="install_form_submit" value="インストール »" />
			</div>
			</fieldset>
			</form>
		<?php
	}

} // end class referer


/* ========================================================
 * Create an instance of the referer class.
 */

${REFERER_OBJECT_NAME} = new Referer();


?>