/**
 * Copyright (c) 2006, yher2.net
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.workstyle.manager;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import net.yher2.commons.io.Classpath;
import net.yher2.commons.torque.Transaction;
import net.yher2.workstyle.TaskQuery;
import net.yher2.workstyle.exception.DuplicateException;
import net.yher2.workstyle.exception.NotFoundException;
import net.yher2.workstyle.test.DBTestCase;
import net.yher2.workstyle.torque.Tag;
import net.yher2.workstyle.torque.TagPeer;
import net.yher2.workstyle.torque.TagType;
import net.yher2.workstyle.torque.TagTypePeer;
import net.yher2.workstyle.torque.bean.TagBean;
import net.yher2.workstyle.torque.bean.TagTypeBean;

import org.apache.torque.NoRowsException;
import org.apache.torque.TooManyRowsException;
import org.apache.torque.TorqueException;
import org.apache.torque.util.Criteria;

public class TagManagerTest extends DBTestCase {
	TagManager manager = null;
	Connection con = null;
	Classpath testData = new Classpath("net/yher2/workstyle/manager/TagManagerTest.xls");

	protected void setUp() throws Exception {
		super.setUp();
		super.prepare(testData);
		
		con = getConnection();
		con.setAutoCommit(false);
		manager = new TagManager(con);
	}

	protected void tearDown() throws Exception {
		super.tearDown();
		Transaction.rollback(con);
		super.clear(testData);
	}

	public void testAdd_TAGテーブルにレコードを追加する() throws Exception{
		TagBean tag = manager.add("新規タグ追加テスト");
		
		assertEquals("新規タグ追加テスト", tag.getName());
		assertEquals(Tag.ACTIVE, tag.getActive());
		assertEquals(TagType.DEFAULT_ID, tag.getTagTypeId());
		assertNotNull(tag.getTagTypeBean());
	}
	public void testAdd_同名のタグがある場合はDuplicateException() throws Exception {
		try {
			manager.add("既存タグ");
		} catch (DuplicateException e) {
			return;
		}
		fail();
	}
	public void testAddIfNotExist_TAGテーブルにレコードを追加する() throws Exception {
		String tagString = "新規タグ追加テスト";
		List<String> tagList = new ArrayList<String>();
		tagList.add(tagString);
		assertEquals(0, TagPeer.doSelectByName(tagList, con).size());
		TagBean tag = manager.add(tagString);
		
		assertEquals(tagString, tag.getName());
		assertEquals(Tag.ACTIVE, tag.getActive());
		assertEquals(TagType.DEFAULT_ID, tag.getTagTypeId());
		assertNotNull(tag.getTagTypeBean());
	}
	public void testAddIfNotExist_同名のタグがある場合は何もしない() throws Exception {
		String tag = "既存タグ";
		List<String> tagList = new ArrayList<String>();
		tagList.add(tag);
		assertEquals(1, TagPeer.doSelectByName(tagList, con).size());

		manager.addIfNotExist(tag);
		assertEquals(1, TagPeer.doSelectByName(tagList, con).size());
	}

	public void testUpdateType_TAGテーブルのタスク種別を更新する() throws Exception {
		TagBean result = manager.get(1001);
		assertEquals(1001, result.getTagTypeId());
		
		result = manager.updateType(result.getTagId(), 1002);
		assertEquals(1002, result.getTagTypeId());
		assertEqualsDB(result);
	}
	public void testUpdateType_タグが存在しない場合はNotFound() throws Exception {
		try {
			manager.updateType(0, 1002);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	public void testUpdateType_タグ種別が存在しない場合はNotFound() throws Exception {
		try {
			manager.updateType(1001, -1);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}

	public void testToggleActivity_タグがACTIVEな場合はINACTIVEにINACTIVEな時はACTIVEに更新する() throws Exception {
		TagBean result = manager.get(1001);
		assertTrue(result.isActiveAsBoolean());
		
		result = manager.toggleActivity(result.getTagId());
		
		assertFalse(result.isActiveAsBoolean());
		assertEqualsDB(result);
		
		result = manager.toggleActivity(result.getTagId());
		
		assertTrue(result.isActiveAsBoolean());
		assertEqualsDB(result);
	}
	public void testToggleActivity_タグが存在しない場合はNotFound() throws Exception {
		try {
			manager.toggleActivity(0);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	
	private void assertEqualsDB(TagBean result) throws NoRowsException, TooManyRowsException, TorqueException {
		TagBean expected = TagPeer.retrieveByPK(result.getTagId(), con).getBean();
		assertEquals(expected.getTagId(), result.getTagId());
		assertEquals(expected.getName(), result.getName());
		assertEquals(expected.getActive(), result.getActive());
		assertEqualsDB(result.getTagTypeBean());
	}
	private void assertEqualsDB(TagTypeBean result) throws NoRowsException, TooManyRowsException, TorqueException {
		TagTypeBean expected = TagTypePeer.retrieveByPK(result.getTagTypeId(), con).getBean();
		assertEquals(expected.getName(), result.getName());
		assertEquals(expected.getStyleClass(), result.getStyleClass());
		assertEquals(expected.getSortOrder(), result.getSortOrder());
	}

	public void testList_ACTIVEなタグのみを返す() throws Exception {
		List<TagBean> result = manager.list();
		assertTrue(result.size() > 0);
		for (TagBean tag : result) {
			assertTrue(tag.isActiveAsBoolean());
		}
	}
	public void testList_ソートはタグ種別のソート順_タグ種別の名前順_名前順() throws Exception {
		List<TagBean> result = manager.list();
		assertTrue(result.size() > 0);
		assertSortOrder(result);
	}
	public void testList_タグ種別も同時に取得する() throws TorqueException {
		List<TagBean> result = manager.list();
		assertTrue(result.size() > 0);
		for (TagBean tag : result) {
			assertNotNull(tag.getTagTypeBean());
		}
	}

	public void testListAll_TAGテーブルの全件を返す() throws TorqueException {
		List<TagBean> result = manager.listAll();
		assertTrue(result.size() > 0);
		assertEquals(TagPeer.doSelect(new Criteria(), con).size(), result.size());
	}
	public void testListAll_ソートはタグ種別のソート順_タグ種別の名前順_名前順() throws TorqueException {
		List<TagBean> result = manager.listAll();
		assertTrue(result.size() > 0);
		assertSortOrder(result);
	}

	private void assertSortOrder(List<TagBean> result) {
		int order = 0;
		String typeName = "";
		String name = "";
		for (TagBean tag : result) {
			TagTypeBean type = tag.getTagTypeBean();
			assertTrue(
					(order < type.getSortOrder()) ||
					((order == type.getSortOrder()) && (typeName.compareTo(type.getName()) < 0)) ||
					((order == type.getSortOrder()) && (typeName.compareTo(type.getName()) == 0)) && (name.compareTo(tag.getName()) < 0));
			name = tag.getName();
			order = type.getSortOrder();
			typeName = type.getName();
		}
	}
	public void testListAll_タグ種別も同時に取得する() throws TorqueException {
		List<TagBean> result = manager.listAll();
		assertTrue(result.size() > 0);
		for (TagBean tag : result) {
			assertNotNull(tag.getTagTypeBean());
		}
	}

	public void testDelete_TAGテーブルからレコードを削除する() throws Exception{
		TagBean tag = manager.get(1001);
		manager.delete(tag.getTagId());
		
		try {
			manager.get(tag.getTagId());
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	public void testDelete_タスクからタグを削除する() throws Exception {
		TagBean tag = manager.get(1001);
		manager.delete(tag.getTagId());
		
		TaskQuery query = new TaskQuery();
		query.setTag(new String[]{tag.getName()});
		assertEquals(0, new TaskManager(con).search(query, null).size());
	}
	public void testDelete_タグが存在しない場合はNotFound() throws Exception {
		try {
			manager.delete(-1);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	public void tesGet_タグとタグ種別を同時に取得する() throws Exception {
		TagBean tag = manager.get(1001);
		
		assertNotNull(tag.getTagTypeBean());
		
		Tag dbTag = TagPeer.retrieveByPK(tag.getTagId(), con);
		TagType dbType = TagTypePeer.retrieveByPK(tag.getTagTypeId(), con);
		
		assertEquals(dbTag.getName(), tag.getName());
		assertEquals(dbTag.getActive(), tag.getActive());
		
		assertEquals(dbType.getName(), tag.getTagTypeBean().getName());
		assertEquals(dbType.getStyleClass(), tag.getTagTypeBean().getStyleClass());
		assertEquals(dbType.getSortOrder(), tag.getTagTypeBean().getSortOrder());
		assertEquals(dbType.getTagTypeId(), tag.getTagTypeBean().getTagTypeId());
	}
	public void tesGet_存在しない場合はNotFound() throws Exception {
		try {
			manager.get(0);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}

}
