/**
 * Copyright (c) 2006, yher2.net
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.workstyle.manager;

import java.sql.Connection;
import java.util.List;

import org.apache.torque.NoRowsException;
import org.apache.torque.TooManyRowsException;
import org.apache.torque.TorqueException;
import org.apache.torque.util.Criteria;

import net.yher2.commons.io.Classpath;
import net.yher2.commons.torque.Transaction;
import net.yher2.workstyle.exception.NotFoundException;
import net.yher2.workstyle.test.DBTestCase;
import net.yher2.workstyle.torque.TagPeer;
import net.yher2.workstyle.torque.TagType;
import net.yher2.workstyle.torque.TagTypePeer;
import net.yher2.workstyle.torque.bean.TagTypeBean;

public class TagTypeManagerTest extends DBTestCase {
	TagTypeManager manager = null;
	Connection con = null;
	Classpath testData = new Classpath("net/yher2/workstyle/manager/TagTypeManagerTest.xls");

	protected void setUp() throws Exception {
		super.setUp();
		super.prepare(testData);
		
		con = getConnection();
		con.setAutoCommit(false);
		manager = new TagTypeManager(con);
	}

	protected void tearDown() throws Exception {
		super.tearDown();
		Transaction.rollback(con);
		super.clear(testData);
	}

	public void testList_デフォルト以外のTAG_TYPEテーブルの全件を返す() throws Exception {
		List<TagTypeBean> result = manager.list();
		assertTrue( result.size() > 0 );
		assertEquals( TagTypePeer.doSelect(new Criteria(), con).size()-1, result.size() );
		
		for (TagTypeBean type : result) {
			assertTrue(type.getTagTypeId() != TagType.DEFAULT_ID);
		}
	}
	public void testList_ソートはソート順_名前順() throws Exception {
		List<TagTypeBean> result = manager.list();
		assertTrue( result.size() > 0 );
		int order = 0;
		String name = "";
		for (TagTypeBean type : result) {
			assertTrue(
					(order < type.getSortOrder()) ||
					((order == type.getSortOrder()) && (name.compareTo(type.getName()) < 0))
			);
			order = type.getSortOrder();
			name = type.getName();
		}
	}
	
	public void testAdd_TAG_TYPEテーブルにデータを挿入する() throws Exception {
		TagTypeBean expected = buildType();
		TagTypeBean result = manager.add(buildType());
		
		assertEquals(expected.getName(), result.getName());
		assertEquals(expected.getSortOrder(), result.getSortOrder());
		assertEquals(TagType.DEFAULT_STYLE, result.getStyleClass());
		assertEqualsDB(result);
	}
	public void testAdd_タグ種別名は同じでもよい() throws Exception {
		TagTypeBean expected = buildType();
		expected.setName("種別名");
		TagTypeBean result = manager.add(expected);
		assertEquals("種別名", result.getName());
		
		assertEqualsDB(result);
	}
	public void testAdd_指定以降のソート順を設定し直す() throws Exception {
		List<TagType> beforeList = listAfterOrder(3);
		
		TagTypeBean src = buildType();
		src.setTagTypeId(1001);
		src.setSortOrder(3);
		manager.add(src);
		
		List<TagType> afterList = listAfterOrder(4);
		assertEquals(beforeList.size(), afterList.size());
		for (int i=0; i<beforeList.size(); i++) {
			assertEquals(beforeList.get(i).getTagTypeId(), afterList.get(i).getTagTypeId());
		}
	}
	public void testUpdate_TAG_TYPEテーブルを更新する() throws Exception {
		TagType dbType = manager.getDBModel(1001);
		TagTypeBean expected = buildType();
		TagTypeBean src = buildType();
		src.setTagTypeId(dbType.getTagTypeId());
		
		TagTypeBean result = manager.update(src);
		
		assertEquals(expected.getName(), result.getName());
		assertEquals(expected.getSortOrder(), result.getSortOrder());
		assertEquals(dbType.getStyleClass(), result.getStyleClass());
		assertEqualsDB(result);
	}
	public void testUpdate_タグ種別名は同じでもよい() throws Exception {
		TagTypeBean src = buildType();
		src.setName("種別名");
		src.setTagTypeId(1001);
		TagTypeBean result = manager.update(src);
		assertEquals("種別名", result.getName());
		
		assertEqualsDB(result);
	}
	private List<TagType> listAfterOrder(int order) throws TorqueException {
		Criteria criteria = new Criteria();
		criteria.add(TagTypePeer.SORT_ORDER, order, Criteria.GREATER_EQUAL);
		criteria.addAscendingOrderByColumn(TagTypePeer.SORT_ORDER);
		criteria.addAscendingOrderByColumn(TagTypePeer.NAME);
		return TagTypePeer.doSelect(criteria, con);
	}
	public void testUpdate_指定以降のソート順を設定し直す() throws Exception {
		List<TagType> beforeList = listAfterOrder(3);
		
		TagTypeBean src = buildType();
		src.setTagTypeId(1001);
		src.setSortOrder(3);
		manager.update(src);
		
		List<TagType> afterList = listAfterOrder(4);
		assertEquals(beforeList.size(), afterList.size());
		for (int i=0; i<beforeList.size(); i++) {
			assertEquals(beforeList.get(i).getTagTypeId(), afterList.get(i).getTagTypeId());
		}
	}
	public void testUpdate_タグ種別が存在しない場合はNotFound() throws Exception {
		TagTypeBean src = buildType();
		src.setTagTypeId(-1);
		try {
			manager.update(src);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	public void testUpdateStyle_TAG_TYPEテーブルを更新する() throws Exception {
		String style = "更新テスト";
		TagTypeBean result = manager.updateStyle(1001, style);
		
		assertEquals(style, result.getStyleClass());
		assertEqualsDB(result);
	}
	public void testUpdateStyle_タグ種別が存在しない場合はNotFound() throws Exception {
		try {
			manager.updateStyle(-1, "hoge");
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	public void testDelete_TAG_TYPEテーブルからデータを削除する() throws Exception {
		TagType type = manager.getDBModel(1001);
		manager.delete(type.getTagTypeId());
		
		try {
			manager.getDBModel(type.getTagTypeId());
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	public void testDelete_指定タグ種別のタグの種別をデフォルトに戻す() throws Exception {
		manager.delete(1001);
		Criteria criteria = new Criteria();
		criteria.add(TagPeer.TAG_TYPE_ID, 1001);
		assertEquals(0, TagPeer.doSelect(criteria, con).size());
	}
	public void testDelete_タグ種別が存在しない場合はNotFound() throws Exception {
		try {
			manager.delete(-1);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	private void assertEqualsDB(TagTypeBean result) throws NoRowsException, TooManyRowsException, TorqueException {
		TagTypeBean expected = TagTypePeer.retrieveByPK(result.getTagTypeId(), con).getBean();
		assertEquals(expected.getName(), result.getName());
		assertEquals(expected.getStyleClass(), result.getStyleClass());
		assertEquals(expected.getSortOrder(), result.getSortOrder());
	}
	private TagTypeBean buildType() {
		TagTypeBean type = new TagTypeBean();
		type.setName("名前");
		type.setSortOrder(3);
		type.setStyleClass("style");
		return type;
	}
}
