/**
 * Copyright (c) 2006, yher2.net
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.workstyle.manager;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import net.yher2.commons.io.Classpath;
import net.yher2.workstyle.FileData;
import net.yher2.workstyle.exception.NotFoundException;
import net.yher2.workstyle.test.DBTestCase;
import net.yher2.workstyle.torque.AttachedFile;
import net.yher2.workstyle.torque.AttachedFilePeer;
import net.yher2.workstyle.torque.FileInfo;
import net.yher2.workstyle.torque.FileInfoPeer;
import net.yher2.workstyle.torque.TaskFileRelationPeer;
import net.yher2.workstyle.torque.bean.FileInfoBean;

import org.apache.torque.NoRowsException;
import org.apache.torque.Torque;
import org.apache.torque.util.Criteria;
import org.apache.torque.util.Transaction;

public class FileManagerTest extends DBTestCase {
	FileManager manager;
	Connection con = null;
	Classpath testData = new Classpath("net/yher2/workstyle/manager/FileManagerTest.xls");

	protected void setUp() throws Exception {
		super.setUp();
		prepare(testData);

		con = Transaction.begin(Torque.getDefaultDB());
		manager = new FileManager(con);
	}

	protected void tearDown() throws Exception {
		super.tearDown();
		Transaction.rollback(con);
		clear(testData);
	}

	/*
	 * Test method for 'net.yher2.workstyle.manager.FileManager.getFile(int)'
	 */
	public void testGetFile_指定のファイルのデータを返す() throws Exception {
		int FILE_ID = 1;
		FileData result = manager.getFile(FILE_ID);
		
		FileInfo expected = manager.getDBModel(FILE_ID);
		
		assertEquals(expected.getName(), result.getName());
		assertEquals(expected.getPath(), result.getDirPath() + result.getName());
		assertTrue(result.getDirPath().length() > 0);
	}
	public void testGetFile_実ファイルが見つからない場合はNotFound() throws Exception {
		int FILE_ID = 2;
		
		try {
			manager.getFile(FILE_ID);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}
	public void testGetFile_DBレコードが見つからない場合はNotFound() throws Exception {
		int FILE_ID = 0;
		
		try {
			manager.getFile(FILE_ID);
		} catch (NotFoundException e) {
			return;
		}
		fail();
	}

	/*
	 * Test method for 'net.yher2.workstyle.manager.FileManager.add(int, FileData)'
	 */
	public void testAdd_TASK_FILE_RELATIONとFILEテーブルにレコードが登録されていること() throws Exception {
		FileData file = new FileData("Test.File", new byte[]{1}, ".");
		try {
			FileInfoBean result = manager.add(1, file);
			
			manager.getDBModel(result.getFileId());
			
			Criteria criteria = new Criteria();
			criteria.add(TaskFileRelationPeer.FILE_ID, result.getFileId());
			assertTrue(TaskFileRelationPeer.doSelect(criteria, con).size() > 0);
			
			FileInfo dbFileInfo = FileInfoPeer.retrieveByPK(result.getFileId(), con);
			assertEquals(file.getName(), result.getName());
			assertEquals(file.getName(), dbFileInfo.getName());
		} finally {
			java.io.File sysFile = new java.io.File(file.getDirPath(), file.getName());
			sysFile.delete();
		}
	}
	public void testAdd_ファイルシステム上にファイルが追加されていること() throws Exception {
		FileData file = new FileData("Test.File", new byte[]{1}, ".");
		try {
			FileInfoBean result = manager.add(1, file);
			
			FileInfo dbFile = manager.getDBModel(result.getFileId());
			java.io.File sysFile = new java.io.File(dbFile.getPath());
			
			assertTrue(sysFile.exists());
		} finally {
			java.io.File sysFile = new java.io.File(file.getDirPath(), file.getName());
			sysFile.delete();
		}
	}

	/*
	 * Test method for 'net.yher2.workstyle.manager.FileManager.delete(int)'
	 */
	public void testDelete_実ファイルが削除されていること() throws Exception {
		FileData data = new FileData("test.file", new byte[]{1}, ".");
		FileInfoBean file = manager.add(1, data);
		manager.delete(file.getFileId());
		
		java.io.File sysFile = new java.io.File(file.getPath());
		assertFalse(sysFile.exists());
	}
	public void testDelete_TASK_FILE_RELATIONテーブルからレコードが削除されていること() throws Exception {
		FileData data = new FileData("test.file", new byte[]{1}, ".");
		FileInfoBean file = manager.add(1, data);
		FileInfo dbFile = manager.getDBModel(file.getFileId());
		
		manager.delete(dbFile.getFileId());
		
		Criteria criteria = new Criteria();
		criteria.add(TaskFileRelationPeer.FILE_ID, dbFile.getFileId());
		assertEquals(0, TaskFileRelationPeer.doSelect(criteria, con).size());
	}
	public void testDelete_FILEテーブルからレコードが削除されていること() throws Exception {
		FileData data = new FileData("test.file", new byte[]{1}, ".");
		FileInfoBean file = manager.add(1, data);
		FileInfo dbFile = manager.getDBModel(file.getFileId());

		manager.delete(dbFile.getFileId());
		
		try {
			FileInfoPeer.retrieveByPK(dbFile.getFileId(), con);
		} catch (NoRowsException e) {
			return;
		}
		fail();
	}
	public void testDelete_レコードが存在しない場合は何もしない() throws Exception {
		manager.delete(0);
	}
	public void testDelete_実ファイルが存在しない場合はレコードだけ削除() throws Exception {
		int fileId = 2;
		manager.delete(fileId);
		
		try {
			FileInfoPeer.retrieveByPK(fileId, con);
		} catch (NoRowsException e) {
			return;
		}
		fail();
	}

	/*
	 * Test method for 'net.yher2.workstyle.manager.FileManager.delete(List<AttachedFile>)'
	 */
	public void testDelete_Fileリストで削除_実ファイルが削除されていること() throws Exception {
		List<FileInfo> fileList = addTestFiles();

		manager.delete(fileList);
		
		for (FileInfo file : fileList) {
			java.io.File sysFile = new java.io.File(file.getPath());
			assertFalse(sysFile.exists());
		}
	}
	public void testDelete_Fileリストで削除_TASK_FILE_RELATIONテーブルからレコードが削除されていること() throws Exception {
		List<FileInfo> fileList = addTestFiles();

		manager.delete(fileList);
		
		List<Integer> idList = new ArrayList<Integer>();
		for (FileInfo file : fileList) {
			idList.add(file.getFileId());
		}
		Criteria criteria = new Criteria();
		criteria.addIn(TaskFileRelationPeer.FILE_ID, idList);
		assertEquals(0, TaskFileRelationPeer.doSelect(criteria, con).size());
	}
	public void testDelete_Fileリストで削除_FILEテーブルからレコードが削除されていること() throws Exception {
		List<FileInfo> fileList = addTestFiles();

		manager.delete(fileList);
		
		List<Integer> idList = new ArrayList<Integer>();
		for (FileInfo file : fileList) {
			idList.add(file.getFileId());
		}
		Criteria criteria = new Criteria();
		criteria.addIn(FileInfoPeer.FILE_ID, idList);
		assertEquals(0, FileInfoPeer.doSelect(criteria, con).size());
	}
	public void testDelete_Fileリストで削除_レコードが存在しない場合は何もしない() throws Exception {
		List<FileInfo> fileList = new ArrayList<FileInfo>();
		FileInfo file = new FileInfo();
		file.setFileId(0);
		file.setPath("hogehoge");
		fileList.add(file);
		fileList.add(file);
		fileList.add(file);
		
		manager.delete(fileList);
	}
	public void testDelete_Fileリストで削除_実ファイルが存在しない場合はレコードだけ削除() throws Exception {
		List<FileInfo> fileList = addTestFiles();
		for (FileInfo file : fileList) {
			java.io.File sysFile = new java.io.File(file.getPath());
			sysFile.delete();
		}
		
		manager.delete(fileList);

		List<Integer> idList = new ArrayList<Integer>();
		for (FileInfo file : fileList) {
			idList.add(file.getFileId());
		}
		Criteria criteria = new Criteria();
		criteria.addIn(FileInfoPeer.FILE_ID, idList);
		assertEquals(0, FileInfoPeer.doSelect(criteria, con).size());
	}
	
	private List<FileInfo> addTestFiles() throws Exception {
		FileData data1 = new FileData("test.file1", new byte[]{1}, ".");
		FileData data2 = new FileData("test.file2", new byte[]{1}, ".");
		FileData data3 = new FileData("test.file3", new byte[]{1}, ".");

		FileInfoBean file1 = manager.add(1, data1);
		FileInfoBean file2 = manager.add(1, data2);
		FileInfoBean file3 = manager.add(1, data3);
		
		Criteria criteria = new Criteria();
		criteria.addIn(FileInfoPeer.FILE_ID, new int[]{file1.getFileId(), file2.getFileId(), file3.getFileId()});
		
		List<FileInfo> fileList = FileInfoPeer.doSelect(criteria, con);
		// 前提条件として実ファイルができていることをテスト
		for (FileInfo file : fileList) {
			java.io.File sysFile = new java.io.File(file.getPath());
			assertTrue(sysFile.exists());
		}
		return fileList;
	}
	public void testDeleteAttachedFile_実ファイルが削除されていること() throws Exception {
		List<AttachedFile> fileList = getAttachedFile(addTestFiles());
		
		manager.deleteAttachedFile(fileList);

		for (AttachedFile file : fileList) {
			java.io.File sysFile = new java.io.File(file.getPath());
			assertFalse(sysFile.exists());
		}
	}
	public void testDeleteAttachedFile_TASK_FILE_RELATIONテーブルからレコードが削除されていること() throws Exception {
		List<AttachedFile> fileList = getAttachedFile(addTestFiles());
		
		manager.deleteAttachedFile(fileList);

		List<Integer> idList = new ArrayList<Integer>();
		for (AttachedFile file : fileList) {
			idList.add(file.getFileId());
		}
		Criteria criteria = new Criteria();
		criteria.addIn(TaskFileRelationPeer.FILE_ID, idList);
		assertEquals(0, TaskFileRelationPeer.doSelect(criteria, con).size());
	}
	public void testDeleteAttachedFile_FILEテーブルからレコードが削除されていること() throws Exception {
		List<AttachedFile> fileList = getAttachedFile(addTestFiles());
		
		manager.deleteAttachedFile(fileList);

		List<Integer> idList = new ArrayList<Integer>();
		for (AttachedFile file : fileList) {
			idList.add(file.getFileId());
		}
		Criteria criteria = new Criteria();
		criteria.addIn(FileInfoPeer.FILE_ID, idList);
		assertEquals(0, FileInfoPeer.doSelect(criteria, con).size());
	}
	public void testDeleteAttachedFile_レコードが存在しない場合は何もしない() throws Exception {
		List<AttachedFile> fileList = new ArrayList<AttachedFile>();
		AttachedFile file = new AttachedFile();
		file.setFileId(0);
		file.setPath("hogehoge");
		fileList.add(file);
		fileList.add(file);
		fileList.add(file);
		
		manager.deleteAttachedFile(fileList);
	}
	public void testDeleteAttachedFile_実ファイルが存在しない場合はレコードだけ削除する() throws Exception {
		List<AttachedFile> fileList = getAttachedFile(addTestFiles());
		for (AttachedFile file : fileList) {
			java.io.File sysFile = new java.io.File(file.getPath());
			sysFile.delete();
		}
		
		manager.deleteAttachedFile(fileList);

		List<Integer> idList = new ArrayList<Integer>();
		for (AttachedFile file : fileList) {
			idList.add(file.getFileId());
		}
		Criteria criteria = new Criteria();
		criteria.addIn(FileInfoPeer.FILE_ID, idList);
		assertEquals(0, FileInfoPeer.doSelect(criteria, con).size());
	}
	
	private List<AttachedFile> getAttachedFile(List<FileInfo> fileList) throws Exception {
		List<Integer> idList = new ArrayList<Integer>();
		for (FileInfo file : fileList) {
			idList.add(file.getFileId());
		}
		Criteria criteria = new Criteria();
		criteria.addIn(AttachedFilePeer.FILE_ID, idList);
		
		return AttachedFilePeer.doSelect(criteria, con);
	}
}
