/**
 * Copyright (c) 2006, yher2.net
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.workstyle.manager;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import net.yher2.workstyle.exception.NotFoundException;
import net.yher2.workstyle.torque.TagPeer;
import net.yher2.workstyle.torque.TagType;
import net.yher2.workstyle.torque.TagTypePeer;
import net.yher2.workstyle.torque.bean.TagTypeBean;

import org.apache.torque.NoRowsException;
import org.apache.torque.TorqueException;
import org.apache.torque.util.Criteria;

public class TagTypeManager {
	private Connection con = null;
	
	public TagTypeManager(Connection con) {
		this.con = con;
	}
	
	public List<TagTypeBean> list() throws TorqueException {
		Criteria criteria = new Criteria();
		criteria.add(TagTypePeer.TAG_TYPE_ID, TagType.DEFAULT_ID, Criteria.NOT_EQUAL);
		criteria.addAscendingOrderByColumn(TagTypePeer.SORT_ORDER);
		criteria.addAscendingOrderByColumn(TagTypePeer.NAME);
		List<TagType> dbList = TagTypePeer.doSelect(criteria, con);
		List<TagTypeBean> beanList = new ArrayList<TagTypeBean>();
		for (TagType tagType : dbList) {
			beanList.add(tagType.getBean());
		}
		return beanList;
	}
	protected TagType getDBModel(int tagTypeId) throws TorqueException, NotFoundException {
		try {
			return TagTypePeer.retrieveByPK(tagTypeId, con);
		} catch (NoRowsException e) {
			throw new NotFoundException("指定のタグ種別("+tagTypeId+")");
		}
	}
	
	public TagTypeBean add(TagTypeBean type) throws TorqueException {
		updateSortOrder(type.getSortOrder());
		TagType dbType = new TagType();
		dbType.setName(type.getName());
		dbType.setSortOrder(type.getSortOrder());
		dbType.setStyleClass(TagType.DEFAULT_STYLE);
		TagTypePeer.doInsert(dbType, con);
		return dbType.getBean();
	}
	public TagTypeBean update(TagTypeBean type) throws NotFoundException, TorqueException {
		updateSortOrder(type.getSortOrder());
		TagType dbType = getDBModel(type.getTagTypeId());
		dbType.setName(type.getName());
		dbType.setSortOrder(type.getSortOrder());
		TagTypePeer.doUpdate(dbType, con);
		return dbType.getBean();
	}
	
	private void updateSortOrder(int order) throws TorqueException {
		Criteria criteria = new Criteria();
		criteria.add(TagTypePeer.SORT_ORDER, order, Criteria.GREATER_EQUAL);
		criteria.addAscendingOrderByColumn(TagTypePeer.SORT_ORDER);
		criteria.addAscendingOrderByColumn(TagTypePeer.NAME);
		List<TagType> list = TagTypePeer.doSelect(criteria, con);
		for (TagType type : list) {
			order++;
			type.setSortOrder(order);
			TagTypePeer.doUpdate(type, con);
		}
	}
	public TagTypeBean updateStyle(int typeId, String style) throws NotFoundException, TorqueException {
		TagType dbType = getDBModel(typeId);
		dbType.setStyleClass(style);
		TagTypePeer.doUpdate(dbType, con);
		return dbType.getBean();
	}
	public void delete(int typeId) throws NotFoundException, TorqueException {
		TagType dbType = getDBModel(typeId);
		
		Criteria criteria = new Criteria();
		criteria.add(TagPeer.TAG_TYPE_ID, typeId);
		Criteria updateValues = new Criteria();
		updateValues.add(TagPeer.TAG_TYPE_ID, TagType.DEFAULT_ID);
		TagPeer.doUpdate(criteria, updateValues, con);
		
		TagTypePeer.doDelete(dbType.getPrimaryKey(), con);
	}
}
