/**
 * Copyright (c) 2006, yher2.net
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.commons.velocity;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.Date;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

import net.yher2.commons.lang.DateUtils;
import net.yher2.commons.lang.NumberUtils;

import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.CustomStringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.velocity.tools.view.tools.ViewTool;

public class VelocityTool implements ViewTool{
	
	public void init(Object initObject) {
	}

	public String formatDate(Date date) {
		return DateUtils.formatDate(date);
	}
	
	public String formatTimestamp(Date date) {
		return DateUtils.formatTimestamp(date);
	}
	public String formatDateRss(Date date) {
		return DateUtils.formatRss(date);
	}
	public String formatDateRfc1123(Date date) {
		return DateUtils.formatRfc1123(date);
	}
	/**
	 * 条件がTRUEなら1つ目の値、FALSEなら2つ目の値を返す。
	 * @param condition 条件
	 * @param trueValue TRUEの場合の値
	 * @param falseValue FALSEの場合の値
	 */
	public String decode(boolean condition, String trueValue, String falseValue) {
		return condition ? trueValue : falseValue;
	}
	/**
	 * 2つの値を比較して、同一ならtrueValue、異なったらfalseValueを返す。
	 * @param src 比較元
	 * @param dest 比較対象
	 * @param trueValue TRUEの場合の値
	 * @param falseValue FALSEの場合の値
	 */
	public String decode(Object src, Object dest, String trueValue, String falseValue) {
		return decode(ObjectUtils.equals(src, dest), trueValue, falseValue);
	}
	
	/**
	 * inputタグなどのchecked属性を取得する。
	 * @param condition
	 */
	public String checked(boolean condition) {
		if (condition) return "checked=\"checked\"";
		return "";
	}
	/**
	 * inputタグなどのchecked属性を取得する。
	 * @param src 検査対象値
	 * @param dest 現在選択値
	 */
	public String checked(int src, Object dest) {
		return checked(Integer.toString(src), dest);
	}
	/**
	 * inputタグなどのchecked属性を取得する。
	 * @param src 検査対象値
	 * @param dest 現在選択値
	 */
	public String checked(int src, int dest) {
		return checked(Integer.toString(src), Integer.toString(dest));
	}
	/**
	 * inputタグなどのchecked属性を取得する。
	 * @param src 検査対象値
	 * @param dest 現在選択値
	 */
	public String checked(Object src, Object dest) {
		if (src == null || dest == null) return "";
		if (src.equals(dest)) return "checked=\"checked\"";
		return "";
	}
	
	/**
	 * selectタグなどのchecked属性を取得する。
	 * @param src 検査対象値
	 * @param dest 現在選択値
	 */
	public String selected(int src, int dest) {
		return selected(Integer.toString(src), Integer.toString(dest));
	}
	/**
	 * selectタグなどのchecked属性を取得する。
	 * @param src 検査対象値
	 * @param dest 現在選択値
	 */
	public String selected(int src, String dest) {
		return selected(Integer.toString(src), dest);
	}
	/**
	 * selectタグなどのchecked属性を取得する。
	 * @param src 検査対象値
	 * @param dest 現在選択値
	 */
	public String selected(int src, Object dest) {
		return selected(Integer.toString(src), dest);
	}
	/**
	 * selectタグなどのchecked属性を取得する。
	 * @param src 検査対象値
	 * @param dest 現在選択値
	 */
	public String selected(Object src, Object dest) {
		if (src == null || dest == null) return "";
		if (src.equals(dest)) return "selected=\"selected\"";
		return "";
	}

	/**
	 * 文章全体の折り返しを行う。
	 * @param text 折り返し対象文章
	 * @param newline 改行コード
	 * @param wrapColumn 一行の長さ
	 * @return String
	 */
	public String wrapText(String text, String newline, int wrapColumn) {
		StringTokenizer lineTokenizer = new StringTokenizer(text, newline, true);
		StringBuffer buf = new StringBuffer();
		
		while (lineTokenizer.hasMoreTokens ()) {
			try {
				String nextLine = lineTokenizer.nextToken();

				if (nextLine.length() > wrapColumn) {
					nextLine = wrapLine(nextLine, newline, wrapColumn);
				}
				buf.append(nextLine);
			} catch (NoSuchElementException e) {
				// thrown by nextToken(), but I don't know why it would
				break;
			}
		}
		return buf.toString();
	}

	/**
	 * 行の折り返しを行う。
	 * @param line 折り返し対象行
	 * @param newline 改行コード
	 * @param wrapColumn 一行の長さ
	 * @return String
	 */
	protected String wrapLine(String line, String newline, int wrapColumn) {
		StringBuffer wrappedLine = new StringBuffer();

		while (line.length() > wrapColumn) {
			wrappedLine.append(line.substring (0, wrapColumn));
			wrappedLine.append(newline);
			line = line.substring(wrapColumn + 1);
		}
		wrappedLine.append(line);

		return wrappedLine.toString();
	}
	
	/**
	 * 複数行テーブルの奇数行と偶数行での表示変更用。<br>
	 * 奇数行の場合はarg1を返し、偶数行の場合はarg2を返す。
	 * @param index 行番号
	 * @param line1 奇数行用の表現
	 * @param line2 偶数業容の表現
	 * @return String
	 */
	public String multiRow(int index, String line1, String line2) {
		if (index % 2 == 1) return line1;
		return line2;
	}
	
	public String formatNumber(String number) {
		try {
			BigDecimal decimalNumber = new BigDecimal(number.replaceAll(",", ""));
			if (decimalNumber.scale() == 0) {
				return formatNumber(decimalNumber.longValue());
			} else {
				return formatNumber(decimalNumber.doubleValue());
			}
		} catch (Exception e) {
			return number;
		}
	}
	public String formatNumber(int number) {
		return NumberFormat.getInstance().format(number);
	}
	public String formatNumber(long number) {
		return NumberFormat.getInstance().format(number);
	}
	public String formatNumber(double number) {
		DecimalFormat format = new DecimalFormat("#,##0.####");
		return format.format(number);
	}
	/**
	 * 「,」区切りをせず、フォームの値として使える形で整形
	 * @param number
	 * @return String
	 */
	public String formatValue(String number) {
		try {
			return formatValue(NumberUtils.parseDouble(number));
		} catch (NumberFormatException e) {
			return number;
		}
	}
	/**
	 * 「,」区切りをせず、フォームの値として使える形で整形
	 * @param number
	 * @return String
	 */
	public String formatValue(double number) {
		DecimalFormat format = new DecimalFormat("##0.####");
		return format.format(number);
	}

	public String formatCurrencyValue(double number) {
		DecimalFormat format = new DecimalFormat("#,##0.####");
		return format.format(number);
	}
	
	public String formatPercentValue(double number) {
		DecimalFormat format = new DecimalFormat("#0.##");
		return format.format(number);
	}
	public String formatHtmlSimple(String str) {
		if (StringUtils.isBlank(str)) return str;
		String urlRegExp = "(s?https?://[-_.!~*'()a-zA-Z0-9;/?:@&=+$,%#]+)";
		str = str.replaceAll(urlRegExp, "\1a href=\3$0\3\2$0\1/a\2");
		str = CustomStringEscapeUtils.escapeXml(str);
		str = str.replaceAll("\1", "<");
		str = str.replaceAll("\2", ">");
		str = str.replaceAll("\3", "\"");
		return str.replaceAll("\n", "<br/>");
	}
	public String formatHtmlPlain(String str) {
		if (StringUtils.isBlank(str)) return str;
		str = CustomStringEscapeUtils.escapeXml(str);
		return str.replaceAll("\n", "<br/>");
	}
	public String formatHtmlFull(String str) {
		return str;
	}
	public String raw(String str) {
		return str;
	}
	public String getStackTrace(Throwable e) {
	    return ExceptionUtils.getFullStackTrace(e);
	}
	public int size(Collection array) {
		if (array == null) return 0;
		return array.size();
	}

	public int size(Object[] array) {
		if (array == null) return 0;
		return array.length;
	}

	public String cut(String text, int length) {
		if (StringUtils.isBlank(text)) return "";
		if (text.length() <= length) return text;
		if (length < 0) return text;
		return text.substring(0, length) + "...";
	}
}
