/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.sql.DataSource;

import woolpack.el.EL;
import woolpack.el.ELUtils;
import woolpack.el.GettingEL;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * RDB にアクセスする部品のユーティリティです。
 * 
 * @author nakamura
 *
 */
public class SqlFnUtils {
	
	/**
	 * {@link Statement#getUpdateCount()}で更新された数を返す関数です。
	 */
	public static final Fn<Statement, Integer, SQLException> GET_COUNT = new Fn<Statement, Integer, SQLException>() {
		public Integer exec(final Statement c) throws SQLException {
			return c.getUpdateCount();
		}
	};
	
	/**
	 * 現在の行の最初の列の値を返す関数です。
	 */
	public static final Fn<ResultSet, Object, SQLException> GET_SINGLE = new Fn<ResultSet, Object, SQLException>() {
		public Object exec(final ResultSet c) throws SQLException {
			return c.getObject(1);
		}
	};
	
	private SqlFnUtils() {
	}
	
	/**
	 * 現在の行をBeanにコピーして返す関数を生成します。
	 * 値の設定に{@link EL}を使用するため、{@link java.util.Map}の入力にも対応しています。
	 * @param <R>
	 * @param beanFactory Beanのファクトリ。
	 * @return 関数。
	 * @see #getBeanResult(Fn, Fn)
	 */
	public static <R> Fn<ResultSet, R, Exception> getBeanResult(final Fn<? super ResultSet, ? extends R, ? extends Exception> beanFactory) {
		return getBeanResult(beanFactory, ELUtils.EL_FACTORY);
	}
	
	/**
	 * 現在の行をBeanにコピーして返す関数を生成します。
	 * 値の設定に{@link EL}を使用するため、{@link java.util.Map}の入力にも対応しています。
	 * <br/>適用しているデザインパターン：Abstract Factoryにオブジェクトの生成を委譲する。
	 * @param <R>
	 * @param beanFactory Beanのファクトリ。
	 * @param elFactory プロパティ名から{@link EL}を生成するファクトリ。
	 * @return 関数。
	 */
	public static <R> Fn<ResultSet, R, Exception> getBeanResult(
			final Fn<? super ResultSet, ? extends R, ? extends Exception> beanFactory,
			final Fn<String, EL, ? extends Exception> elFactory) {
		return new Fn<ResultSet, R, Exception>() {
			public R exec(final ResultSet c) throws Exception{
				final R r = beanFactory.exec(c);
				final ResultSetMetaData metaData = c.getMetaData();
				final int length = metaData.getColumnCount();
				for (int i = 0; i < length; i++) {
					elFactory.exec(metaData.getColumnName(i + 1)).setValue(r, c.getObject(i + 1));
				}
				return r;
			}
		};
	}

	/**
	 * {@link Statement#getResultSet()}の{@link ResultSet#next()}でカーソルを移動しながら
	 * 委譲先にレコード情報の生成を委譲して、
	 * その結果を{@link List}に格納して返す関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <R>
	 * @param rowFn 委譲先。
	 * @param max 最大この数まで{@link ResultSet#next()}を呼び出す閾値。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 */
	public static <R> Fn<Statement, List<R>, Exception> getList(
			final Fn<? super ResultSet, ? extends R, ? extends Exception> rowFn,
			final int max,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return new Fn<Statement, List<R>, Exception>() {
			public List<R> exec(final Statement c) throws Exception {
				SQLException e0 = null;
				try {
					final ResultSet resultSet = c.getResultSet();
					try {
						final List<R> list = new ArrayList<R>();
						int i = 0;
						while (resultSet.next()) {
							i++;
							if (i > max) {
								break;
							}
							list.add(rowFn.exec(resultSet));
						}
						return list;
					} catch (final SQLException e1) {
						errorFn.exec(e1);
						e0 = e1;
					} finally {
						resultSet.close();
					}
				} catch (final SQLException e1) {
					errorFn.exec(e1);
					if (e0 == null) {
						e0 = e1;
					}
				}
				// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
				if (e0 != null) {
					throw e0;
				} else {
					return null;
				}
			}
		};
	}

	/**
	 * {@link Statement#getResultSet()}の{@link ResultSet#next()}でカーソルを移動しながら
	 * 委譲先にレコード情報の生成を委譲して、
	 * その結果を{@link List}に格納して返す関数を生成します。
	 * 最大{@link Integer#MAX_VALUE}回{@link ResultSet#next()}を呼び出します。
	 * @param <R>
	 * @param fn 委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 * @see #getList(Fn, int, Fn)
	 */
	public static <R> Fn<Statement, List<R>, Exception> getList(
			final Fn<? super ResultSet, ? extends R, ? extends Exception> fn,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return getList(fn, Integer.MAX_VALUE, errorFn);
	}

	/**
	 * 委譲先から{@link PreparedStatementInfo}を取得し、引数を{@link PreparedStatement}に設定して
	 * {@link PreparedStatement#execute()}を実行し、返却値の生成を委譲する関数を生成します。
	 * 値の設定に{@link GettingEL}を使用するため、{@link java.util.Map}の入力にも対応しています。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param dataSource データソース。
	 * @param queryFactory クエリ情報生成の委譲先。
	 * @param converter 変換の委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @param elFactory 引数から値を取得するための{@link GettingEL}のファクトリ。
	 * @return 関数。
	 */
	public static <C, R> Fn<C, R, Exception> inputBean(
			final DataSource dataSource,
			final Fn<? super C, ? extends PreparedStatementInfo, ? extends Exception> queryFactory,
			final Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn,
			final Fn<String, ? extends GettingEL, ? extends Exception> elFactory) {
		return new Fn<C, R, Exception>() {
			public R exec(final C c) throws Exception {
				SQLException e0 = null;
				try {
					final Connection connection = dataSource.getConnection();
					try {
						final PreparedStatementInfo info = queryFactory.exec(c);
						final PreparedStatement statement = connection.prepareStatement(info.getQuery());
						try {
							for (int i = 0; i < info.getList().size(); i++) {
								final String propertyName = info.getList().get(i);
								// WebアプリケーションがMap<String, Object[]>であることの対応。
								final Iterable<?> iterable =
									(Iterable<?>) elFactory.exec(propertyName).getValue(c, Iterable.class);
								final Iterator<?> iterator = iterable.iterator();
								statement.setObject(i + 1, iterator.hasNext() ? iterator.next() : null);
							}
							statement.execute();
							return converter.exec(statement);
						} catch (final SQLException e1) {
							errorFn.exec(e1);
							e0 = e1;
						} finally {
							statement.close();
						}
					} catch (final SQLException e1) {
						errorFn.exec(e1);
						if (e0 == null) {
							e0 = e1;
						}
					} finally {
						connection.close();
					}
				} catch (final SQLException e1) {
					errorFn.exec(e1);
					if (e0 == null) {
						e0 = e1;
					}
				}
				// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
				if (e0 != null) {
					throw e0;
				} else {
					return null;
				}
			}
		};
	}

	/**
	 * 委譲先から{@link PreparedStatementInfo}を取得し、引数を{@link PreparedStatement}に設定して
	 * {@link PreparedStatement#execute()}を実行し、返却値の生成を委譲する関数を生成します。
	 * 値の設定に{@link GettingEL}を使用するため、{@link java.util.Map}の入力にも対応しています。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param dataSource データソース。
	 * @param queryFactory クエリ情報生成の委譲先。
	 * @param converter 変換の委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 */
	public static <C, R> Fn<C, R, Exception> inputBean(
			final DataSource dataSource,
			final Fn<? super C, ? extends PreparedStatementInfo, ? extends Exception> queryFactory,
			final Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return inputBean(dataSource, queryFactory, converter, errorFn, ELUtils.EL_FACTORY);
	}

	/**
	 * 委譲先から{@link PreparedStatementInfo}を取得し、引数を{@link PreparedStatement}に設定して
	 * {@link PreparedStatement#execute()}を実行し、返却値の生成を委譲する関数を生成します。
	 * 値の設定に{@link GettingEL}を使用するため、{@link java.util.Map}の入力にも対応しています。
	 * @param <C>
	 * @param <R>
	 * @param dataSource データソース。
	 * @param info クエリ情報。
	 * @param converter 変換の委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 * @see #inputBean(DataSource, Fn, Fn, Fn)
	 */
	public static <C, R> Fn<C, R, Exception> inputBean(
			final DataSource dataSource,
			final PreparedStatementInfo info,
			final Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return inputBean(dataSource, FnUtils.fix(info), converter, errorFn);
	}

	/**
	 * 検索結果を一行だけ取得して返す関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <R>
	 * @param rowFn 一行を返却値に変換する委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 */
	public static <R> Fn<Statement, R, Exception> getOne(
			final Fn<? super ResultSet, ? extends R, ? extends Exception> rowFn,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return new Fn<Statement, R, Exception>() {
			public R exec(final Statement c) throws Exception {
				SQLException e0 = null;
				try {
					final ResultSet resultSet = c.getResultSet();
					try {
						if (resultSet.next()) {
							return rowFn.exec(resultSet);
						} else {
							return null;
						}
					} catch (final SQLException e1) {
						errorFn.exec(e1);
						e0 = e1;
					} finally {
						resultSet.close();
					}
				} catch (final SQLException e1) {
					errorFn.exec(e1);
					if (e0 == null) {
						e0 = e1;
					}
				}
				// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
				if (e0 != null) {
					throw e0;
				} else {
					return null;
				}
			}
		};
	}

	/**
	 * 委譲先からクエリを取得し、
	 * 引数を{@link PreparedStatement}に設定して
	 * {@link PreparedStatement#execute()}を実行し、
	 * 返却値の生成を委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param dataSource データソース。
	 * @param queryFactory クエリ生成の委譲先。
	 * @param converter 変換の委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 */
	public static <C, R> Fn<C, R, Exception> inputSingle(
			final DataSource dataSource,
			final Fn<? super C, ? extends String, ? extends Exception> queryFactory,
			final Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return new Fn<C, R, Exception>() {
			public R exec(final C c) throws Exception {
				SQLException e0 = null;
				try {
					final Connection connection = dataSource.getConnection();
					try {
						final PreparedStatement statement = connection.prepareStatement(queryFactory.exec(c));
						try {
							statement.setObject(1, c);
							statement.execute();
							return converter.exec(statement);
						} catch (final SQLException e1) {
							errorFn.exec(e1);
							e0 = e1;
						} finally {
							statement.close();
						}
					} catch (final SQLException e1) {
						errorFn.exec(e1);
						if (e0 == null) {
							e0 = e1;
						}
					} finally {
						connection.close();
					}
				} catch (final SQLException e1) {
					errorFn.exec(e1);
					if (e0 == null) {
						e0 = e1;
					}
				}
				// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
				if (e0 != null) {
					throw e0;
				} else {
					return null;
				}
			}
		};
	}

	/**
	 * 委譲先からクエリを取得し、
	 * 引数を{@link PreparedStatement}に設定して
	 * {@link PreparedStatement#execute()}を実行し、
	 * 返却値の生成を委譲する関数を生成します。
	 * @param <C>
	 * @param <R>
	 * @param dataSource データソース。
	 * @param query クエリ。
	 * @param converter 変換の委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 * @see #inputSingle(DataSource, Fn, Fn, Fn)
	 */
	public static <C, R> Fn<C, R, Exception> inputSingle(
			final DataSource dataSource,
			final String query,
			final Fn<? super PreparedStatement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return inputSingle(dataSource, FnUtils.fix(query), converter, errorFn);
	}

	/**
	 * 委譲先からクエリを取得して
	 * {@link Statement#execute(String)}を実行し、
	 * 返却値の生成を委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param dataSource データソース。
	 * @param queryFactory クエリ生成の委譲先。
	 * @param converter 変換の委譲先。
	 * @param errorFn {@link SQLException}が発生した場合の委譲先。1回の{@link Fn#exec(Object)}呼び出しで複数回委譲する可能性があります。
	 * @return 関数。
	 */
	public static <C, R> Fn<C, R, Exception> inputStatement(
			final DataSource dataSource,
			final Fn<? super C, ? extends String, ? extends Exception> queryFactory,
			final Fn<? super Statement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return new Fn<C, R, Exception>() {
			public R exec(final C c) throws Exception {
				SQLException e0 = null;
				try {
					final Connection connection = dataSource.getConnection();
					try {
						final Statement statement = connection.createStatement();
						try {
							statement.execute(queryFactory.exec(c));
							return converter.exec(statement);
						} catch (final SQLException e1) {
							errorFn.exec(e1);
							e0 = e1;
						} finally {
							statement.close();
						}
					} catch (final SQLException e1) {
						errorFn.exec(e1);
						if (e0 == null) {
							e0 = e1;
						}
					} finally {
						connection.close();
					}
				} catch (final SQLException e1) {
					errorFn.exec(e1);
					if (e0 == null) {
						e0 = e1;
					}
				}
				// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
				if (e0 != null) {
					throw e0;
				} else {
					return null;
				}
			}
		};
	}

	/**
	 * 委譲先からクエリを取得して
	 * {@link Statement#execute(String)}を実行し、
	 * 返却値の生成を委譲する関数を生成します。
	 * @param <C>
	 * @param <R>
	 * @param dataSource データソース。
	 * @param query クエリ。
	 * @param converter 変換の委譲先。
	 * @return 関数。
	 * @see #inputStatement(DataSource, Fn, Fn, Fn)
	 */
	public static <C, R> Fn<C, R, Exception> inputStatement(
			final DataSource dataSource,
			final String query,
			final Fn<? super Statement, ? extends R, ? extends Exception> converter,
			final Fn<? super SQLException, ?, ? extends Exception> errorFn) {
		return inputStatement(dataSource, FnUtils.fix(query), converter, errorFn);
	}
}
