/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.idweb;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Pattern;

import junit.framework.TestCase;
import woolpack.bool.BoolUtils;
import woolpack.convert.ConvertUtils;
import woolpack.el.PathEL;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.id.IdUtils;
import woolpack.utils.Utils;
import woolpack.validator.ValidatorContext;
import woolpack.validator.ValidatorUtils;

public class IdWebUtilsTest extends TestCase {
	private Fn<IdWebContext, Void, RuntimeException> newInstance(final boolean dFlag, final boolean rFlag) {
		return IdWebUtils.validate(
				FnUtils.seq(Utils.<Fn<? super ValidatorContext, Boolean, ? extends RuntimeException>>
				list(ValidatorUtils.localKey("param0", ValidatorUtils.convertValue(FnUtils.join(
						ConvertUtils.TO_STRING,
						ConvertUtils.convertRegExp(Pattern.compile("(.+)"), "@$1@")
				))))
				.list(FnUtils.join(IdUtils.GET_ID, BoolUtils.checkRegExp(Pattern.compile("a\\d"))))
		),
		new Fn<IdWebContext, Void, RuntimeException>() {
			public Void exec(IdWebContext c) throws RuntimeException {
				c.getRequest().put("param0", c.getInput().get("param0"));
				c.getRequest().put("status", "true");
				return null;
			}
		},
		new Fn<IdWebContext, Void, RuntimeException>() {
			public Void exec(IdWebContext c) throws RuntimeException {
				c.getRequest().put("param0", c.getInput().get("param0"));
				c.getRequest().put("status", "false");
				return null;
			}
		},
		FnUtils.fix("hoge"),
		new PathEL("request.messageList"),
		dFlag,
		rFlag);
	}
	
	public void testValidateFalseFalse() {
		final Fn<IdWebContext, Void, RuntimeException> fn = newInstance(false, false);
		final IdWebContext context = new IdWebContext();
		context.setInput(new HashMap<String, List<Object>>());
		context.setRequest(new HashMap<String, Object>());

		context.getInput().clear();
		context.getRequest().clear();
		context.setId("a0");
		context.getInput().put("param0", Arrays.<Object>asList("value0"));
		fn.exec(context);
		assertEquals(Utils.<String, Object>
		map("param0", Arrays.asList("value0")), context.getInput());
		assertEquals(Utils.<String, Object>
		map("status", "true")
		.map("param0", Arrays.asList("value0")), context.getRequest());

		context.getInput().clear();
		context.getRequest().clear();
		context.setId("b0");
		context.getInput().put("param0", Arrays.<Object>asList("value0"));
		fn.exec(context);
		assertEquals(Utils.<String, Object>
		map("param0", Arrays.asList("value0")), context.getInput());
		assertEquals(Utils.<String, Object>
		map("status", "false")
		.map("param0", Arrays.asList("value0"))
		.map("messageList", "hoge"), context.getRequest());
	}
	
	public void testValidateFalseTrue() {
		final Fn<IdWebContext, Void, RuntimeException> fn = newInstance(false, true);
		final IdWebContext context = new IdWebContext();
		context.setInput(new HashMap<String, List<Object>>());
		context.setRequest(new HashMap<String, Object>());

		context.getInput().clear();
		context.getRequest().clear();
		context.setId("a0");
		context.getInput().put("param0", Arrays.<Object>asList("value0"));
		fn.exec(context);
		assertEquals(Utils.<String, Object>
		map("param0", Arrays.asList("@value0@")), context.getInput());
		assertEquals(Utils.<String, Object>
		map("status", "true")
		.map("param0", Arrays.asList("value0")), context.getRequest());

		context.getInput().clear();
		context.getRequest().clear();
		context.setId("b0");
		context.getInput().put("param0", Arrays.<Object>asList("value0"));
		fn.exec(context);
		assertEquals(Utils.<String, Object>
		map("param0", Arrays.asList("@value0@")), context.getInput());
		assertEquals(Utils.<String, Object>
		map("status", "false")
		.map("param0", Arrays.asList("value0"))
		.map("messageList", "hoge"), context.getRequest());
	}
	
	public void testValidateTrueTrue() {
		final Fn<IdWebContext, Void, RuntimeException> fn = newInstance(true, true);
		final IdWebContext context = new IdWebContext();
		context.setInput(new HashMap<String, List<Object>>());
		context.setRequest(new HashMap<String, Object>());

		context.getInput().clear();
		context.getRequest().clear();
		context.setId("a0");
		context.getInput().put("param0", Arrays.<Object>asList("value0"));
		fn.exec(context);
		assertEquals(Utils.<String, Object>
		map("param0", Arrays.asList("@value0@")), context.getInput());
		assertEquals(Utils.<String, Object>
		map("status", "true")
		.map("param0", Arrays.asList("@value0@")), context.getRequest());

		context.getInput().clear();
		context.getRequest().clear();
		context.setId("b0");
		context.getInput().put("param0", Arrays.<Object>asList("value0"));
		fn.exec(context);
		assertEquals(Utils.<String, Object>
		map("param0", Arrays.asList("@value0@")), context.getInput());
		assertEquals(Utils.<String, Object>
		map("status", "false")
		.map("param0", Arrays.asList("@value0@"))
		.map("messageList", "hoge"), context.getRequest());
	}
}
