/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.idweb;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import woolpack.el.EL;
import woolpack.fn.Fn;
import woolpack.id.IdContext;
import woolpack.validator.AddressedMessage;
import woolpack.validator.AddressedMessageCollector;
import woolpack.validator.ValidatorContext;
import woolpack.web.WebContext;

/**
 * 識別子と{@link WebContext}の両方の性質をもつコンテキストを操作するユーティリティです。
 * 
 * @author nakamura
 *
 */
public final class IdWebUtils {
	private IdWebUtils() {}
	
	/**
	 * {@link WebContext}と{@link ValidatorContext}のアダプタを生成します。
	 * 本設計では値検証結果を表示する定義と業務遷移の定義が分割されます。
	 * <br/>適用しているデザインパターン：{@link ValidatorContext}を操作する{@link Fn}のAdapter。
	 * @param <C>
	 * @param <E>
	 * @param validatorFn 値検証の委譲先。
	 * @param trueFn 値の検証結果が true の場合の委譲先。
	 * @param falseFn 値の検証結果が false の場合の委譲先。
	 * @param messageListConverter メッセージ情報一覧の変換器。
	 * @param messageListSetter 変換したメッセージ一覧の設定場所。
	 * @param convertDelegateFlag 値検証の委譲で変換が伴った場合、その結果を委譲先で参照するならtrue。 
	 * @param convertReturnFlag 値検証の委譲で変換が伴った場合、その結果を返却後で参照するならtrue。
	 * @return 関数。
	 */
	public static <C extends WebContext & IdContext, E extends Exception>
	Fn<C, Void, E> validate(
			final Fn<? super ValidatorContext, Boolean, ? extends E> validatorFn,
			final Fn<? super C, Void, ? extends E> trueFn,
			final Fn<? super C, Void, ? extends E> falseFn,
			final Fn<? super List<AddressedMessage>, ?, ? extends E> messageListConverter,
			final EL messageListSetter,
			final boolean convertDelegateFlag,
			final boolean convertReturnFlag) {
		return new Fn<C, Void, E>() {
			public Void exec(final C c) throws E {
				final ValidatorContext validatorContext = new ValidatorContext();
				final AddressedMessageCollector collector = new AddressedMessageCollector();
				validatorContext.setCollectable(collector);
				validatorContext.setId(c.getId());
				final Map<String, List<Object>> beforeMap = c.getInput();
				final Map<String, List<Object>> afterMap = new HashMap<String, List<Object>>();
				for (final Entry<String, List<Object>> entry : beforeMap.entrySet()) {
					afterMap.put(entry.getKey(), new ArrayList<Object>(entry.getValue()));
				}
				validatorContext.setInputMap(afterMap);
				try {
					final boolean result = validatorFn.exec(validatorContext);
					c.setInput(convertDelegateFlag ? afterMap : beforeMap);
					if (result) {
						trueFn.exec(c);
					} else {
						messageListSetter.setValue(c, messageListConverter.exec(collector.getList()));
						falseFn.exec(c);
					}
				} finally {
					c.setInput(convertReturnFlag ? afterMap : beforeMap);
				}
				return null;
			}
		};
	}
}
