/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.id;

import woolpack.fn.Fn;

/**
 * 識別子のコンテキストに関するユーティリティです。
 * 
 * @author nakamura
 *
 */
public final class IdUtils {
	
	/**
	 * {@link IdContext#getId()}を返す関数です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final Fn<IdContext, String, RuntimeException> GET_ID = getId();
	
	private IdUtils() {
	}
	
	/**
	 * {@link IdContext#getId()}を返す関数を生成します。
	 * <br/>適用しているデザインパターン：Accessor。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<IdContext, String, E> getId() {
		return new Fn<IdContext, String, E>() {
			public String exec(final IdContext c) {
				return c.getId();
			}
		};
	}
	
	/**
	 * id の変換を委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：参照透過と副作用のAdapter。
	 * @param <R>
	 * @param <E>
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <R, E extends Exception> Fn<IdContext, R, E> convertId(
			final Fn<? super String, String, ? extends E> fn) {
		return new Fn<IdContext, R, E>() {
			public R exec(final IdContext c) throws E {
				c.setId(fn.exec(c.getId()));
				return null;
			}
		};
	}
	
	/**
	 * 委譲先から復帰したときに識別子の値を呼び出し時の状態に初期化する{@link Fn}です。
	 * <br/>適用しているデザインパターン：副作用のSand Box。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <C extends IdContext, R, E extends Exception> Fn<C, R, E> localId(
			final Fn<? super C, ? extends R, ? extends E> fn) {
		return new Fn<C, R, E>() {
			public R exec(final C c) throws E {
				final String id = c.getId();
				try {
					return fn.exec(c);
				} finally {
					c.setId(id);
				}
			}
		};
	}
}
