/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.fn;

/**
 * 引数をひとつとり、返却値をひとつとる関数を表します。
 * 本インタフェースの実装では以下のデザインパターンを適用することができます。
 * 
 * <table border>
 * <tr><th>デザインパターン</th><th>適用方法</th></tr>
 * <tr><td>Interpreter</td><td>引数にコンテキスト役を割り当てる</td></tr>
 * <tr><td>Abstract Factory</td><td>実装で返却値を生成する</td></tr>
 * <tr><td>Composite</td><td>実装で本インタフェースの変数を宣言する</td></tr>
 * <tr><td>Proxy</td><td>委譲を制御する</td></tr>
 * <tr><td>Adapter</td><td>別のインタフェースを持つインタフェースやクラス、ライブラリに委譲する</td></tr>
 * <tr><td>Decorator</td><td>引数を変換して委譲し、返却値を変換して返す</td></tr>
 * <tr><td>Currying</td><td>実行前に確定している変数を実装のインスタンス変数に定義し、実行時に確定する変数をコンテキスト役として指定する</td></tr>
 * </table>
 * <br/>適用しているデザインパターン：Function。
 * 
 * @author nakamura
 *
 * @param <C> コンテキスト役。
 * @param <R> 返却値。
 * @param <E> 例外。
 */
public interface Fn<C, R, E extends Exception> {
	/**
	 * 関数を実行します。
	 * @param c コンテキスト役。
	 * @return 返却値。
	 * @throws E 例外。
	 */
	R exec(C c) throws E;
}
