/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.locale;

import java.util.Arrays;
import java.util.Locale;

import junit.framework.TestCase;
import woolpack.adapter.JXPUtils;
import woolpack.samples.SamplesAllTests;
import woolpack.test.TestDomContextBuilder;
import woolpack.utils.Utils;

public class LocaleServletTest extends TestCase {

	public void testNormalNull() throws Exception {
		final LocaleServlet servlet = new LocaleServlet();
		servlet.threadLocal.set(null);
		servlet.toNode.setFn(SamplesAllTests.getFileToNode("^(.*)$", "html/sample/locale/$1.html"));
		final TestDomContextBuilder builder = new TestDomContextBuilder(JXPUtils.ONE);
		{
			// 値検証通過
			builder.init();
			builder.get().setId("bean0_input");
			servlet.fn.exec(builder.get());
			
			builder.initXPath("//FORM");
			builder.getInput().get("cache").set(0, "1000");
			builder.getInput().put("address", Arrays.<Object>asList("1"));
			servlet.fn.exec(builder.get());
			assertTrue(builder.exists("//H4[text() = \'bean0_confirm_ja\']"));
			assertEquals(Utils
					.map("address", Arrays.asList("Naichi"))
					.map("cache", Arrays.asList("1000")),
					builder.selectEmbedded());
		}
	}

	public void testNormalJapanese() throws Exception {
		final LocaleServlet servlet = new LocaleServlet();
		servlet.threadLocal.set(Locale.JAPANESE);
		servlet.toNode.setFn(SamplesAllTests.getFileToNode("^(.*)$", "html/sample/locale/$1.html"));
		final TestDomContextBuilder builder = new TestDomContextBuilder(JXPUtils.ONE);
		{
			// 値検証通過
			builder.init();
			builder.get().setId("bean0_input");
			servlet.fn.exec(builder.get());
			
			builder.initXPath("//FORM");
			builder.getInput().get("cache").set(0, "100");
			builder.getInput().put("address", Arrays.<Object>asList("1"));
			servlet.fn.exec(builder.get());
			assertTrue(builder.exists("//H4[text() = \'bean0_confirm_ja\']"));
			assertEquals(Utils
					.map("address", Arrays.asList("Naichi"))
					.map("cache", Arrays.asList("100")),
					builder.selectEmbedded());
		}
		{
			// 値検証ブロック
			builder.init();
			builder.get().setId("bean0_input");
			servlet.fn.exec(builder.get());
			
			builder.initXPath("//FORM");
			builder.getInput().get("cache").set(0, "1000");
			builder.getInput().put("address", Arrays.<Object>asList("1"));
			servlet.fn.exec(builder.get());
			assertTrue(builder.exists("//P[@id=\'errorValidate\' and text() = \'cache must three degits(ja).\']"));
		}
	}

	public void testNormalEnglish() throws Exception {
		final LocaleServlet servlet = new LocaleServlet();
		servlet.threadLocal.set(Locale.ENGLISH);
		servlet.toNode.setFn(SamplesAllTests.getFileToNode("^(.*)$", "html/sample/locale/$1.html"));
		final TestDomContextBuilder builder = new TestDomContextBuilder(JXPUtils.ONE);
		{
			// 値検証通過
			builder.init();
			builder.get().setId("bean0_input");
			servlet.fn.exec(builder.get());
			
			builder.initXPath("//FORM");
			builder.getInput().get("cache").set(0, "10");
			builder.getInput().put("address", Arrays.<Object>asList("1"));
			servlet.fn.exec(builder.get());
			assertTrue(builder.exists("//H4[text() = \'bean0_confirm_en\']"));
			assertEquals(Utils
					.map("address", Arrays.asList("NAITCHI"))
					.map("cache", Arrays.asList("10")),
					builder.selectEmbedded());
		}
		{
			// 値検証ブロック
			builder.init();
			builder.get().setId("bean0_input");
			servlet.fn.exec(builder.get());
			
			builder.initXPath("//FORM");
			builder.getInput().get("cache").set(0, "100");
			builder.getInput().put("address", Arrays.<Object>asList("1"));
			servlet.fn.exec(builder.get());
			assertTrue(builder.exists("//P[@id=\'errorValidate\' and text() = \'cache must two degits(en).\']"));
		}
	}
}
