/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.web;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentMap;

import woolpack.container.ContainerContext;
import woolpack.container.ContainerUtils;
import woolpack.el.AbstractGettingEL;
import woolpack.el.GettingEL;
import woolpack.factory.FactoryUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.utils.Utils;

/**
 * 各スコープを操作するユーティリティです。
 * @author nakamura
 *
 */
public final class WebUtils {

	/**
	 * {@link WebContext}を基点とみなして
	 * {@link WebContext#getInput()}を返す関数です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final GettingEL INPUT_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getInput();
		}
	};

	/**
	 * {@link WebContext}を基点とみなして
	 * リクエストスコープにアクセスする式言語です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final GettingEL REQUEST_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getRequest();
		}
	};
	
	/**
	 * {@link WebContext}を基点とみなして
	 * セッションスコープにアクセスする式言語です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final GettingEL SESSION_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getSession();
		}
	};

	/**
	 * {@link WebContext}を基点とみなして
	 * アプリケーションスコープにアクセスする式言語です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final GettingEL APPLICATION_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getApplication();
		}
	};

	/**
	 * {@link WebContext}を基点とみなして
	 * {@link WebContext#getContainer()}を返す関数です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final GettingEL CONTAINER_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getContainer();
		}
	};
	
	/**
	 * セッションスコープをクリアする関数です。
	 */
	public static final Fn<WebContext, Void, RuntimeException> CLEAR_SESSION = new Fn<WebContext, Void, RuntimeException>() {
		public Void exec(final WebContext context) {
			context.getSession().clear();
			return null;
		}
	};
	
	private static <E extends Exception> Fn<ContainerContext<WebContext>, String, Exception> getKey() {
		return new Fn<ContainerContext<WebContext>, String, Exception>() {
			public String exec(final ContainerContext<WebContext> c) {
				return "woolpack.web.WebUtils." + c.getKey();
			}
		};
	}
	
	private WebUtils() {
	}
	
	/**
	 * 引数のキーが全て文字列型・値が全てオブジェクトの一覧とみなして変換します。
	 * サーブレットAPIの request.getParameterMap() を変換するために定義しています。
	 * 返却値を更新しても引数には影響しません。
	 * @param map 変換対象。
	 * @return 変換結果。
	 */
	public static Map<String, List<Object>> convert(final Map map) {
		final Map<String, List<Object>> map1 = new HashMap<String, List<Object>>();
		for (final Object entryObject : map.entrySet()) {
			final Entry entry = (Entry) entryObject;
			final Iterable<?> c = Utils.toIterable(entry.getValue());
			final List<Object> list = new ArrayList<Object>();
			for (final Object o : c) {
				list.add(o);
			}
			map1.put((String) entry.getKey(), list);
		}
		return map1;
	}
	
	/**
	 * 入力スコープを返す関数を生成します。
	 * <br/>適用しているデザインパターン：Accessor。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, Map<String, List<Object>>, E> inputFn() {
		return new Fn<WebContext, Map<String, List<Object>>, E>() {
			public Map<String, List<Object>> exec(final WebContext c) {
				return c.getInput();
			}
		};
	}
	
	/**
	 * リクエストスコープを返す関数を生成します。
	 * <br/>適用しているデザインパターン：Accessor。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, Map<String, Object>, E> requestFn() {
		return new Fn<WebContext, Map<String, Object>, E>() {
			public Map<String, Object> exec(final WebContext c) {
				return c.getRequest();
			}
		};
	}
	
	/**
	 * セッションスコープを返す関数を生成します。
	 * <br/>適用しているデザインパターン：Accessor。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, ConcurrentMap<String, Object>, E> sessionFn() {
		return new Fn<WebContext, ConcurrentMap<String, Object>, E>() {
			public ConcurrentMap<String, Object> exec(final WebContext c) {
				return c.getSession();
			}
		};
	}
	
	/**
	 * アプリケーションスコープを返す関数を生成します。
	 * <br/>適用しているデザインパターン：Accessor。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, ConcurrentMap<String, Object>, E> applicationFn() {
		return new Fn<WebContext, ConcurrentMap<String, Object>, E>() {
			public ConcurrentMap<String, Object> exec(final WebContext c) {
				return c.getApplication();
			}
		};
	}
	
	/**
	 * コンテナを返す関数を生成します。
	 * <br/>適用しているデザインパターン：Accessor。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, Object, E> containerFn() {
		return new Fn<WebContext, Object, E>() {
			public Object exec(final WebContext c) {
				return c.getContainer();
			}
		};
	}

	/**
	 * リクエストスコープでキャッシュする関数を生成します。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param fn オブジェクト生成の委譲先。
	 * @return 関数。
	 */
	public static <C extends ContainerContext<WebContext>>
	Fn<C, Object, Exception> request(
			final Fn<? super C, ?, ? extends Exception> fn) {
		return FactoryUtils.cache(
				FnUtils.join(
						ContainerUtils.<WebContext, C>getSubContext(),
						WebUtils.requestFn()),
				WebUtils.getKey(),
				fn);
	}
	
	/**
	 * セッションスコープでキャッシュする関数を生成します。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param fn オブジェクト生成の委譲先。
	 * @return 関数。
	 */
	public static <C extends ContainerContext<WebContext>>
	Fn<C, Object, Exception> session(
			final Fn<? super C, ?, ? extends Exception> fn) {
		return FactoryUtils.concurrentCache(
				FnUtils.join(
						ContainerUtils.<WebContext, C>getSubContext(),
						WebUtils.sessionFn()),
				WebUtils.getKey(),
				fn);
	}
	
	/**
	 * アプリケーションスコープでキャッシュする関数を生成します。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param fn オブジェクト生成の委譲先
	 * @return 関数。
	 */
	public static <C extends ContainerContext<WebContext>>
	Fn<C, Object, Exception> application(
			final Fn<? super C, ?, ? extends Exception> fn) {
		return FactoryUtils.concurrentCache(
				FnUtils.join(
						ContainerUtils.<WebContext, C>getSubContext(),
						WebUtils.applicationFn()),
				WebUtils.getKey(),
				fn);
	}
	
	/**
	 * {@link WebContext#setContainer(Object)}に{@link ContainerContext}を設定する関数を生成します。
	 * <br/>適用しているデザインパターン：Double Dispatchの関数をコンテキストに設定する。
	 * @param fn 設定する関数。
	 * @return 関数。
	 * @see ContainerContextSetter
	 */
	public static Fn<WebContext, Void, RuntimeException> setContainerContext(
			final Fn<? super ContainerContext<WebContext>, ?, ? extends Exception> fn) {
		return new ContainerContextSetter<RuntimeException>(fn);
	}
	
	/**
	 * トランザクショントークンを検証して登録/更新する関数を生成します。
	 * 画面遷移順序を検証するための機能です。
	 * 本設計ではトークンを検証する定義と業務遷移の定義が分割されます。
	 * <br/>適用しているデザインパターン：Proxy、{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param key トークンを格納する{@link WebContext#getSession()}上のキー。
	 * @param targetCheckFn 検証対象であるかを判定する委譲先。
	 * @param trueFn 妥当なトランザクションの場合の委譲先。
	 * @param falseFn 妥当でないトランザクションの場合の委譲先。
	 * @param size トークンのサイズ。
	 * @return 関数。
	 * @see TransactionTokenFn
	 */
	public static <C extends WebContext, R, E extends Exception> Fn<C, R, E> checkTransactionToken(
			final String key,
			final Fn<? super C, Boolean, ? extends E> targetCheckFn,
			final Fn<? super C, R, ? extends E> trueFn,
			final Fn<? super C, R, ? extends E> falseFn,
			final int size) {
		return new TransactionTokenFn<C, R, E>(
				key, targetCheckFn, trueFn, falseFn, size);
	}
	
	/**
	 * トランザクショントークンを検証して登録/更新する関数を生成します。
	 * トークンのサイズを32とします。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param key トークンを格納する{@link WebContext#getSession()}上のキー。
	 * @param targetCheckFn 検証対象であるかを判定する委譲先。
	 * @param trueFn 妥当なトランザクションの場合の委譲先。
	 * @param falseFn 妥当でないトランザクションの場合の委譲先。
	 * @return 関数。
	 * @see #checkTransactionToken(String, Fn, Fn, Fn, int)
	 */
	public static <C extends WebContext, R, E extends Exception> Fn<C, R, E> checkTransactionToken(
			final String key,
			final Fn<? super C, Boolean, ? extends E> targetCheckFn,
			final Fn<? super C, R, ? extends E> trueFn,
			final Fn<? super C, R, ? extends E> falseFn) {
		return checkTransactionToken(
				key, targetCheckFn, trueFn, falseFn, TransactionTokenFn.TOKEN_LENGTH);
	}
}
