/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.web;

import java.util.List;
import java.util.Random;

import woolpack.fn.Fn;
import woolpack.utils.Utils;

public class TransactionTokenFn<C extends WebContext, R, E extends Exception> implements Fn<C, R, E> {
	public static final int TOKEN_LENGTH = 32;
	private static final int DEGIT_COUNT = 10;
	private static final int TOKEN_CHAR_COUNT = 36;

	private int size;
	private String key;
	private Fn<? super C, Boolean, ? extends E> targetCheckFn;
	private Fn<? super C, R, ? extends E> trueFn;
	private Fn<? super C, R, ? extends E> falseFn;
	private Random r;
	
	public TransactionTokenFn(
			final String key,
			final Fn<? super C, Boolean, ? extends E> targetCheckFn,
			final Fn<? super C, R, ? extends E> trueFn,
			final Fn<? super C, R, ? extends E> falseFn,
			final int size) {
		super();
		this.key = key;
		this.targetCheckFn = targetCheckFn;
		this.trueFn = trueFn;
		this.falseFn = falseFn;
		this.size = size;
		this.r = new Random();
	}

	public R exec(final C c) throws E {
		if (targetCheckFn.exec(c)) {
			final Object oldToken = c.getSession().get(key);
			final List requestedTokens = Utils.toList(c.getInput().get(key));
			final Object newToken = generateToken();

			if (oldToken != null
					&& oldToken.equals(requestedTokens.get(0))
					&& c.getSession().replace(key, oldToken, newToken)) {
				trueFn.exec(c);
			} else {
				falseFn.exec(c);
			}
		} else {
			while (true) {
				final Object oldToken = c.getSession().get(key);
				final Object newToken = generateToken();
				if (oldToken == null) {
					if (c.getSession().putIfAbsent(key, newToken) == null) {
						break;
					}
				} else {
					if (c.getSession().replace(key, oldToken, newToken)) {
						break;
					}
				}
			}
			trueFn.exec(c);
		}
		return null;
	}

	String generateToken() {
		final StringBuilder sb = new StringBuilder();
		for (int i = 0; i < size; i++) {
			// jは正の数
			final int j = r.nextInt(TOKEN_CHAR_COUNT);
			sb.append((j < DEGIT_COUNT)
					? ((char) ('0' + j))
							: ((char) ('A' + j - DEGIT_COUNT)));
		}
		return sb.toString();
	}

	public Fn<? super C, R, ? extends E> getFalseFn() {
		return falseFn;
	}
	public void setFalseFn(final Fn<? super C, R, ? extends E> falseFn) {
		this.falseFn = falseFn;
	}
	public String getKey() {
		return key;
	}
	public void setKey(final String key) {
		this.key = key;
	}
	public int getSize() {
		return size;
	}
	public void setSize(final int size) {
		this.size = size;
	}
	public Fn<? super C, Boolean, ? extends E> getTargetCheckFn() {
		return targetCheckFn;
	}
	public void setTargetCheckFn(final Fn<? super C, Boolean, ? extends E> targetCheckFn) {
		this.targetCheckFn = targetCheckFn;
	}
	public Fn<? super C, R, ? extends E> getTrueFn() {
		return trueFn;
	}
	public void setTrueFn(final Fn<? super C, R, ? extends E> trueFn) {
		this.trueFn = trueFn;
	}
}
