/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.Iterator;

/**
 * プロパティ名に関するユーティリティです。
 * 
 * @author nakamura
 * 
 */
public final class PropertyUtils {

	private PropertyUtils() {
	}
	
	/**
	 * 属性名パターンの候補を順に返す{@link Iterable}を生成して返します。
	 * 候補を以下の例の順序で返します。ただし、引数と同一の値を省略します。
	 * <ol>
	 * <li>引数そのもの
	 * <li>HOGE_PIYO
	 * <li>hoge_piyo
	 * <li>Hoge_Piyo
	 * <li>hogePiyo
	 * <li>HogePiyo
	 * <li>hogepiyo
	 * <li>HOGEPIYO
	 * </ol>
	 * <br/>適用しているデザインパターン：Iterator。
	 * @param s 属性名。
	 * @param separator セパレータ。
	 * @return 属性名パターンの候補を順に返す{@link Iterable}。
	 */
	public static Iterable<String> getGuessingIterable(final String s, final char separator) {
		return new Iterable<String>() {
			public Iterator<String> iterator() {
				return new Iterator<String>() {
					private int i = 0;
					private String tmpString;
					public boolean hasNext() {
						return i <= 7;
					}
					public String next() {
						do {
							switch(i) {
							case 0:tmpString = s;break;
							case 1:tmpString = PropertyUtils.toSQLName(s, separator);break;
							case 2:tmpString = tmpString.toLowerCase();break;
							case 3:tmpString = sqlWordHeadToUpper(tmpString);break;
							case 4:tmpString = PropertyUtils.toJavaName(s, separator);break;
							case 5:tmpString = Character.toUpperCase(tmpString.charAt(0)) + tmpString.substring(1);break;
							case 6:tmpString = tmpString.toLowerCase();break;
							case 7:tmpString = tmpString.toUpperCase();break;
							default:tmpString = null;return null;
							}
							i++;
						} while (i != 1 && tmpString.equals(s));
						return tmpString;
					}
					public void remove() {
						throw new UnsupportedOperationException();
					}
				};
			}
		};
	}
	
	private static String sqlWordHeadToUpper(final String s) {
		final StringBuilder sb = new StringBuilder(s);
		boolean flag = false;
		for (int i = 0; i < sb.length(); i++) {
			final char c = sb.charAt(i);
			if (flag || i == 0) {
				sb.setCharAt(i, Character.toUpperCase(c));
				flag = false;
			}
			if (c == '_') {
				flag = true;
			}
		}
		return sb.toString();
	}

	public static Iterable<String> getGuessingIterable(final String s) {
		return getGuessingIterable(s, '_');
	}

	/**
	 * DB における名称を Java における名称(先頭小文字)に変換します。
	 * @param s 変換対象。
	 * @return 変換結果。
	 */
	public static String toJavaName(final String s) {
		return toJavaName(s, '_');
	}

	/**
	 * DB における名称を Java における名称(先頭小文字)に変換します。
	 * @param s 変換対象。
	 * @param separator セパレータ。
	 * @return 変換結果。
	 */
	public static String toJavaName(final String s, final char separator) {
		if (s == null || s.indexOf(separator) < 0) {
			if (s != null && s.length() >= 1) {
				return Character.toLowerCase(s.charAt(0)) + s.substring(1);
			}
			return s;
		}
		final StringBuilder sb = new StringBuilder(s);
		boolean separateFlag = false;
		int i = 0;
		while (i < sb.length()) {
			char c = sb.charAt(i);
			if (c == separator) {
				separateFlag = true;
				sb.delete(i, i + 1);
			} else {
				if (separateFlag) {
					c = Character.toUpperCase(c);
					separateFlag = false;
				} else {
					c = Character.toLowerCase(c);
				}
				sb.setCharAt(i, c);
				i++;
			}
		}
		return sb.toString();
	}

	/**
	 * java における名称を DB における名称(大文字)に変換します。
	 * @param s 変換対象。
	 * @return 変換結果。
	 */
	public static String toSQLName(final String s) {
		return toSQLName(s, '_');
	}

	/**
	 * java における名称を DB における名称(大文字)に変換します。
	 * @param s 変換対象。
	 * @param separator セパレータ。
	 * @return 変換結果。
	 */
	public static String toSQLName(final String s, final char separator) {
		if (s == null || s.length() == 0) {
			return s;
		}
		final StringBuilder sb = new StringBuilder();
		final int length = s.length();
		boolean upper = false;
		for (int i = length - 1; i >= 0; i--) {
			final char c = s.charAt(i);
			if (Character.isLowerCase(c) && upper) {
				sb.insert(0, separator);
			}
			upper = Character.isUpperCase(c);
			sb.insert(0, Character.toUpperCase(c));
		}
		return sb.toString();
	}
}
