/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.io.IOException;
import java.io.Writer;

/**
 * {@link Writer}から{@link Appendable}へのアダプタです。
 * {@link Writer}のみを引数として持つ機能に対して{@link Appendable}を作用させる際に使用します。
 * <br/>適用しているデザインパターン：{@link Appendable}のAdapter。
 * 
 * @author nakamura
 * 
 */
public class AppendableWriter extends Writer {
	private Appendable appendable;

	public AppendableWriter(final Appendable appendable) {
		super();
		this.appendable = appendable;
	}

	@Override
	public Writer append(final CharSequence csq) throws IOException {
		appendable.append(csq);
		return this;
	}

	@Override
	public Writer append(final CharSequence csq, final int start, final int end)
			throws IOException {
		appendable.append(csq, start, end);
		return this;
	}

	@Override
	public Writer append(final char c) throws IOException {
		appendable.append(c);
		return this;
	}

	@Override
	public void write(final char[] cbuf, final int off, final int len)
			throws IOException {
		for (int i = off; i < off + len; i++) {
			appendable.append(cbuf[i]);
		}
	}

	@Override
	public void write(final int c) throws IOException {
		appendable.append((char) c);
	}

	@Override
	public void write(final String str) throws IOException {
		appendable.append(str);
	}

	@Override
	public void write(final String str, final int off, final int len)
			throws IOException {
		appendable.append(str, off, off + len);
	}

	@Override
	public void flush() throws IOException {
		// nothing to do.
	}

	@Override
	public void close() throws IOException {
		// nothing to do.
	}

	public Appendable getAppendable() {
		return appendable;
	}

	public void setAppendable(final Appendable appendable) {
		this.appendable = appendable;
	}
}
