/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.test;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.ee.EEContext;
import woolpack.fn.Fn;
import woolpack.utils.Utils;

/**
 * テスト用の{@link EEContext}のビルダです。
 * 
 * @author nakamura
 * 
 */
public class TestDomContextBuilder {
	/**
	 * {@link #selectEmbedded()}内で DOM ノードを検索するための属性名一覧のデフォルト値です。
	 */
	public static final Iterable<String> ATTR_NAMES = Collections.singleton("id");

	private final Iterable<String> attrNames;
	private final Map<String, List<Object>> input;
	private final EEContext context;
	private final Fn<String, ? extends Fn<? super Node, ? extends Node, ? extends RuntimeException>, ? extends RuntimeException> factory;

	/**
	 * @param attrNames {@link #selectEmbedded()}内で DOM ノードを検索するための属性名一覧。
	 * @param factory ノードを検索する{@link Fn}のファクトリ。
	 */
	public TestDomContextBuilder(
			final Iterable<String> attrNames,
			final Fn<String, ? extends Fn<? super Node, ? extends Node, ? extends RuntimeException>, ? extends RuntimeException> factory) {

		this.attrNames = attrNames;
		this.factory = factory;

		input = new HashMap<String, List<Object>>();
		context = new EEContext();
		context.setConfig(new HashMap<String, Object>());
		context.setInput(input);
		context.setRequest(new HashMap<String, Object>());
		context.setSession(Utils.concurrentMap(
				new HashMap<String, Object>(), new Object()));
		context.setApplication(Utils.concurrentMap(
				new HashMap<String, Object>(), new Object()));
	}

	/**
	 * @param factory ノードを検索する{@link Fn}のファクトリ。
	 */
	public TestDomContextBuilder(final Fn<String, ? extends Fn<? super Node, ? extends Node, ? extends RuntimeException>, ? extends RuntimeException> factory) {
		this(ATTR_NAMES, factory);
	}

	/**
	 * {@link EEContext}を初期化します。
	 */
	public void init() {
		context.getInput().clear();
		context.getRequest().clear();
		context.setNode(null);
	}

	/**
	 * {@link EEContext}を初期化します。
	 * 引数を XPath として検索した結果が FORM エレメントまたは A エレメント(アンカー)の場合は
	 * キーと値を{@link EEContext#getInput()}に設定し、
	 * {@link EEContext#setId(String)}に action 属性値または href 属性値を設定します。
	 * 
	 * @param criteria ノードの検索条件。
	 */
	public void initXPath(final String criteria) {
		context.getInput().clear();
		final Node n = factory.exec(criteria).exec(context.getNode());
		if (n != null && n.getNodeType() == Node.ELEMENT_NODE) {
			final Element e = (Element) n;
			if (e.getNodeName().equals("FORM")) {
				input.putAll(TestUtils.selectForm(e));
				context.setId(e.getAttribute("action"));
			} else if (e.getNodeName().equals("A")) {
				final String href = e.getAttribute("href");
				input.putAll(TestUtils.selectQuery(href));
				context.setId(href.substring(0, href.indexOf('?') >= 0 ? href
						.indexOf('?') : href.length()));
			}
		}
		context.getRequest().clear();
		context.setNode(null);
	}

	/**
	 * {@link EEContext#getInput()}を返します。
	 * @return {@link EEContext#getInput()}。
	 */
	public Map<String, List<Object>> getInput() {
		return input;
	}

	/**
	 * {@link EEContext}を返します。
	 * @return {@link EEContext}。
	 */
	public EEContext get() {
		return context;
	}

	/**
	 * 引数で指定したノードが存在する場合は true、それ以外の場合は false を返します。
	 * @param criteria XPath。
	 * @return 引数で指定したノードが存在する場合は true。それ以外の場合は false。
	 */
	public boolean exists(final String criteria) {
		return selectNode(criteria) != null;
	}

	/**
	 * DOM ノードを検索します。
	 * @param criteria XPath。
	 * @return 検索されたDOM ノード。該当するノードが存在しない場合は null。
	 */
	public Node selectNode(final String criteria) {
		return factory.exec(criteria).exec(context.getNode());
	}

	/**
	 * 埋め込まれたキーと値の{@link Map}を返します。
	 * @return 埋め込まれたキーと値の{@link Map}。
	 */
	public Map<String, List<Object>> selectEmbedded() {
		return TestUtils.selectEmbedded(attrNames, context.getNode());
	}
}
