/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.adapter;

import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

/**
 * {@link Connection#setAutoCommit(boolean)}を設定する{@link DataSource}です。
 * <br/>適用しているデザインパターン：委譲先で取得したオブジェクトの状態を変化させるDecorator。
 * @author nakamura
 *
 */
public class AutoCommitSettingDataSource extends DataSourceAdapter {
	private boolean autoCommit;

	public AutoCommitSettingDataSource(final DataSource parent, final boolean autoCommit) {
		super(parent);
		this.autoCommit = autoCommit;
	}

	public Connection getConnection() throws SQLException {
		final Connection c = super.getConnection();
		setAutoCommitTo(c);
		return c;
	}
	public Connection getConnection(final String username, final String password) throws SQLException {
		final Connection c = super.getConnection(username, password);
		setAutoCommitTo(c);
		return c;
	}
	
	private void setAutoCommitTo(final Connection c) throws SQLException {
		try {
			c.setAutoCommit(autoCommit);
		} catch (final SQLException e) {
			try {
				c.close();
			} catch (final SQLException e2) {
				// nothing to do.
			}
			throw e;
		}
	}
}
