/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.misc;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Semaphore;

import woolpack.fn.Fn;

public class LoadBalancer<C, R, E extends Exception, E1 extends Exception> implements Fn<C, R, E> {
	private final Fn<Object, ? extends Fn<? super C, ? extends R, ? extends E>, ? extends E1> factory;
	private final int length;
	
	private final Semaphore[] semaphoreArray;
	private final List<Fn<? super C, ? extends R, ? extends E>> fnList;
	// position の原子性を厳密に制御する必要はない。
	private int position = 0;
	
	public LoadBalancer(
			final Fn<Object, ? extends Fn<? super C, ? extends R, ? extends E>, ? extends E1> factory,
			final int length) throws E1 {
		if (length < 1) {
			throw new IllegalArgumentException("length must be equals or more than 1 but " + length);
		}
		this.factory = factory;
		this.length = length;
		semaphoreArray = new Semaphore[length];
		fnList = new ArrayList<Fn<? super C, ? extends R, ? extends E>>(length);
		for (int i = 0; i < length; i++) {
			semaphoreArray[i] = new Semaphore(1);
			synchronized (factory) {
				fnList.add(factory.exec(null));
			}
		}
	}

	public R exec(final C c) throws E {
		int i = position;
		while (true) {
			for (; i < length; i++) {
				if (semaphoreArray[i].tryAcquire()) {
					try {
						return fnList.get(i).exec(c);
					} finally {
						semaphoreArray[i].release();
						position = (i + 1) % length;
					}
				}
				Thread.yield();
			}
			i = 0;
		}
	}

	public int getLength() {
		return length;
	}
	public Fn<Object, ? extends Fn<? super C, ? extends R, ? extends E>, ? extends E1> getFactory() {
		return factory;
	}
}
