/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.idweb;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import woolpack.el.EL;
import woolpack.fn.Fn;
import woolpack.id.IdContext;
import woolpack.validator.AddressedMessage;
import woolpack.validator.AddressedMessageCollector;
import woolpack.validator.ValidatorContext;
import woolpack.web.WebContext;

public class ValidatorFn<C extends WebContext & IdContext, E extends Exception> implements Fn<C, Void, E> {
	private Fn<? super ValidatorContext, Boolean, ? extends E> validatorFn;
	private Fn<? super C, Void, ? extends E> trueFn;
	private Fn<? super C, Void, ? extends E> falseFn;
	private Fn<? super List<AddressedMessage>, ?, ? extends E> messageListConverter;
	private EL messageListSetter;
	private boolean convertDelegateFlag;
	private boolean convertReturnFlag;
	
	public ValidatorFn(
			final Fn<? super ValidatorContext, Boolean, ? extends E> validatorFn,
			final Fn<? super C, Void, ? extends E> trueFn,
			final Fn<? super C, Void, ? extends E> falseFn,
			final Fn<? super List<AddressedMessage>, ?, ? extends E> messageListConverter,
			final EL messageListSetter,
			final boolean convertDelegateFlag,
			final boolean convertReturnFlag) {
		super();
		this.validatorFn = validatorFn;
		this.trueFn = trueFn;
		this.falseFn = falseFn;
		this.messageListConverter = messageListConverter;
		this.messageListSetter = messageListSetter;
		this.convertDelegateFlag = convertDelegateFlag;
		this.convertReturnFlag = convertReturnFlag;
	}

	public Void exec(final C c) throws E {
		final ValidatorContext validatorContext = new ValidatorContext();
		final AddressedMessageCollector collector = new AddressedMessageCollector();
		validatorContext.setCollectable(collector);
		validatorContext.setId(c.getId());
		final Map<String, List<Object>> beforeMap = c.getInput();
		final Map<String, List<Object>> afterMap = new HashMap<String, List<Object>>();
		for (final Entry<String, List<Object>> entry : beforeMap.entrySet()) {
			afterMap.put(entry.getKey(), new ArrayList<Object>(entry.getValue()));
		}
		validatorContext.setInputMap(afterMap);
		try {
			final boolean result = validatorFn.exec(validatorContext);
			c.setInput(convertDelegateFlag ? afterMap : beforeMap);
			if (result) {
				trueFn.exec(c);
			} else {
				messageListSetter.setValue(c, messageListConverter.exec(collector.getList()));
				falseFn.exec(c);
			}
		} finally {
			c.setInput(convertReturnFlag ? afterMap : beforeMap);
		}
		return null;
	}

	public boolean isConvertDelegateFlag() {
		return convertDelegateFlag;
	}
	public void setConvertDelegateFlag(final boolean convertDelegateFlag) {
		this.convertDelegateFlag = convertDelegateFlag;
	}
	public boolean isConvertReturnFlag() {
		return convertReturnFlag;
	}
	public void setConvertReturnFlag(final boolean convertReturnFlag) {
		this.convertReturnFlag = convertReturnFlag;
	}
	public Fn<? super C, Void, ? extends E> getFalseFn() {
		return falseFn;
	}
	public void setFalseFn(final Fn<? super C, Void, ? extends E> falseFn) {
		this.falseFn = falseFn;
	}
	public Fn<? super List<AddressedMessage>, ?, ? extends E> getMessageListConverter() {
		return messageListConverter;
	}
	public void setMessageListConverter(
			final Fn<? super List<AddressedMessage>, ?, ? extends E> messageListConverter) {
		this.messageListConverter = messageListConverter;
	}
	public EL getMessageListSetter() {
		return messageListSetter;
	}
	public void setMessageListSetter(final EL messageListSetter) {
		this.messageListSetter = messageListSetter;
	}
	public Fn<? super C, Void, ? extends E> getTrueFn() {
		return trueFn;
	}
	public void setTrueFn(final Fn<? super C, Void, ? extends E> trueFn) {
		this.trueFn = trueFn;
	}
	public Fn<? super ValidatorContext, Boolean, ? extends E> getValidatorFn() {
		return validatorFn;
	}
	public void setValidatorFn(
			final Fn<? super ValidatorContext, Boolean, ? extends E> validatorFn) {
		this.validatorFn = validatorFn;
	}
}
