/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.factory;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.fn.Fn;

/**
 * ファクトリのユーティリティです。
 * @author nakamura
 *
 */
public final class FactoryUtils {
	/**
	 * {@link LinkedHashMap}を生成する関数です。
	 * <br/>適用しているデザインパターン：{@link Map}のAbstract Factory。
	 */
	public static final Fn<Object, Map<String, Object>, RuntimeException> LINKED_MAP_FACTORY = new Fn<Object, Map<String, Object>, RuntimeException>() {
		public Map<String, Object> exec(final Object c) {
			return new LinkedHashMap<String, Object>();
		}
	};

	private FactoryUtils() {
	}
	
	/**
	 * 委譲先で生成したオブジェクトを{@link Map}にキャッシュして返す関数を生成します。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param <K>
	 * @param <E>
	 * @param mapGetter キャッシュ用マップを取得する委譲先。
	 * @param keyGetter キャッシュに格納する際のキーを取得する委譲先。
	 * @param maker キャッシュするファイルを生成するファクトリ。
	 * @return 関数。
	 * @see MapCache
	 */
	public static <C, K, E extends Exception> Fn<C, Object, E> cache(
			final Fn<? super C, ? extends Map<? super K, Object>, ? extends E> mapGetter,
			final Fn<? super C, K, ? extends E> keyGetter,
			final Fn<? super C, ?, ? extends E> maker) {
		return new MapCache<C, K, E>(mapGetter, keyGetter, maker);
	}
	
	/**
	 * 委譲先で生成したオブジェクトを{@link ConcurrentMap}にキャッシュして返す関数を生成します。
	 * {@link ConcurrentMap#putIfAbsent(Object, Object)}を使用して返却値の一意性を保証しますが、
	 * ひとつの返却値に対応するインスタンス生成の委譲が複数回発生する可能性があります。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param <K>
	 * @param <E>
	 * @param mapGetter キャッシュ用マップを取得する委譲先。
	 * @param keyGetter キャッシュに格納する際のキーを取得する委譲先。
	 * @param maker キャッシュするファイルを生成するファクトリ。
	 * @return 関数。
	 * @see ConcurrentMapCache
	 */
	public static <C, K, E extends Exception> Fn<C, Object, E> concurrentCache(
			final Fn<? super C, ? extends ConcurrentMap<? super K, Object>, ? extends E> mapGetter,
			final Fn<? super C, K, ? extends E> keyGetter,
			final Fn<? super C, ?, ? extends E> maker) {
		return new ConcurrentMapCache<C, K, E>(mapGetter, keyGetter, maker);
	}
	
	/**
	 * 委譲先で生成したオブジェクトを{@link Map}にキャッシュして
	 * コンテキスト役に設定する関数を生成します。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param <K>
	 * @param <E>
	 * @param mapGetter キャッシュ用マップを取得する委譲先。
	 * @param keyGetter キャッシュに格納する際のキーを取得する委譲先。
	 * @param maker キャッシュするファイルを生成するファクトリ。
	 * @param el makerと本インスタンスによって値が格納される位置。
	 * @return 関数。
	 * @see SideEffectMapCache
	 */
	public static <C, K, E extends Exception> Fn<C, Void, E> cache(
			final Fn<? super C, ? extends Map<? super K, Object>, ? extends E> mapGetter,
			final Fn<? super C, ? extends K, ? extends E> keyGetter,
			final Fn<? super C, ?, ? extends E> maker,
			final EL el) {
		return new SideEffectMapCache<C, K, E>(mapGetter, keyGetter, maker, el);
	}
	
	/**
	 * 委譲先で生成したオブジェクトを{@link ConcurrentMap}にキャッシュして
	 * コンテキスト役に設定する関数を生成します。
	 * {@link ConcurrentMap#putIfAbsent(Object, Object)}を使用して返却値の一意性を保証しますが、
	 * ひとつの返却値に対応するインスタンス生成の委譲が複数回発生する可能性があります。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param <K>
	 * @param <E>
	 * @param mapGetter キャッシュ用マップを取得する委譲先。
	 * @param keyGetter キャッシュに格納する際のキーを取得する委譲先。
	 * @param maker キャッシュするファイルを生成するファクトリ。
	 * @param el makerと本インスタンスによって値が格納される位置。
	 * @return 関数。
	 * @see SideEffectConcurrentMapCache
	 */
	public static <C, K, E extends Exception> Fn<C, Void, E> concurrentCache(
			final Fn<? super C, ? extends ConcurrentMap<? super K, Object>, ? extends E> mapGetter,
			final Fn<? super C, ? extends K, ? extends E> keyGetter,
			final Fn<? super C, ?, ? extends E> maker,
			final EL el) {
		return new SideEffectConcurrentMapCache<C, K, E>(mapGetter, keyGetter, maker, el);
	}

	/**
	 * {@link GettingEL}に委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：Adapter。
	 * @param gettingEL 委譲先。
	 * @return 関数。
	 * @see GettingELFn
	 */
	public static Fn<Object, Object, RuntimeException> doEL(final GettingEL gettingEL) {
		return new GettingELFn<RuntimeException>(gettingEL);
	}

	/**
	 * 委譲先で取得した{@link InputStream}を{@link Reader}に変換する関数を生成します。
	 * 文字セットが不正な場合、取得した{@link InputStream#close()}を呼び出します。
	 * <br/>適用しているデザインパターン：{@link InputStream}のAbstract FactoryのAdapter。
	 * @param <C>
	 * @param inputStreamFactory {@link InputStream}のファクトリ。
	 * @param charset 文字セット。
	 * @return 関数。
	 * @see InputStreamReaderFactory
	 */
	public static <C> Fn<C, Reader, IOException> inputStreamReaderFactory(
			final Fn<? super C, ? extends InputStream, ? extends IOException> inputStreamFactory,
			final String charset) {
		return new InputStreamReaderFactory<C>(inputStreamFactory, charset);
	}

	/**
	 * {@link Class#newInstance()}を使用するオブジェクトファクトリを生成します。
	 * <br/>適用しているデザインパターン：オブジェクトのAbstract Factory。
	 * @param <R>
	 * @param clazz 生成するクラスの型。
	 * @return 関数。
	 * @see NewInstanceFactory
	 */
	public static <R> Fn<Object, R, Exception> newInstance(final Class<R> clazz) {
		return new NewInstanceFactory<R>(clazz);
	}

	/**
	 * 文字列のバイト配列表現を返すテスト用の関数を生成します。
	 * <br/>適用しているデザインパターン：{@link InputStream}のAbstract Factory。
	 * @param charset 文字セット。
	 * @param s 文字列。
	 * @return 関数。
	 * @see StringInputStreamFactory
	 */
	public static Fn<Object, InputStream, UnsupportedEncodingException> stringInputStreamFactory(final String charset, final String s) {
		return new StringInputStreamFactory(charset, s);
	}

	/**
	 * {@link StringReader}を返すテスト用の関数を生成します。
	 * <br/>適用しているデザインパターン：{@link StringReader}のAbstract Factory。
	 * @param s 文字列。
	 * @return 関数。
	 * @see StringReaderFactory
	 */
	public static Fn<Object, StringReader, RuntimeException> stringReaderFactory(final String s) {
		return new StringReaderFactory<RuntimeException>(s);
	}
}
