/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.convert;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.Format;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 変換処理のユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 *
 */
public final class ConvertUtils {

	/**
	 * 入力を{@link Number}にキャストして
	 * {@link BigDecimal}に変換する関数です。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * @see ToBigDecimalConverter
	 */
	public static final Fn<Object, BigDecimal, RuntimeException> TO_BIG_DECIMAL = new ToBigDecimalConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link BigInteger}に変換する関数です。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link ArithmeticException}を投げます。
	 * @see ToBigIntegerConverter
	 */
	public static final Fn<Object, BigInteger, RuntimeException> TO_BIG_INTEGER = new ToBigIntegerConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Byte}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 * @see ToByteConverter
	 */
	public static final Fn<Object, Byte, RuntimeException> TO_BYTE = new ToByteConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Double}に変換する関数です。
	 * {@link Number#doubleValue()}を使用するため丸め誤差が発生する可能性があります。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * @see ToDoubleConverter
	 */
	public static final Fn<Object, Double, RuntimeException> TO_DOUBLE = new ToDoubleConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Float}に変換する関数です。
	 * 一度値を double 型で取得して float 型に変換するため、
	 * 丸め誤差が発生する可能性があります。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * @see ToFloatConverter
	 */
	public static final Fn<Object, Float, RuntimeException> TO_FLOAT = new ToFloatConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Integer}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 * @see ToIntegerConverter
	 */
	public static final Fn<Object, Integer, RuntimeException> TO_INTEGER = new ToIntegerConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Long}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 * @see ToLongConverter
	 */
	public static final Fn<Object, Long, RuntimeException> TO_LONG = new ToLongConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Short}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 * @see ToShortConverter
	 */
	public static final Fn<Object, Short, RuntimeException> TO_SHORT = new ToShortConverter<RuntimeException>();
	
	/**
	 * 入力を{@link java.util.Date}にキャストして
	 * {@link java.sql.Date}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * @see ToSqlDateConverter
	 */
	public static final Fn<Object, java.sql.Date, RuntimeException> TO_SQL_DATE = new ToSqlDateConverter<RuntimeException>();
	
	/**
	 * 入力を{@link java.util.Date}にキャストして
	 * {@link Time}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * @see ToTimeConverter
	 * 
	 */
	public static final Fn<Object, Time, RuntimeException> TO_TIME = new ToTimeConverter<RuntimeException>();
	
	/**
	 * 入力を{@link java.util.Date}にキャストして
	 * {@link Timestamp}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * @see ToTimestampConverter
	 */
	public static final Fn<Object, Timestamp, RuntimeException> TO_TIMESTAMP = new ToTimestampConverter<RuntimeException>();
	
	/**
	 * 入力を{@link Date}にキャストして
	 * {@link Calendar}に変換する関数です。
	 * @see ToCalendarConverter
	 * 
	 */
	public static final Fn<Object, Calendar, RuntimeException> TO_CALENDAR = new ToCalendarConverter<RuntimeException>();
	
	/**
	 * {@link Object#toString()}で変換する関数です。
	 * @see ToStringConverter
	 */
	public static final Fn<Object, String, RuntimeException> TO_STRING = new ToStringConverter<RuntimeException>();
	
	private ConvertUtils() {
	}
	
	/**
	 * {@link Map#keySet()}を返す関数を生成します。
	 * @param <K>
	 * @return 関数。
	 * @see KeySetGetter
	 */
	public static <K> Fn<Map<K, ?>, Set<K>, RuntimeException> keySet() {
		return new KeySetGetter<K, RuntimeException>();
	}
	
	/**
	 * {@link Format}の新規のインスタンスを{@link Format#clone()}により複製するファクトリを生成します。
	 * {@link java.text.Format}の各サブクラスはスレッドセーフでなく
	 * {@link java.lang.Cloneable}を実装して Prototype デザインパターンを適用していますが、
	 * 本クラスはこれを Abstract Factory デザインパターンに変換しています。
	 * <br/>適用しているデザインパターン：{@link Format}のAbstract Factory、Prototype。
	 * @param format プロトタイプ。
	 * @return 関数。
	 * @see CloneFormatFactory
	 */
	public static Fn<Object, Format, RuntimeException> formatFactory(final Format format) {
		return new CloneFormatFactory<RuntimeException>(format);
	}
	
	/**
	 * {@link Format#format(Object)}で変換する関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param <E>
	 * @param formatFactory フォーマットのファクトリ。
	 * @return 関数。
	 * @see FormatConverter
	 */
	public static <E extends Exception> Fn<Object, String, E> format(
			final Fn<?, ? extends Format, ? extends E> formatFactory) {
		return new FormatConverter<E>(formatFactory);
	}
	
	/**
	 * コンテキスト役をIPv4アドレスとしてネットワークアドレスを返す関数を生成します。
	 * <br/>適用しているデザインパターン：ネットマスクとアドレスのCurrying。
	 * @param netMask ネットマスク。
	 * @return 関数。
	 * @see NetMaskConverter
	 */
	public static Fn<String, String, RuntimeException> netMask(final String netMask) {
		return new NetMaskConverter<RuntimeException>(netMask);
	}
	
	/**
	 * {@link Format#parseObject(String)}で変換する関数を生成します。
	 * 「NumberFormat の構文解析問題の解決」
	 * (http://www-06.ibm.com/jp/developerworks/java/library/j-numberformat/index.shtml)
	 * のテクニックを使用してデータの欠損を検出しています。
	 * パースに失敗した場合は{@link java.text.ParseException}を投げます。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param formatFactory フォーマットのファクトリ。
	 * @return 関数。
	 * @see ParseConverter
	 */
	public static Fn<String, Object, Exception> parse(final Fn<?, ? extends Format, ? extends Exception> formatFactory) {
		return new ParseConverter(formatFactory);
	}
	
	/**
	 * 正規表現で変換する関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param pattern 正規表現。
	 * @param replacement 変換パターン。
	 * @return 関数。
	 * @see RegExpConverter
	 */
	public static Fn<String, String, RuntimeException> convertRegExp(
			final Pattern pattern,
			final String replacement) {
		return new RegExpConverter<RuntimeException>(pattern, replacement);
	}
	
	/**
	 * 成功するまで順に委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param iterable 委譲先の一覧。
	 * @param errorFn エラー発生時の委譲先。
	 * @param finalFn 最後まで成功しなかった場合の委譲先。
	 * @return 関数。
	 * @see TrysFn
	 */
	public static <C, R, E extends Exception> Fn<C, R, E> trys(
			final Iterable<? extends Fn<? super C, ? extends R, ? extends Exception>> iterable,
			final Fn<? super Exception, ?, ? extends E> errorFn,
			final Fn<? super Exception, ? extends R, ? extends E> finalFn) {
		return new TrysFn<C, R, E>(iterable, errorFn, finalFn);
	}
	
	/**
	 * 成功するまで順に委譲する関数を生成します。
	 * 最後まで成功しなかった場合は{@link IllegalStateException}を投げます。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param iterable 委譲先の一覧。
	 * @return 関数。
	 * @see #trys(Iterable, Fn, Fn)
	 */
	public static <C, R> Fn<C, R, IllegalStateException> trys(
			final Iterable<? extends Fn<? super C, ? extends R, ? extends Exception>> iterable) {
		return trys(
				iterable,
				FnUtils.<Exception, R, IllegalStateException>fixThrows(null),
				new Fn<Exception, R, IllegalStateException>() {
					public R exec(final Exception c) throws IllegalStateException {
						throw new IllegalStateException(c);
					}
				});
	}
	
	/**
	 * マップの指定されたキーを残す関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param collection 残すキーのコレクション。
	 * @return 関数。
	 * @see RetainKeys
	 */
	public static Fn<Map<?, ?>, Boolean, RuntimeException> retainKeys(final Collection<?> collection) {
		return new RetainKeys<RuntimeException>(collection);
	}
}
