/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.container;

import java.util.ArrayList;
import java.util.List;

import woolpack.fn.Fn;

/**
 * インタープリタ デザインパターンとダブルディスパッチを使用した、
 * オブジェクトの生成をフラット構造で定義するためコンテキストです。
 * <br/>適用しているデザインパターン：InterpreterのContext役、Double Dispatch。
 * @author nakamura
 *
 * @param <S> サブコンテキスト。
 */
public class ContainerContext<S> {
	/**
	 * ダブルディスパッチの深さ閾値のデフォルト値です。
	 */
	public static final int DEFAULT_DEPTH = 32;
	
	private Fn<? super ContainerContext<S>, ?, ? extends Exception> fn;
	private S subContext;
	private int depth;
	
	private Object key;
	private int currentDepth;
	private List<Object> keyList;
	
	public ContainerContext() {
		this.depth = DEFAULT_DEPTH;
		currentDepth = 0;
	}
	
	/**
	 * 引数のキーを一時的に設定して委譲し、
	 * 委譲先から復帰したときに呼び出し時の状態に初期化します。
	 * @param key
	 * @return 生成されたオブジェクト。
	 * @throws IllegalStateException ダブルディスパッチの深さが閾値を超えたか、またはキー参照の循環を検出した場合。
	 */
	public Object visit(final Object key) throws Exception {
		if (currentDepth >= depth) {
			throw new IllegalStateException("depth out of bounds : " + keyList);
		}
		if (keyList == null) {
			keyList = new ArrayList<Object>(depth);
		} else if (keyList.contains(key)) {
			throw new IllegalStateException("cyclic key : " + keyList);
		}
		final Object baseKey = this.key;
		this.key = key;
		currentDepth++;
		keyList.add(key);
		try {
			return fn.exec(this);
		} finally {
			this.key = baseKey;
			currentDepth--;
			keyList.remove(keyList.size() - 1);
		}
	}
	
	public Object getKey() {
		return key;
	}
	public void setKey(final Object key) {
		this.key = key;
	}
	public Fn<? super ContainerContext<S>, ?, ? extends Exception> getFn() {
		return fn;
	}
	public void setFn(final Fn<? super ContainerContext<S>, ?, ? extends Exception> fn) {
		this.fn = fn;
	}
	public S getSubContext() {
		return subContext;
	}
	public void setSubContext(final S subContext) {
		this.subContext = subContext;
	}
	public int getDepth() {
		return depth;
	}
	public void setDepth(final int depth) {
		this.depth = depth;
	}
}
