/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.config;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import woolpack.el.AbstractEL;
import woolpack.el.EL;
import woolpack.factory.FactoryUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 設定情報をマップで管理するユーティリティです。
 * 
 * @author nakamura
 * 
 */
public final class ConfigUtils {
	/**
	 * {@link ConfigContext#getConfig()}を返す関数です。
	 * <br/>適用しているデザインパターン：Accessor。
	 * @see ConfigGetter
	 */
	public static final Fn<ConfigContext, Map<String, Object>, RuntimeException> GET_CONFIG = new ConfigGetter<RuntimeException>();

	/**
	 * {@link ConfigContext}のアクセサにアクセスする式言語です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final EL CONFIG_EL = new AbstractEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((ConfigContext) root).getConfig();
		}
		@Override
		public boolean setValue(final Object root, final Object value) {
			((ConfigContext) root).setConfig((Map<String, Object>) value);
			return true;
		}
	};
	
	private ConfigUtils() {
	}

	/**
	 * 設定情報の{@link java.util.Map}をキャッシュする関数を生成します。
	 * <br/>適用しているデザインパターン：オブジェクト生成処理のProxy、Flyweight。
	 * @param <C>
	 * @param <E>
	 * @param maker 設定情報を生成する委譲先。
	 * @return 関数。
	 */
	public static <C extends ConfigContext, E extends Exception> Fn<C, Void, E> cache(
			final Fn<? super C, ?, ? extends E> maker) {
		return FactoryUtils.cache(
				FnUtils.<C, Map<String, Object>, E>fixThrows(new HashMap<String, Object>(1)),
				FnUtils.<C, String, E>fixThrows(""),
				maker,
				ConfigUtils.CONFIG_EL
		);
	}
	
	/**
	 * {@link Map}の生成を委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：参照透過と副作用のAdapter。
	 * @param <C>
	 * @param <E>
	 * @param fn 委譲先。
	 * @return 関数。
	 * @see ConfigSetter
	 */
	public static <C extends ConfigContext, E extends Exception> Fn<C, Void, E> setConfig(
			final Fn<? super C, ? extends Map<String, Object>, ? extends E> fn) {
		return new ConfigSetter<C, E>(fn);
	}

	/**
	 * {@link ResourceBundle}の内容を追加する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link ResourceBundle}のAdapter。
	 * @param <C>
	 * @param <E>
	 * @param fn リソースバンドルのファクトリ。
	 * @return 関数。
	 * @see PutResourceBundle
	 */
	public static <C extends ConfigContext, E extends Exception> Fn<C, Void, E> putResourceBundle(
			final Fn<? super C, ? extends ResourceBundle, ? extends E> fn) {
		return new PutResourceBundle<C, E>(fn);
	}
	
	/**
	 * {@link ThreadLocal}から{@link Locale}を取得して使用する{@link ResourceBundle}のファクトリです。
	 * <br/>適用しているデザインパターン：{@link ResourceBundle}のAbstract Factory。
	 * @param path リソースバンドルのパス。
	 * @param fn ロケールのファクトリ。
	 * @return 関数。
	 */
	public static <C> Fn<C, ResourceBundle, RuntimeException> resourceBundleFactory(
			final String path,
			final Fn<? super C, ? extends Locale, ? extends RuntimeException> fn) {
		return new ResourceBundleFactory<C, RuntimeException>(path, fn);
	}
	
	/**
	 * 値一覧とラベル一覧をマージして、値とラベルの{@link LinkedHashMap}を生成する関数を生成します。
	 * {@link woolpack.html.HtmlUtils#makeRadio(woolpack.el.GettingEL)}や
	 * {@link woolpack.html.HtmlUtils#makeSelect(woolpack.el.GettingEL)}を使用して
	 * HTML のラジオボタン・チェックボックス・セレクトを再生成するための準備段階で使用します。
	 * このクラスでは例えば次の設定内容から
	 * 「part.hobby={0:bycycle, 1:car, 2:bike, 3:spacecraft}」を
	 * 生成することができます。
	 * <table border="1"><tr><td>
	 * part.hobby.value=0,1,2,3<br/>
	 * part.hobby.label=bycycle,car,bike,spacecraft
	 * </td></tr></table>
	 * <br/>適用しているデザインパターン：設定項目のBuilder、設定マップへのSide Effect。
	 * @param newMapListPosition
	 *            変換後の{@link LinkedHashMap}を格納する
	 *            {@link ConfigContext#getConfig()}上の位置。
	 * @param valueSourcePosition 値一覧を取得する{@link ConfigContext#getConfig()}上の位置。
	 * @param labelSourcePosition ラベル一覧を取得する{@link ConfigContext#getConfig()}上の位置。
	 * @param separator 一覧(値・ラベル)それぞれ値・ラベルに分解するためのセパレータ。
	 * @return 関数。
	 */
	public static Fn<ConfigContext, Void, RuntimeException> toLinkedHashMap(
			final String newMapListPosition,
			final String valueSourcePosition,
			final String labelSourcePosition,
			final String separator) {
		return new ToLinkedHashMap<RuntimeException>(newMapListPosition, valueSourcePosition, labelSourcePosition, separator);
	}
}
