/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.bool;

import java.util.Collection;
import java.util.regex.Pattern;

import woolpack.fn.FixFn;
import woolpack.fn.Fn;
import woolpack.fn.IfFn;

/**
 * 真偽値処理のユーティリティです。
 * 
 * @author nakamura
 *
 */
public final class BoolUtils {
	/**
	 * 引数が{@link Throwable}のサブクラスでないことを評価する関数です。
	 */
	public static final Fn<Object, Boolean, RuntimeException> NOT_THROWABLE = not(matchObject(Throwable.class));
	
	/**
	 * コンテキストを否定する関数です。
	 * @see NotFn
	 */
	public static final Fn<Boolean, Boolean, RuntimeException> NOT = new NotFn<RuntimeException>();
	
	/**
	 * null でない、かつ文字列の場合は空でないことをチェックする関数です。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * @see NotEmptyChecker
	 */
	public static final Fn<Object, Boolean, RuntimeException> NOT_EMPTY = new NotEmptyChecker<RuntimeException>();

	/**
	 * null であることをチェックする関数です。
	 */
	public static final Fn<Object, Boolean, RuntimeException> IS_NULL = new NullChecker<RuntimeException>();
	
	/**
	 * 引数が null または{@link Boolean#FALSE}の場合のみ
	 * {@link Boolean#FALSE}を返却する関数です。
	 * @see BooleanConverter
	 */
	public static final Fn<Object, Boolean, RuntimeException> TO_BOOLEAN = new BooleanConverter<RuntimeException>();

	/**
	 * 引数が null または{@link Boolean#FALSE} または文字列"false"(大小を区別しない)の場合のみ
	 * {@link Boolean#FALSE}を返却する関数です。
	 * @see ViewBooleanConverter
	 */
	public static final Fn<Object, Boolean, RuntimeException> TO_BOOLEAN_VIEW = new ViewBooleanConverter<RuntimeException>();
	
	/**
	 * 論理AND演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 * @see BooleanStateAndAndFactory
	 */
	public static final Fn<Object, BooleanState, RuntimeException> ANDAND = new BooleanStateAndAndFactory();
	
	/**
	 * ビットAND演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 * @see BooleanStateAndFactory
	 */
	public static final Fn<Object, BooleanState, RuntimeException> AND = new BooleanStateAndFactory();
	
	/**
	 * 論理同値演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 * @see BooleanStateEqEqFactory
	 */
	public static final Fn<Object, BooleanState, RuntimeException> EQEQ = new BooleanStateEqEqFactory();
	
	/**
	 * ビット同値演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 * @see BooleanStateEqFactory
	 */
	public static final Fn<Object, BooleanState, RuntimeException> EQ = new BooleanStateEqFactory();
	
	/**
	 * ビットOR演算子のファクトリです
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 * @see BooleanStateOrFactory
	 */
	public static final Fn<Object, BooleanState, RuntimeException> OR = new BooleanStateOrFactory();
	
	/**
	 * 論理OR演算子のファクトリです。
	 * <br/>適用しているデザインパターン：{@link BooleanState}のAbstract Factory。
	 * @see BooleanStateOrOrFactory
	 */
	public static final Fn<Object, BooleanState, RuntimeException> OROR = new BooleanStateOrOrFactory();
	
	private BoolUtils() {
	}
	
	/**
	 * 委譲先を順次呼び出して集計する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <E>
	 * @param factory 真偽集計器のファクトリ。
	 * @param iterable 委譲先の一覧。
	 * @return 関数。
	 * @see BoolSeq
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> boolSeq(
			final Fn<Object, ? extends BooleanState, ? extends RuntimeException> factory,
			final Iterable<? extends Fn<? super C, Boolean, ? extends E>> iterable) {
		return new BoolSeq<C, E>(factory, iterable);
	}
	
	/**
	 * 比較する関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param <C>
	 * @param value 比較元({@link Comparable#compareTo(Object)}の引数でないほう)。
	 * @return 関数。
	 * @see CompareFn
	 */
	public static <C extends Comparable<C>> Fn<C, Integer, RuntimeException> compare(final C value) {
		return new CompareFn<C, RuntimeException>(value);
	}
	
	/**
	 * コレクションがすべて初期コレクションに含まれることをチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param value 初期コレクション。
	 * @return 関数。
	 * @see ContainsAllChecker
	 */
	public static Fn<Collection<?>, Boolean, RuntimeException> containsAll(final Collection<?> value) {
		return new ContainsAllChecker<RuntimeException>(value);
	}
	
	/**
	 * コンテキスト役が初期コレクションに含まれることをチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param value 初期コレクション。
	 * @return 関数。
	 * @see ContainsChecker
	 */
	public static Fn<Object, Boolean, RuntimeException> contains(final Collection<?> value) {
		return new ContainsChecker<RuntimeException>(value);
	}
	
	/**
	 * {@link Object#equals(Object)}で同値関係をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param value 初期値。
	 * @return 関数。
	 * @see EqualsChecker
	 */
	public static Fn<Object, Boolean, RuntimeException> checkEquals(final Object value) {
		return new EqualsChecker<RuntimeException>(value);
	}
	
	/**
	 * 最大値をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param <C>
	 * @param value 最大値。
	 * @return 関数。
	 * @see MaxChecker
	 */
	public static <C extends Comparable<C>> Fn<C, Boolean, RuntimeException> checkMax(final C value) {
		return new MaxChecker<C, RuntimeException>(value);
	}
	
	/**
	 * 文字列の最大文字数をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param value 最大文字数。
	 * @return 関数。
	 * @see MaxLengthChecker
	 */
	public static Fn<String, Boolean, RuntimeException> checkMaxLength(final int value) {
		return new MaxLengthChecker<RuntimeException>(value);
	}
	
	/**
	 * 最小値をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param <C>
	 * @param value 最小値。
	 * @return 関数。
	 * @see MinChecker
	 */
	public static <C extends Comparable<C>> Fn<C, Boolean, RuntimeException> checkMin(final C value) {
		return new MinChecker<C, RuntimeException>(value);
	}
	
	/**
	 * 文字列の最小文字数をチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：二項演算子のCurrying。
	 * @param value 最小文字数。
	 * @return 関数。
	 * @see MinLengthChecker
	 */
	public static Fn<String, Boolean, RuntimeException> checkMinLength(final int value) {
		return new MinLengthChecker<RuntimeException>(value);
	}
	
	/**
	 * 委譲先の結果の否定する関数を生成します。
	 * <br/>適用しているデザインパターン：返却値のDecorator。
	 * @param <C>
	 * @param <E>
	 * @param fn 委譲先。
	 * @return 関数。
	 * @see NotDelegator
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> not(final Fn<? super C, Boolean, ? extends E> fn) {
		return new NotDelegator<C, E>(fn);
	}
	
	/**
	 * プロパティ値により以下の判定を行う関数を生成します。
	 * null の場合は、コンテキストが null であることを検証します。
	 * {@link Class}クラスのインスタンスの場合は、
	 * コンテキストがそのクラスの変数に代入できることを検証します。
	 * 上記以外の場合は、値が等しいことを検証します。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param value 初期値。
	 * @return 関数。
	 * @see ObjectMatcher
	 */
	public static Fn<Object, Boolean, RuntimeException> matchObject(final Object value) {
		return new ObjectMatcher<RuntimeException>(value);
	}
	
	/**
	 * 正規表現でチェックする関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param pattern 正規表現。
	 * @return 関数。
	 * @see RegExpChecker
	 */
	public static Fn<String, Boolean, RuntimeException> checkRegExp(final Pattern pattern) {
		return new RegExpChecker<RuntimeException>(pattern);
	}
	
	/**
	 * 評価結果が{@link Boolean#TRUE}の場合に委譲するを生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <E>
	 * @param ifFn 評価の委譲先。
	 * @param trueFn 評価結果が{@link Boolean#TRUE}の場合の委譲先。
	 * @return 関数。
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> ifTrue(
			final Fn<? super C, ?, ? extends E> ifFn,
			final Fn<? super C, ? extends Boolean, ? extends E> trueFn) {
		return new IfFn<C, Boolean, E>(ifFn, trueFn, new FixFn<C, Boolean, E>(false));
	}
	
	/**
	 * 評価結果が{@link Boolean#TRUE}でない場合に委譲する{@link Fn}を返します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <E>
	 * @param ifFn 評価の委譲先。
	 * @param falseFn 評価結果が{@link Boolean#TRUE}でない場合の委譲先。
	 * @return 関数。
	 */
	public static <C, E extends Exception> Fn<C, Boolean, E> ifNot(
			final Fn<? super C, ?, ? extends E> ifFn,
			final Fn<? super C, ? extends Boolean, ? extends E> falseFn) {
		return new IfFn<C, Boolean, E>(ifFn, new FixFn<C, Boolean, E>(true), falseFn);
	}
}
