/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.adapter;

import java.util.List;

import org.apache.commons.jxpath.JXPathContext;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.fn.Fn;

/**
 * JXPath を使用して DOM ノードを検索する関数を生成するユーティリティです。
 * 委譲先でノードを操作した場合、次の検索結果に影響します。
 * つまりスナップショットではありません。
 * <br/>適用しているデザインパターン：JXPathのAdapter。
 * @author nakamura
 *
 */
public final class JXPUtils {
	
	/**
	 * ノードを検索する関数を生成する関数です。
	 */
	public static final Fn<String, Fn<Object, Node, RuntimeException>, RuntimeException> ONE =
		new Fn<String, Fn<Object, Node, RuntimeException>, RuntimeException>() {
			public Fn<Object, Node, RuntimeException> exec(final String c) {
				return one(c);
			}
	};
	
	/**
	 * ノードリストを検索する関数を生成する関数です。
	 */
	public static final Fn<String, Fn<Object, NodeList, RuntimeException>, RuntimeException> LIST =
		new Fn<String, Fn<Object, NodeList, RuntimeException>, RuntimeException>() {
			public Fn<Object, NodeList, RuntimeException> exec(final String c) {
				return list(c);
			}
	};
	
	private JXPUtils() {
	}
	
	/**
	 * ノードを検索する関数を生成します。
	 * @param expression XPath文字列。
	 * @return 関数。
	 * @throws org.apache.commons.jxpath.JXPathException XPath のコンパイルに失敗した場合。
	 */
	public static Fn<Object, Node, RuntimeException> one(final String expression) {
		JXPathContext.compile(expression);
		return new Fn<Object, Node, RuntimeException>() {
			public Node exec(final Object c) throws RuntimeException {
				return (Node) JXPathContext.newContext(c).selectSingleNode(expression);
			}
		};
	}
	
	/**
	 * ノードリストを検索する関数を生成します。
	 * list("//*[@" + attrName + "=\"" + attrValue + "\"]") と同一です。
	 * @param attrName 属性名。
	 * @param attrValue 属性値。
	 * @return 関数。
	 * @throws org.apache.commons.jxpath.JXPathException XPath のコンパイルに失敗した場合。
	 */
	public static Fn<Object, NodeList, RuntimeException> list(final String attrName, final String attrValue) {
		return list("//*[@" + attrName + "=\"" + attrValue + "\"]");
	}
	
	/**
	 * ノードリストを検索する関数を生成します。
	 * @param expression XPath文字列。
	 * @return 関数。
	 * @throws org.apache.commons.jxpath.JXPathException XPath のコンパイルに失敗した場合。
	 */
	public static Fn<Object, NodeList, RuntimeException> list(final String expression) {
		JXPathContext.compile(expression);
		return new Fn<Object, NodeList, RuntimeException>() {
			public NodeList exec(final Object c) throws RuntimeException {
				final List list = JXPathContext.newContext(c).selectNodes(expression);
				return new NodeList() {
					public Node item(final int index) {
						return (Node) list.get(index);
					}

					public int getLength() {
						return list.size();
					}
				};
			}
		};
	}
}
