/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.HashMap;
import java.util.Map;

import woolpack.el.ELUtils;
import woolpack.el.GettingEL;
import woolpack.web.WebUtils;

/**
 * 特定のパターンの{@link ActionDef}を{@link Map}に登録するビルダです。
 * <br/>適用しているデザインパターン：{@link Map}のBuilder。
 * @author nakamura
 *
 */
public class ActionDefMaker {

	private final Map<String, ActionDef> map;
	
	public ActionDefMaker() {
		this.map = new HashMap<String, ActionDef>();
	}
	
	/**
	 * 生成された{@link Map}を返します。
	 * @return 生成された{@link Map}。
	 */
	public Map<String, ActionDef> get() {
		return map;
	}
	
	/**
	 * アクションの定義を登録します。
	 * @param id id。
	 * @param actionDef アクションの定義。
	 * @return このオブジェクトへの参照。
	 */
	public ActionDefMaker put(final String id, final ActionDef actionDef) {
		map.put(id, actionDef);
		return this;
	}
	
	private ActionDefMaker put(
			final String actionId,
			final GettingEL componentEL,
			final GettingEL methodEL,
			final String forwardId,
			final GettingEL forwardEL) {
		return put(actionId, new ActionDef(componentEL, methodEL, new ForwardDef(forwardId, forwardEL)));
	}
	
	/**
	 * アクションの定義を追加します。
	 * メソッドの返却値がエラーでないチェック条件で、
	 * かつ遷移先の id をアクション id と同一とした遷移先ひとつを遷移先一覧とします。
	 * @param id id。
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではありません。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよいです。
	 * @param forwardEL 遷移先のコンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではありません。
	 * @return このオブジェクトへの参照。
	 */
	public ActionDefMaker put(
			final String id,
			final GettingEL componentEL,
			final GettingEL methodEL,
			final GettingEL forwardEL) {
		return put(id, componentEL, methodEL, id, forwardEL);
	}
	
	/**
	 * 遷移先の id をアクション id と同一とし、
	 * 副作用が発生しないアクション定義を登録します。
	 * @param id id。
	 * @return このオブジェクトへの参照。
	 */
	public ActionDefMaker putForward(final String id) {
		return put(id, ELUtils.NULL, ELUtils.NULL, id, ELUtils.NULL);
	}
	
	/**
	 * 遷移先のコンポーネントへの参照を画面からの
	 * 入力値そのものとするアクションの定義を登録します。
	 * @param actionId アクション id。
	 * @param forwardId 遷移先の id。
	 * @return このオブジェクトへの参照。
	 */
	public ActionDefMaker putEcho(final String actionId, final String forwardId) {
		return put(actionId, ELUtils.NULL, ELUtils.NULL, forwardId, WebUtils.INPUT_EL);
	}
	
	/**
	 * 遷移先の id をアクション id と同一とし、
	 * 遷移先のコンポーネントへの参照を画面からの入力値そのものとするアクションの定義を登録します。
	 * @param id id。
	 * @return このオブジェクトへの参照。
	 */
	public ActionDefMaker putEcho(final String id) {
		return put(id, ELUtils.NULL, ELUtils.NULL, id, WebUtils.INPUT_EL);
	}
}
