/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.acquirable;

import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.locks.Lock;

import woolpack.fn.Fn;

/**
 * 実行権制御のユーティリティです。
 * @author nakamura
 * 
 */
public final class AcquirableUtils {

	/**
	 * 無制限に実行権を与える実行権マネージャです。
	 * <br/>適用しているデザインパターン：Null Object。
	 */
	public static final Acquirable<RuntimeException> ANY = new Acquirable<RuntimeException>() {
		public boolean acquire() {
			return true;
		}
		public void release() {
			// nothing to do.
		}
	};

	/**
	 * 常に実行権を与えない実行権マネージャです。
	 * <br/>適用しているデザインパターン：Null Object。
	 */
	public static final Acquirable<RuntimeException> NONE = new Acquirable<RuntimeException>() {
		public boolean acquire() {
			return false;
		}
		public void release() {
			// nothing to do.
		}
	};

	/**
	 * 常に実行権を与えず、実行したスレッドに対し{@link Thread#interrupt()}を実行する実行権マネージャです。
	 * <br/>適用しているデザインパターン：Null Object。
	 */
	public static final Acquirable<RuntimeException> INTERRUPT = new Acquirable<RuntimeException>() {
		public boolean acquire() {
			Thread.currentThread().interrupt();
			return false;
		}
		public void release() {
			// nothing to do.
		}
	};

	/**
	 * 常に実行権を与えず、
	 * 実行したスレッドに対し{@link Thread#interrupt()}を実行し、
	 * {@link InterruptedException}を原因として持つ
	 * {@link IllegalStateException}を投げる実行権マネージャです。
	 * <br/>適用しているデザインパターン：Null Object。
	 */
	public static final Acquirable<InterruptedException> INTERRUPTED_EXCEPTION = new Acquirable<InterruptedException>() {
		public boolean acquire() throws InterruptedException {
			Thread.currentThread().interrupt();
			throw new InterruptedException();
		}

		public void release() {
			// nothing to do.
		}
	};

	private AcquirableUtils() {
	}
	
	/**
	 * 取得した{@link Acquirable}で実行権を制御する関数を生成します。
	 * <br/>適用しているデザインパターン：trueFnへ委譲する処理のBalking。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param acquireGetter 実行権マネージャのファクトリ。
	 * @param trueFn 実行権の取得に成功したか{@link Acquirable}が存在しない場合の委譲先。
	 * @param falseFn 実行権の取得に失敗した場合の委譲先。
	 * @return 関数。
	 * @see AcquireFn
	 */
	public static <C, R, E extends Exception> Fn<C, R, E> acquire(
			final Fn<? super C, ? extends Acquirable<? extends E>, ? extends E> acquireGetter,
			final Fn<? super C, ? extends R, ? extends E> trueFn,
			final Fn<? super C, ? extends R, ? extends E> falseFn) {
		return new AcquireFn<C, R, E>(acquireGetter, trueFn, falseFn);
	}
	
	/**
	 * {@link Acquirable#acquire()}で{@link Lock#lock()}を実行し、
	 * {@link Acquirable#release()}で{@link Lock#unlock()}
	 * を実行する実行権マネージャを生成します。
	 * <br/>適用しているデザインパターン：{@link Lock}のAdapter。
	 * @param lock ロッククラス。
	 * @return 実行権マネージャ。
	 * @see DoLock
	 */
	public static Acquirable<RuntimeException> doLock(final Lock lock) {
		return new DoLock<RuntimeException>(lock);
	}
	
	/**
	 * {@link Acquirable#acquire()}で{@link Semaphore#acquire()}を実行し、
	 * {@link Acquirable#release()}で{@link Semaphore#release()}を実行する実行権マネージャを生成します。
	 * <br/>適用しているデザインパターン：{@link Semaphore}のAdapter。
	 * @param semaphore セマフォ。
	 * @return 実行権マネージャ。
	 * @see DoSemaphore
	 */
	public static Acquirable<InterruptedException> doSemaphore(final Semaphore semaphore) {
		return new DoSemaphore(semaphore);
	}
	
	/**
	 * {@link Acquirable#acquire()}で{@link Lock#tryLock()}を実行し、
	 * {@link Acquirable#release()}で{@link Lock#unlock()}を実行する実行権マネージャを生成します。
	 * <br/>適用しているデザインパターン：{@link Lock}のAdapter。
	 * @param lock ロッククラス。
	 * @return 実行権マネージャ。
	 * @see TryLock
	 */
	public static Acquirable<RuntimeException> tryLock(final Lock lock) {
		return new TryLock<RuntimeException>(lock);
	}
	
	/**
	 * {@link Acquirable#acquire()}で{@link Semaphore#tryAcquire()}を実行し、
	 * {@link Acquirable#release()}で{@link Semaphore#release()}を実行する実行権マネージャを生成します。
	 * <br/>適用しているデザインパターン：{@link Semaphore}のAdapter。
	 * @param semaphore セマフォ。
	 * @return 実行権マネージャ。
	 * @see TrySemaphore
	 */
	public static Acquirable<RuntimeException> trySemaphore(final Semaphore semaphore) {
		return new TrySemaphore<RuntimeException>(semaphore);
	}
	
	/**
	 * コンストラクタ引数で指定された{@link Acquirable}を順に
	 * {@link Acquirable#acquire()}する実行権マネージャを生成します。
	 * {@link Acquirable#acquire()}による実行権の取得に失敗した場合、
	 * 既に実行権を取得した{@link Acquirable}
	 * を逆順に{@link Acquirable#release()}します。
	 * <br/>適用しているデザインパターン：{@link Acquirable}のComposite。
	 * @param <E>
	 * @param list 実行権マネージャの配列。
	 * @return 関数。
	 * @see AcquirableChain
	 */
	public static <E extends Exception> Acquirable<E> chain(final List<? extends Acquirable<? extends E>> list) {
		return new AcquirableChain<E>(list);
	}
	
	/**
	 * {@link Acquirable#acquire()}で{@link Semaphore#acquire()}を実行し、
	 * {@link Acquirable#release()}で{@link Semaphore#release()}を実行する実行権マネージャ
	 * のファクトリを生成します。
	 * <br/>適用しているデザインパターン：{@link Acquirable}のAbstract Factory。
	 * @param <E>
	 * @param permit 初期パーミット数。
	 * @param fair フェアなセマフォを作成する場合は true。
	 * @return 関数。
	 * @see DoSemaphoreFactory
	 */
	public static <E extends Exception> Fn<Object, ? extends Acquirable<InterruptedException>, E> doSemaphoreFactory(
			final int permit,
			final boolean fair) {
		return new DoSemaphoreFactory<E>(permit, fair);
	}
	
	/**
	 * {@link Acquirable#acquire()}で{@link Semaphore#tryAcquire()}を実行し、
	 * {@link Acquirable#release()}で{@link Semaphore#release()}を実行する実行権マネージャ
	 * のファクトリを生成します。
	 * <br/>適用しているデザインパターン：{@link Acquirable}のAbstract Factory。
	 * @param <E>
	 * @param permit 初期パーミット数。
	 * @param fair フェアなセマフォを作成する場合は true。
	 * @return 関数。
	 * @see TrySemaphoreFactory
	 */
	public static <E extends Exception> Fn<Object, ? extends Acquirable<E>, E> trySemaphoreFactory(
			final int permit,
			final boolean fair) {
		return new TrySemaphoreFactory<E>(permit, fair);
	}
}
