/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.utils.Utils;
import woolpack.xml.XmlTransformerUtils;
import woolpack.xml.XmlUtils;

public class TransactionBuilderTest extends TestCase {

	public void testGet() throws Exception {
		final TransactionBuilder transactionBuilder = new TransactionBuilder(
				"token", Arrays.asList("id0"), 32);
		assertEquals("token", transactionBuilder.getKey());
		final Collection<String> expected = Arrays.asList("id0");
		assertTrue(expected.containsAll(transactionBuilder.getIdCollection()));
		assertTrue(transactionBuilder.getIdCollection().containsAll(expected));
	}

	public void testGetAppendExpression() throws Exception {
		final TransactionBuilder transactionBuilder = new TransactionBuilder(
				Arrays.asList("id0"));
		final EEContext context = new EEContext();
		context.setSession(Utils.concurrentMap(
				new HashMap<String, Object>(), new Object()));
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN",
				"token0");
		context.setNode(XmlTransformerUtils.convert("<HTML><BODY>"
				+ "<A href=\"a0.html\">a0</A>"
				+ "<A href=\"a1.html?k0=v0\">a1</A>"
				+ "<A href=\"a2.html?k0=v0&amp;k1=v1\">a2</A>"
				+ "<FORM/>"
				+ "</BODY></HTML>"));
		
		transactionBuilder.getAppendExpression().exec(context);

		assertTrue(XmlUtils.equalsNode(XmlTransformerUtils.convert("<HTML><BODY>"
				+ "<A href=\"a0.html?woolpack.transaction.TRANSACTION_TOKEN=token0\">a0</A>"
				+ "<A href=\"a1.html?k0=v0&amp;woolpack.transaction.TRANSACTION_TOKEN=token0\">a1</A>"
				+ "<A href=\"a2.html?k0=v0&amp;k1=v1&amp;woolpack.transaction.TRANSACTION_TOKEN=token0\">a2</A>"
				+ "<FORM><INPUT type=\"hidden\" name=\"woolpack.transaction.TRANSACTION_TOKEN\" value=\"token0\" /></FORM>"
				+ "</BODY></HTML>"), context.getNode()));
	}

	public void testGetCheckExpression() throws Exception {
		final List<String> list = new ArrayList<String>();
		final TransactionBuilder transactionBuilder = new TransactionBuilder(Arrays.asList("id0"));
		final Fn<EEContext, Void, RuntimeException> fn = transactionBuilder.getCheckExpression(
				FnUtils.<EEContext, Void, RuntimeException>recode(null, "0", list),
				FnUtils.<EEContext, Void, RuntimeException>recode(null, "1", list));
		final EEContext context = new EEContext();
		context.setInput(new HashMap());
		context.setSession(Utils.concurrentMap(
				new HashMap<String, Object>(), new Object()));
		context.setInput(Utils.map("woolpack.transaction.TRANSACTION_TOKEN", "token0"));

		context.setId("id0");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token0");
		fn.exec(context);

		System.out.println(context.getSession().get(TransactionBuilder.KEY));
		assertEquals(32, context.getSession().get(TransactionBuilder.KEY).toString().length());

		context.setId("id0");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token1");
		fn.exec(context);

		context.setId("id1");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token0");
		fn.exec(context);

		context.setId("id1");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token1");
		fn.exec(context);

		context.setId("id1");
		context.getSession().remove("woolpack.transaction.TRANSACTION_TOKEN");
		fn.exec(context);
		
		assertEquals(Arrays.asList("0", "1", "0", "0", "0"), list);
	}
}
