/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.adapter;

import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;

import org.apache.commons.jxpath.JXPathException;

import woolpack.el.ELTargetRuntimeException;
import woolpack.test.TestBean;

public class JXETest extends TestCase {
	public void testSetValueGetValue() {
		final Map<String, String> map = new HashMap<String, String>();
		map.put("param0", "value0");
		assertEquals("value0", new JXE("param0").getValue(map));
		assertEquals("value0", new JXE("param0").getValue(map, String.class));

		assertTrue(new JXE("param1").setValue(map, "value1"));
		assertEquals("value1", new JXE("param1").getValue(map));

		assertNull(new JXE("param2").getValue(map));

		assertTrue(new JXE("param3").setValue(map, null));
		assertNull(new JXE("param3").getValue(map));
	}

	public void testString() {
		final Map<String, Object> map = new HashMap<String, Object>();
		map.put("param0", Integer.valueOf(7));
		assertEquals(Integer.valueOf(7), new JXE("param0").getValue(map));
		assertTrue(new JXE("param0").getValue(map, String.class) instanceof String);
		assertEquals("7", new JXE("param0").getValue(map, String.class));

		assertTrue(new JXE("param1").setValue(map, Integer.valueOf(11)));
		assertEquals(Integer.valueOf(11), new JXE("param1").getValue(map));
		assertEquals("11", new JXE("param1").getValue(map, String.class));
	}

	// public void testCollection() {
	// assertTrue(TestUtils.equals(Arrays.asList(13, 17), new
	// JXE("#this").getValue(new int[]{13, 17}, List.class)));
	// }

	public void testGetValueException() {
		final Map map = new HashMap();
		try {
			new JXE("param0/param1").getValue(map);
			fail();
		} catch (final JXPathException expected) {
		}
	}

	public void testGetStringException() {
		final Map map = new HashMap();
		try {
			new JXE("param0/param1").getValue(map, String.class);
			fail();
		} catch (final JXPathException expected) {
		}
	}

	public void testSetValueException() {
		final Map map = new HashMap();
		try {
			new JXE("param0/param1").setValue(map, "a");
			fail();
		} catch (final JXPathException expected) {
		}
	}

	public void testInvoke() {
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		{
			final Object result = new JXE("executeVoid(component0)")
					.getValue(root);
			assertNull(result);
		}
		{
			Object result = new JXE("executeInt(component0)").getValue(root);
			assertEquals(Integer.valueOf(1), result);

			result = new JXE("executeInt(component0)").getValue(root);
			assertEquals(Integer.valueOf(2), result);
		}
		try {
			new JXE("executeException(component0)").getValue(root);
			fail();
		} catch (final ELTargetRuntimeException expected) {
			assertTrue(expected.getCause() instanceof IndexOutOfBoundsException);
		}
	}

	public void testInvokeMethodNotFound() {
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		try {
			new JXE("component0/executeNotFound()").getValue(root);
			fail();
		} catch (final JXPathException expected) {
		}
	}

	public void testInvokeException() {
		final Map<String, Object> root = new HashMap<String, Object>();
		root.put("component0", new TestBean());
		try {
			new JXE("component0/notFoundProperty").getValue(root);
			fail();
		} catch (final JXPathException expected) {
		}
	}
}
