/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import javax.xml.namespace.QName;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.fn.Fn;

/**
 * {@link javax.xml.xpath.XPath}を使用して DOM ノードを検索する{@link Fn}を生成するユーティリティです。
 * 委譲先でノードを操作した場合、次の検索結果に影響します(スナップショットではありません)。
 * <br/>適用しているデザインパターン：Adapter。
 * @author nakamura
 *
 */
public final class XPathUtils {
	private XPathUtils() {
	}
	
	private static XPathExpression compile(final String s) throws XPathExpressionException {
		final XPath base = XPathFactory.newInstance().newXPath();
		return base.compile(s);
	}

	private static Object evaluate(
			final String s,
			final QName kind,
			final Object node) throws XPathExpressionException {
		final XPathExpression expression = compile(s);
		return expression.evaluate(node, kind);
	}
	
	/**
	 * ノードを検索する関数を生成します。
	 * @param expression XPath文字列。
	 * @return 関数。
	 * @throws XPathExpressionException
	 */
	public static Fn<Object, Node, XPathExpressionException> one(final String expression) throws XPathExpressionException {
		compile(expression);
		return new Fn<Object, Node, XPathExpressionException>() {
			public Node exec(final Object c) throws XPathExpressionException {
				return (Node) evaluate(expression, XPathConstants.NODE, c);
			}
		};
	}
	
	/**
	 * ノードリストを検索する関数を生成します。
	 * @param expression XPath文字列。
	 * @return 関数。
	 * @throws XPathExpressionException
	 */
	public static Fn<Object, NodeList, XPathExpressionException> list(final String expression) throws XPathExpressionException {
		compile(expression);
		return new Fn<Object, NodeList, XPathExpressionException>() {
			public NodeList exec(final Object c) throws XPathExpressionException {
				return (NodeList) evaluate(expression, XPathConstants.NODESET, c);
			}
		};
	}
}
