/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.web;

import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import woolpack.container.ContainerContext;
import woolpack.container.ContainerUtils;
import woolpack.el.AbstractGettingEL;
import woolpack.el.GettingEL;
import woolpack.factory.ConcurrentMapCache;
import woolpack.factory.FactoryUtils;
import woolpack.factory.MapCache;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 各スコープを操作するユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * @author nakamura
 *
 */
public final class WebUtils {

	/**
	 * {@link WebContext}を基点とみなして
	 * リクエストスコープにアクセスする式言語です。
	 */
	public static final GettingEL REQUEST_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getRequest();
		}
	};
	
	/**
	 * {@link WebContext}を基点とみなして
	 * セッションスコープにアクセスする式言語です。
	 */
	public static final GettingEL SESSION_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getSession();
		}
	};

	/**
	 * {@link WebContext}を基点とみなして
	 * アプリケーションスコープにアクセスする式言語です。
	 */
	public static final GettingEL APPLICATION_EL = new AbstractGettingEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((WebContext) root).getApplication();
		}
	};
	
	/**
	 * セッションスコープをクリアする関数です。
	 */
	public static final Fn<WebContext, Void, RuntimeException> CLEAR_SESSION = new Fn<WebContext, Void, RuntimeException>() {
		public Void exec(final WebContext context) {
			context.getSession().clear();
			return null;
		}
	};
	
	private static <E extends Exception> Fn<ContainerContext<WebContext>, String, Exception> getKey() {
		return new Fn<ContainerContext<WebContext>, String, Exception>() {
			public String exec(final ContainerContext<WebContext> c) {
				return "woolpack.web.WebUtils." + c.getKey();
			}
		};
	}
	
	private WebUtils() {
	}
	
	/**
	 * リクエストスコープを返す関数を生成します。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, Map<String, Object>, E> requestFn() {
		return new Fn<WebContext, Map<String, Object>, E>() {
			public Map<String, Object> exec(final WebContext c) {
				return c.getRequest();
			}
		};
	}
	
	/**
	 * セッションスコープを返す関数を生成します。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, ConcurrentMap<String, Object>, E> sessionFn() {
		return new Fn<WebContext, ConcurrentMap<String, Object>, E>() {
			public ConcurrentMap<String, Object> exec(final WebContext c) {
				return c.getSession();
			}
		};
	}
	
	/**
	 * アプリケーションスコープを返す関数を生成します。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<WebContext, ConcurrentMap<String, Object>, E> applicationFn() {
		return new Fn<WebContext, ConcurrentMap<String, Object>, E>() {
			public ConcurrentMap<String, Object> exec(final WebContext c) {
				return c.getApplication();
			}
		};
	}

	/**
	 * リクエストスコープでキャッシュする関数を生成します。
	 * @param <C>
	 * @param fn オブジェクト生成の委譲先。
	 * @return 関数。
	 */
	public static <C extends ContainerContext<WebContext>>
	MapCache<C, String, Exception> request(
			final Fn<? super C, ?, ? extends Exception> fn) {
		return FactoryUtils.cache(
				FnUtils.join(
						ContainerUtils.<WebContext, C>getSubContext(),
						WebUtils.requestFn()),
				WebUtils.getKey(),
				fn);
	}
	
	/**
	 * セッションスコープでキャッシュする関数を生成します。
	 * @param <C>
	 * @param fn オブジェクト生成の委譲先。
	 * @return 関数。
	 */
	public static <C extends ContainerContext<WebContext>>
	ConcurrentMapCache<C, String, Exception> session(
			final Fn<? super C, ?, ? extends Exception> fn) {
		return FactoryUtils.concurrentCache(
				FnUtils.join(
						ContainerUtils.<WebContext, C>getSubContext(),
						WebUtils.sessionFn()),
				WebUtils.getKey(),
				fn);
	}
	
	/**
	 * アプリケーションスコープでキャッシュする関数を生成します。
	 * @param <C>
	 * @param fn オブジェクト生成の委譲先
	 * @return 関数。
	 */
	public static <C extends ContainerContext<WebContext>>
	ConcurrentMapCache<C, String, Exception> application(
			final Fn<? super C, ?, ? extends Exception> fn) {
		return FactoryUtils.concurrentCache(
				FnUtils.join(
						ContainerUtils.<WebContext, C>getSubContext(),
						WebUtils.applicationFn()),
				WebUtils.getKey(),
				fn);
	}
	
	/**
	 * {@link WebContext#setContainer(Object)}に{@link ContainerContext}を設定する関数を生成します。
	 * @param fn 設定する関数。
	 * @return 関数。
	 */
	public static Fn<WebContext, Void, RuntimeException> setContainerContext(
			final Fn<? super ContainerContext<WebContext>, ?, ? extends Exception> fn) {
		return new ContainerContextSetter<RuntimeException>(fn);
	}
}
